/*
 * The MIT License
 *
 * Copyright 2015 Thibault Debatty.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package info.debatty.java.stringsimilarity;

/**
 * @author Thibault Debatty
 */
public class Cosine extends SetBasedStringSimilarity {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        Cosine cos = new Cosine(3);
        
        // ABC BCE
        // 1  0
        // 1  1
        // angle = 45°
        // => similarity = .71
        
        System.out.println(cos.similarity("ABC", "ABCE"));
        
        cos = new Cosine(2);
        // AB BA
        // 2  1
        // 1  1
        // similarity = .95
        System.out.println(cos.similarity("ABAB", "BAB"));
    }
    
    /**
     * Implements Cosine Similarity.
     * The strings are first transformed in vectors of occurrences of k-shingles 
     * (sequences of k characters). In this n-dimensional space, the similarity
     * between the two strings is the cosine of their respective vectors.
     * 
     * @param k 
     */
    public Cosine(int k) {
        super(k);
    }
    
    public Cosine() {
        super(3);
    }


    public double similarity(int[] profile1, int[] profile2) {
        
        return dotProduct(profile1, profile2) / (norm(profile1) * norm(profile2));
    }


    
    /**
     * Compute the norm L2 : sqrt(Sum_i( v_i^2))
     * @param profile
     * @return L2 norm
     */
    protected static double norm(int[] profile) {
        double agg = 0;
        
        for (int v : profile) {
            agg += v * v;
        }
        
        return Math.sqrt(agg);
    }
    
    protected static double dotProduct(int[] profile1, int[] profile2) {
        int length = Math.max(profile1.length, profile2.length);
        profile1 = java.util.Arrays.copyOf(profile1, length);
        profile2 = java.util.Arrays.copyOf(profile2, length);
        
        double agg = 0;
        for (int i = 0; i < length; i++) {
            agg += profile1[i] * profile2[i];
        }
        return agg;
    }
}
