/*
 * Copyright (c) 2015-2017, Intel Deutschland GmbH
 * Copyright (c) 2011-2015, Intel Mobile Communications GmbH
 *
 * This file is part of the Inheritance plug-in for Jenkins.
 *
 * The Inheritance plug-in is free software: you can redistribute it
 * and/or modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation in version 3
 * of the License
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>.
 */


/**
 * Groovy Jelly view for the InheritanceListAction Jenkins action.
 */
import java.text.MessageFormat;
import java.util.List;

import hudson.model.Project;

import hudson.plugins.project_inheritance.projects.InheritanceProject;
import hudson.plugins.project_inheritance.projects.actions.InheritanceListAction;


f = namespace(lib.FormTagLib);
l = namespace(lib.LayoutTagLib);
ct = namespace(lib.CustomTagLib);

pt = taglib(com.intel.commons.taglibs.PluginTagLib)


/**
 * Entry point.
 */
render(it);


/**
 * Renders the sections with information on the job inheritance trees.
 */
def render(final InheritanceListAction model) {

	def icon = "/plugin/project-inheritance/images/48x48/BinaryTree.png";

	renderChildJobTable(model, icon);
	renderParentJobTable(model, icon);
}


def renderParentJobTable(
		final InheritanceListAction model,
		final String icon) {

	def titleOne = "Inherits from one other job";
	def titleMany = "Inherits from {0} other jobs";
	def isWithLastBuild = false; // Do not show the last build column.

	List<InheritanceProject> parentJobs = model.getParentJobs();

	renderSectionWithJobTable(
			parentJobs, icon, titleOne, titleMany, isWithLastBuild);
}


def renderChildJobTable(
		final InheritanceListAction model,
		final String icon) {

	def titleOne = "Inherited by one other job";
	def titleMany = "Inherited by {0} other jobs";
	def isWithLastBuild = true; // Do show the last build column.

	List<InheritanceProject> childJobs = model.getChildJobs();

	renderSectionWithJobTable(
			childJobs, icon, titleOne, titleMany, isWithLastBuild);
}


def renderSectionWithJobTable(
		final List<InheritanceProject> jobs,
		final String icon,
		final String titleOneJob,
		final String titleManyJobs,
		final boolean isWithLastBuild) {

	int jobCount = jobs.size();

	if ( jobCount > 0 ) {
		def title =
				((jobCount==1)
				 ? titleOneJob
				 : MessageFormat.format(titleManyJobs, jobCount.toString()));

		pt.section(icon: icon, title: title) {
			renderJobTable(jobs, isWithLastBuild);
		}
	} else {
		// When there are no jobs, we render nothing.
	}
}


/**
 * Renders the actual HTML table with a list of jobs.
 */
def renderJobTable(
		final List<InheritanceProject> jobs,
		final boolean isWithLastBuild) {

	def headerLabels = 
		isWithLastBuild ?
		[ _("Job"), _("Class"), _("Last Build") ] :
		[ _("Job"), _("Class") ];
	def extraClass = [ class: "sortable" ];

	pt.bigTable(headerLabels, extraClass) {
		jobs.each { job ->
			pt.bigTableRow {
				pt.bigTableCell {
					a(href: job.getAbsoluteUrl(), job.getFullName());
				}
				pt.bigTableCell {
					text(job.getPronoun());
				}
				if ( isWithLastBuild ) {
					renderLastBuildCell(job);
				}
			}
		}
	}
}


/**
 * Renders the table cell with information on the latest build of the
 * given job.
 */
def renderLastBuildCell(final InheritanceProject job) {

	def lastBuild = job.getLastBuild();
	def contentsRenderer = null;
	def dataForSorting = "";

	if ( lastBuild != null ) {
		contentsRenderer = {
			ct.buildtime(
				link: job.getAbsoluteUrl() + lastBuild.getNumber(),
				buildtime: lastBuild.getTime(),
				buildStatusUrl: job.getBuildStatusUrl(),
				buildDisplayId: lastBuild.getNumber() );
		};
		dataForSorting = formatDateAsString(lastBuild.getTime());
	} else {
		contentsRenderer = {
			// No last build, so we render nothing.
		};
	}

	// The Jenkins GUI tries to use an attribute named "data" in the
	// column cells for ordering the table lines by that column. If no
	// "data" attribute is presents, Jenkins will use the cell string
	// contents. As the string ordering for this cell is not the same
	// as the build date ordering, we explicitly set the "data"
	// attribute to a string whose ordering is the same as the build
	// date.
	pt.bigTableCell([data: dataForSorting], contentsRenderer);
}


/**
 * Converts the given date into a string. The ordering of strings
 * generated by this method is guaranteed to be the same ordering as
 * the corresponding dates.
 */
def formatDateAsString(final Date date) {

	String result = String.format("%019d", date.getTime());

	return result;
}
