/*
 * Copyright (C) 2017 Olmo Gallegos Hernández.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package es.voghdev.redsysonandroid.library;

import android.support.annotation.NonNull;

import java.io.UnsupportedEncodingException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;

import sis.redsys.api.ApiMacSha256;

public class RedsysPayment {
    float amount;
    int currency;
    int transactionType;
    int terminal;
    long orderNumber;
    String successUrl;
    String failureUrl;
    String merchantUrl;
    String secretKey;
    String merchantId;
    String signatureType;

    public int getCurrency() {
        return currency;
    }

    public void setCurrency(int currency) {
        this.currency = currency;
    }

    public String getSuccessUrl() {
        return successUrl;
    }

    public void setSuccessUrl(String successUrl) {
        this.successUrl = successUrl;
    }

    public String getFailureUrl() {
        return failureUrl;
    }

    public void setFailureUrl(String failureUrl) {
        this.failureUrl = failureUrl;
    }

    public String getMerchantUrl() {
        return merchantUrl;
    }

    public void setMerchantUrl(String merchantUrl) {
        this.merchantUrl = merchantUrl;
    }

    public float getAmount() {
        return amount;
    }

    public void setAmount(float amount) {
        this.amount = amount;
    }

    public long getOrderNumber() {
        return orderNumber;
    }

    public void setOrderNumber(long orderNumber) {
        this.orderNumber = orderNumber;
    }

    public String getSecretKey() {
        return secretKey;
    }

    public void setSecretKey(String secretKey) {
        this.secretKey = secretKey;
    }

    public String getMerchantId() {
        return merchantId;
    }

    public void setMerchantId(String merchantId) {
        this.merchantId = merchantId;
    }

    public int getTransactionType() {
        return transactionType;
    }

    public void setTransactionType(int transactionType) {
        this.transactionType = transactionType;
    }

    public int getTerminal() {
        return terminal;
    }

    public void setTerminal(int terminal) {
        this.terminal = terminal;
    }

    public void setSignatureType(@NonNull String signatureType) {
        this.signatureType = signatureType;
    }

    private ApiMacSha256 getEncryptionObject() {
        ApiMacSha256 apiMacSha256 = new ApiMacSha256();
        apiMacSha256.setParameter(Redsys.MERCHANT_AMOUNT, Float.toString(getAmount()));
        apiMacSha256.setParameter(Redsys.MERCHANT_ORDER, Long.toString(orderNumber));
        apiMacSha256.setParameter(Redsys.MERCHANT_MERCHANTCODE, getMerchantId());
        apiMacSha256.setParameter(Redsys.MERCHANT_CURRENCY, Integer.toString(getCurrency()));
        apiMacSha256.setParameter(Redsys.MERCHANT_TRANSACTIONTYPE, Integer.toString(transactionType));
        apiMacSha256.setParameter(Redsys.MERCHANT_TERMINAL, Integer.toString(terminal));
        apiMacSha256.setParameter(Redsys.MERCHANT_MERCHANTURL, getMerchantUrl());
        apiMacSha256.setParameter(Redsys.MERCHANT_URLOK, getSuccessUrl());
        apiMacSha256.setParameter(Redsys.MERCHANT_URLKO, getFailureUrl());
        return apiMacSha256;
    }

    public String createMerchantParameters() throws UnsupportedEncodingException {
        return getEncryptionObject().createMerchantParameters();
    }

    public String createMerchantSignature() throws UnsupportedEncodingException, InvalidKeyException,
            NoSuchAlgorithmException, IllegalStateException, NoSuchPaddingException,
            InvalidAlgorithmParameterException, IllegalBlockSizeException, BadPaddingException {
        return getEncryptionObject().createMerchantSignature(secretKey);
    }

    public String getSignatureType() {
        return signatureType;
    }

    //region Builder
    public static class Builder {
        private int currency = Redsys.CURRENCY_EURO;
        private int transactionType = Redsys.DEFAULT_TRANSACTION_TYPE;
        private int terminal = Redsys.DEFAULT_TERMINAL;
        private float amount = 0f;
        private long orderNumber = 0L;
        private String signatureType = Redsys.DEFAULT_SIGNATURE_TYPE;
        private String successUrl = "";
        private String failureUrl = "";
        private String merchantUrl = "";
        private String secretKey = "";
        private String merchantId = "";

        public Builder setAmount(float amount) {
            this.amount = amount;
            return this;
        }

        public Builder setCurrency(int currency) {
            this.currency = currency;
            return this;
        }

        public Builder setOrderNumber(long orderNumber) {
            this.orderNumber = orderNumber;
            return this;
        }

        public Builder setSuccessUrl(@NonNull String successUrl) {
            this.successUrl = successUrl;
            return this;
        }

        public Builder setFailureUrl(@NonNull String failureUrl) {
            this.failureUrl = failureUrl;
            return this;
        }

        public Builder setMerchantUrl(@NonNull String merchantUrl) {
            this.merchantUrl = merchantUrl;
            return this;
        }

        public Builder setSecretKey(@NonNull String secretKey) {
            this.secretKey = secretKey;
            return this;
        }

        public Builder setMerchantId(@NonNull String merchantId) {
            this.merchantId = merchantId;
            return this;
        }

        public Builder setTransactionType(int transactionType) {
            this.transactionType = transactionType;
            return this;
        }

        public Builder setTerminal(int terminal) {
            this.terminal = terminal;
            return this;
        }

        public Builder setSignatureType(@NonNull String signatureType) {
            this.signatureType = signatureType;
            return this;
        }

        public RedsysPayment create() {
            RedsysPayment payment = new RedsysPayment();
            payment.setAmount(amount);
            payment.setCurrency(currency);
            payment.setOrderNumber(orderNumber);
            payment.setSuccessUrl(successUrl);
            payment.setFailureUrl(failureUrl);
            payment.setMerchantUrl(merchantUrl);
            payment.setSecretKey(secretKey);
            payment.setMerchantId(merchantId);
            payment.setTransactionType(transactionType);
            payment.setTerminal(terminal);
            payment.setSignatureType(signatureType);
            return payment;
        }
    }
    //endregion
}
