package dev.micheleferretti.mapboxpluginoffline.model

import android.os.Parcelable
import com.mapbox.mapboxsdk.offline.OfflineRegion
import com.mapbox.mapboxsdk.offline.OfflineRegionStatus
import dev.micheleferretti.mapboxpluginoffline.utils.OfflineUtils
import kotlinx.android.parcel.Parcelize

/**
 * This model represents the download managed by
 * the [OfflineService][dev.micheleferretti.mapboxpluginoffline.OfflineService].
 *
 * It holds the `options` with which the download has been started, the id (`regionId`) of the region generated by
 * Mapbox with `OfflineManager.createOfflineRegion()` and some of the `OfflineRegionStatus` data returned by
 * the `OfflineRegionObserver` attached to the region in order to monitor its download state.
 *
 * @property regionId `OfflineRegion` id.
 * @property options Options used to start the download.
 * @constructor Creates a `OfflineDownload` with the given values.
 */
@Parcelize
class OfflineDownload(
    val regionId: Long,
    val options: OfflineDownloadOptions,
    private var completedResourceCount: Long,
    private var requiredResourceCount: Long,
    private var completedResourceSize: Long,
    private var isActive: Boolean
): Parcelable {

    /**
     * Creates a `OfflineDownload` with the given values and no status data.
     */
    constructor(regionId: Long, options: OfflineDownloadOptions): this(regionId, options, 0L, 0L, 0L, false)

    /**
     * As in `OfflineRegionStatus`, returns the number of resources that have been fully download.
     * @return the number of resources that have been fully download.
     */
    fun getCompletedResourceCount() = completedResourceCount

    /**
     * As in `OfflineRegionStatus`, returns the number of resources that are known to be required.
     * @return the number of resources that are known to be required.
     */
    fun getRequiredResourceCount() = requiredResourceCount

    /**
     * As in `OfflineRegionStatus`, returns the cumulative size, in bytes, of all resources that have been fully downloaded.
     * @return the cumulative size, in bytes, of all resources that have been fully downloaded.
     */
    fun getCompletedResourceSize() = completedResourceSize

    /**
     * Returns the download's resources completion percentage.
     * @return the download's resources completion percentage.
     * @see OfflineUtils.getCompletionPercentage
     */
    fun getPercentage() = OfflineUtils.getCompletionPercentage(completedResourceCount, requiredResourceCount)

    /**
     * Validates if `OfflineRegionStatus.getDownloadState()` equals `OfflineRegion.STATE_ACTIVE`.
     * @return `true` if download is active, `false` if not.
     */
    fun isActive() = isActive

    /**
     * As in `OfflineRegionStatus`, validates if the region download has completed.
     * @return `true` if download is complete, `false` if not.
     */
    fun isComplete() = completedResourceCount >= requiredResourceCount

    /**
     * Updates the download status properties
     */
    internal fun setStatus(status: OfflineRegionStatus) {
        completedResourceCount = status.completedResourceCount
        requiredResourceCount = status.requiredResourceCount
        completedResourceSize = status.completedResourceSize
        isActive = status.downloadState == OfflineRegion.STATE_ACTIVE
    }

    /**
     * Returns a `String` representation of this object.
     * @return a `String` representation of this object.
     */
    override fun toString() = "OfflineDownload(" +
            "regionId=$regionId, " +
            "options=$options, " +
            "completedResourceCount=$completedResourceCount, " +
            "requiredResourceCount=$requiredResourceCount, " +
            "completedResourceSize=$completedResourceSize, " +
            "isActive=$isActive" +
            ")"
}