package de.flapdoodle.embed.mongo.packageresolver;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * Immutable implementation of {@link FeatureSet}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableFeatureSet.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableFeatureSet.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableFeatureSet extends FeatureSet {
  private final Set<Feature> features;

  private ImmutableFeatureSet(Iterable<Feature> features) {
    this.features = createUnmodifiableEnumSet(features);
  }

  private ImmutableFeatureSet(
      ImmutableFeatureSet original,
      Set<Feature> features) {
    this.features = features;
  }

  /**
   * @return The value of the {@code features} attribute
   */
  @Override
  protected Set<Feature> features() {
    return features;
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link FeatureSet#features() features}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableFeatureSet withFeatures(Feature... elements) {
    Set<Feature> newValue = createUnmodifiableEnumSet(Arrays.asList(elements));
    return new ImmutableFeatureSet(this, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link FeatureSet#features() features}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of features elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableFeatureSet withFeatures(Iterable<Feature> elements) {
    if (this.features == elements) return this;
    Set<Feature> newValue = createUnmodifiableEnumSet(elements);
    return new ImmutableFeatureSet(this, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableFeatureSet} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableFeatureSet
        && equalTo(0, (ImmutableFeatureSet) another);
  }

  private boolean equalTo(int synthetic, ImmutableFeatureSet another) {
    return features.equals(another.features);
  }

  /**
   * Computes a hash code from attributes: {@code features}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + features.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code FeatureSet} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "FeatureSet{"
        + "features=" + features
        + "}";
  }

  /**
   * Construct a new immutable {@code FeatureSet} instance.
   * @param features The value for the {@code features} attribute
   * @return An immutable FeatureSet instance
   */
  public static ImmutableFeatureSet of(Set<Feature> features) {
    return of((Iterable<Feature>) features);
  }

  /**
   * Construct a new immutable {@code FeatureSet} instance.
   * @param features The value for the {@code features} attribute
   * @return An immutable FeatureSet instance
   */
  public static ImmutableFeatureSet of(Iterable<Feature> features) {
    return new ImmutableFeatureSet(features);
  }

  /**
   * Creates an immutable copy of a {@link FeatureSet} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable FeatureSet instance
   */
  public static ImmutableFeatureSet copyOf(FeatureSet instance) {
    if (instance instanceof ImmutableFeatureSet) {
      return (ImmutableFeatureSet) instance;
    }
    return ImmutableFeatureSet.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableFeatureSet ImmutableFeatureSet}.
   * <pre>
   * ImmutableFeatureSet.builder()
   *    .addFeatures|addAllFeatures(de.flapdoodle.embed.mongo.packageresolver.Feature) // {@link FeatureSet#features() features} elements
   *    .build();
   * </pre>
   * @return A new ImmutableFeatureSet builder
   */
  public static ImmutableFeatureSet.Builder builder() {
    return new ImmutableFeatureSet.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableFeatureSet ImmutableFeatureSet}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private EnumSet<Feature> features = EnumSet.noneOf(Feature.class);

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code FeatureSet} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(FeatureSet instance) {
      Objects.requireNonNull(instance, "instance");
      addAllFeatures(instance.features());
      return this;
    }

    /**
     * Adds one element to {@link FeatureSet#features() features} set.
     * @param element A features element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addFeatures(Feature element) {
      this.features.add(Objects.requireNonNull(element, "features element"));
      return this;
    }

    /**
     * Adds elements to {@link FeatureSet#features() features} set.
     * @param elements An array of features elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addFeatures(Feature... elements) {
      for (Feature element : elements) {
        this.features.add(Objects.requireNonNull(element, "features element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link FeatureSet#features() features} set.
     * @param elements An iterable of features elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder features(Iterable<Feature> elements) {
      this.features.clear();
      return addAllFeatures(elements);
    }

    /**
     * Adds elements to {@link FeatureSet#features() features} set.
     * @param elements An iterable of features elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllFeatures(Iterable<Feature> elements) {
      for (Feature element : elements) {
        this.features.add(Objects.requireNonNull(element, "features element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableFeatureSet ImmutableFeatureSet}.
     * @return An immutable instance of FeatureSet
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableFeatureSet build() {
      return new ImmutableFeatureSet(null, createUnmodifiableEnumSet(features));
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  @SuppressWarnings("unchecked")
  private static <T extends Enum<T>> Set<T> createUnmodifiableEnumSet(Iterable<T> iterable) {
    if (iterable instanceof EnumSet<?>) {
      return Collections.unmodifiableSet(EnumSet.copyOf((EnumSet<T>) iterable));
    }
    List<T> list = createSafeList(iterable, true, false);
    switch(list.size()) {
    case 0: return Collections.emptySet();
    case 1: return Collections.singleton(list.get(0));
    default: return Collections.unmodifiableSet(EnumSet.copyOf(list));
    }
  }
}
