package de.cidaas.oauth.interceptor;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import javax.annotation.security.DenyAll;
import javax.annotation.security.PermitAll;
import javax.annotation.security.RolesAllowed;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.util.StringUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerExecutionChain;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import com.fasterxml.jackson.databind.ObjectMapper;

import de.cidaas.oauth.model.OAuthAccessResourceRequest;
import de.cidaas.oauth.model.OAuthUser;
import de.cidaas.oauth.model.ServerResponse;
import de.cidaas.oauth.model.TokenIntrospectionRequest;
import de.cidaas.oauth.model.TokenIntrospectionResponse;

public class OAuthInterceptor implements Filter {
	private static final Logger LOGGER = LoggerFactory.getLogger(OAuthInterceptor.class);

	/** The Constant typeJson. */
	private static final List<MediaType> typeJson = new ArrayList<MediaType>();

	/** The Constant typeXML. */
	private static final List<MediaType> typeXML = new ArrayList<MediaType>();

//	@Autowired
	private org.springframework.context.ApplicationContext appContext;

	static {
		typeJson.add(MediaType.APPLICATION_JSON);
		typeXML.add(MediaType.APPLICATION_XML);
		typeXML.add(MediaType.APPLICATION_ATOM_XML);
		typeXML.add(MediaType.TEXT_XML);
	}

	/** The Constant XML_RESPONSE_ERROR. */
//	private static final String XML_RESPONSE_ERROR = "<error>%s</error>";

	/** The Constant JSON_RESPONSE_ERROR. */
//	private static final String JSON_RESPONSE_ERROR = "{\"error\":\"%s\"}";

	/** The Constant ACCESS_DENIED. */
	private static final String ACCESS_DENIED = "Access denied for this resource";

	public OAuthInterceptor() {

	}

	public OAuthInterceptor(org.springframework.context.ApplicationContext applicationContext) {
		if (applicationContext != null) {
			appContext = applicationContext;
		}
	}

	@Override
	public void init(FilterConfig filterConfig) throws ServletException {

	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response, FilterChain filterChain)
			throws IOException, ServletException {

		HttpServletRequest httpRequest = (HttpServletRequest) request;
		HttpServletResponse httpResponse = (HttpServletResponse) response;
		LOGGER.info("Logging Request  {} : {}", httpRequest.getMethod(), httpRequest.getRequestURI());


		HandlerMethod handlerMethod = null;
		try {
			RequestMappingHandlerMapping req2HandlerMapping = (RequestMappingHandlerMapping) appContext
					.getBean("requestMappingHandlerMapping");

			HandlerExecutionChain handlerExeChain = req2HandlerMapping.getHandler(httpRequest);
			if (Objects.nonNull(handlerExeChain)) {
				handlerMethod = (HandlerMethod) handlerExeChain.getHandler();
				Method method = handlerMethod.getMethod();
				if (method.isAnnotationPresent(PermitAll.class)) {
					LOGGER.info("Method has PermitAll:" + method.getName());
					filterChain.doFilter(request, response);
					return;
				}

				if (method.isAnnotationPresent(DenyAll.class)) {
					LOGGER.info("Method has DenyAll:" + method.getName());			
					getStatusMessage(httpResponse,HttpStatus.SC_UNAUTHORIZED,ACCESS_DENIED);
					return;
				}

			}
		} catch (Exception e) {
			LOGGER.warn("Lookup the handler method", e);
		}

		// prepare request info for introspect service

		TokenIntrospectionRequest requestInfo;
		try {
			requestInfo = prepareTokenRequest(httpRequest, handlerMethod);
		} catch (Exception ex) {
			LOGGER.error("Exception {}", ex);
			getStatusMessage(httpResponse,HttpStatus.SC_UNAUTHORIZED,ACCESS_DENIED);
			return;
		}

		String accessToken = null;
		try {
			OAuthAccessResourceRequest oauthRequest = new OAuthAccessResourceRequest(httpRequest);
			accessToken = oauthRequest.getAccessToken();
			LOGGER.info("Access Token : {}", accessToken);
			requestInfo.setToken(accessToken);
		} catch (Exception ex) {
			LOGGER.error("Exception {}", ex);
			getStatusMessage(httpResponse,HttpStatus.SC_UNAUTHORIZED,ACCESS_DENIED);
			return;
		}

		// call the introspect method

		TokenIntrospectionResponse tokenValidationResp = ValidateToken.getInstance().validateAccessToken(requestInfo);

		if (tokenValidationResp != null && tokenValidationResp.isActive()) {
			OAuthUser u = new OAuthUser(accessToken, tokenValidationResp.getSub());
			LOGGER.info("Interceptor got user {} from token {}.", u.getUserId(), accessToken);
			request.setAttribute("User-details", u);
			filterChain.doFilter(request, response);
		} else {
			LOGGER.error("Access denied for URL {}", requestInfo.getRequest_url());
			getStatusMessage(httpResponse,HttpStatus.SC_UNAUTHORIZED,ACCESS_DENIED);
			return;
		}
		// call next filter in the filter chain

		LOGGER.info("Logging Response :{}", response.getContentType());
	}

	@Override
	public void destroy() {

	}

	private TokenIntrospectionRequest prepareTokenRequest(HttpServletRequest request, HandlerMethod handlerMethod) {
		TokenIntrospectionRequest tokenIntrospectionRequest = new TokenIntrospectionRequest();
		if (handlerMethod != null && handlerMethod.getMethod() != null) {
			tokenIntrospectionRequest.setScopes(getAnotationRequestedScope(handlerMethod.getMethod()));
			tokenIntrospectionRequest.setRoles(getAnotationRequestedRoles(handlerMethod.getMethod()));
		}
		tokenIntrospectionRequest.setRequest_time(new Date().getTime());
		tokenIntrospectionRequest.setRequest_url(request.getRequestURL().toString());

		List<String> allHeaders = Collections.list(request.getHeaderNames());
		for (String headerKey : allHeaders) {
			String _headerKey = headerKey.toLowerCase();
			if (!_headerKey.contains("cookie") && !_headerKey.equalsIgnoreCase("authorization")
					&& !_headerKey.equals("__access_token")) {
				tokenIntrospectionRequest.getRequest_headers().put(headerKey, request.getHeader(headerKey));
			}
		}

		String ipAddress = request.getHeader("X-FORWARDED-FOR");
		if (ipAddress == null) {
			ipAddress = request.getRemoteAddr();
		}
		if (ipAddress != null && ipAddress.contains(",")) {
			ipAddress = ipAddress.split(",")[0];
		}
		if (!StringUtils.isEmpty(ipAddress)) {
			tokenIntrospectionRequest.getRequest_headers().put("x-forwarded-for", ipAddress);
		}

		return tokenIntrospectionRequest;
	}

	protected String[] getAnotationRequestedScope(Method method) {
		if (method.isAnnotationPresent(OAuthScopes.class)) {
			OAuthScopes oAuthScopes = method.getAnnotation(OAuthScopes.class);
			if (oAuthScopes != null && oAuthScopes.scopes().length > 0) {
				return oAuthScopes.scopes();
			}
		}
		return null;
	}

	protected String[] getAnotationRequestedRoles(Method method) {
		if (method.isAnnotationPresent(RolesAllowed.class)) {
			RolesAllowed rolesAnnotation = method.getAnnotation(RolesAllowed.class);
			if (rolesAnnotation != null && rolesAnnotation.value().length > 0) {
				return rolesAnnotation.value();
			}
		}
		return null;
	}
	
	private HttpServletResponse getStatusMessage(HttpServletResponse httpResponse,int status,String message) throws IOException {
		httpResponse.setStatus(status);
		String json = new ObjectMapper().writeValueAsString(new ServerResponse(message));
		httpResponse.getWriter().write(json);
		httpResponse.flushBuffer();
		return httpResponse;
	}
}
