package de.cidaas.oauth.util;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Configuration loader to load java properties either as single property or as whole {@link Properties} object
 * 
 * Note: warc.homepath must be set as System.property
 * 
 * @author jan-henrik.preuss
 *
 */
public class ConfigurationLoader {

  private static final Logger logger = LoggerFactory.getLogger(ConfigurationLoader.class);
  private static final String path = System.getProperty("warc.homepath");

  /**
   * Loads the given properties file
   * 
   * @param pFilename
   *          Property File
   * @return null if file dose not exist
   */
  public static Properties getPropertiesOfFile(String pFilename) {
	if( StringUtils.isEmpty( path ) ){
		throw new RuntimeException( "warc.homepath not set while reading Propertyfile:" + pFilename );
	}
	if( StringUtils.isEmpty( pFilename ) ){
		throw new RuntimeException( "No properties filename to read from specified, warc.homepath=" + path );
	}
	
    try {
      Properties props = new Properties();
      InputStream in;
      in = getFullqualifiedFilename(pFilename);
      props.load(in);
      
      for (Object key: props.keySet()) {
    	  String propValue = System.getenv(key.toString());
    	  if (propValue != null){
    		  props.setProperty(key.toString(), propValue);
    	  }
      }
      
      in.close();
      return props;
    } catch (FileNotFoundException e) {
      logger.error(pFilename + " not found.", e);
      return null;
    } catch (IOException e) {
      logger.error("IOException while loading " + pFilename, e);
      return null;
    }
  }

  /**
   * Loads a XML-Configuration from the XML-File with JAXB.
   * 
   * @param configClass
   *          class representing the root element of the XML-Structure
   * @param fileName - the XML-File config path
   * @return the configuration object
   * @throws JAXBException - if the XML is invalid
   * @throws FileNotFoundException - if the XML file is not found or available
   */
  public static Object loadJAXB(Class<?> configClass, String fileName) throws JAXBException, FileNotFoundException {
    JAXBContext lCtx = JAXBContext.newInstance(configClass);
    Unmarshaller lUnMarshaller = lCtx.createUnmarshaller();
    InputStream in = getFullqualifiedFilename(fileName);
    Object lConfiguration = lUnMarshaller.unmarshal(in);
    return lConfiguration;
  }

  private static InputStream getFullqualifiedFilename(String pFilename) throws FileNotFoundException {
    StringBuilder lSB = new StringBuilder(path);
    if (path != null && !path.endsWith("/"))
      lSB.append("/");
    lSB.append("conf/");
    lSB.append(pFilename);
    final String lFilePath = lSB.toString();
    logger.debug("load properties from file:" + lFilePath);
    InputStream inputStream = ConfigurationLoader.class.getClassLoader().getResourceAsStream(lFilePath);
    if (inputStream == null) {
      inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(lFilePath);
    }
    if (inputStream == null) {
      inputStream = new FileInputStream(lFilePath);
    }
    return inputStream;
  }

  /**
   * Get a single property from baufi.properties
   * 
   * @param key
   *          the property key
   * @param defaultValue
   *          the default value
   * @param pFilename
   *          Property File in your warc.homepath/conf folder
   * @return the defaultValue if property dose not exist
   */
  public static String getPropertyByKey(String key, String defaultValue, String pFilename) {
    Properties properties = getPropertiesOfFile(pFilename);
    if (properties != null) {
      return properties.getProperty(key);
    }
    return defaultValue;
  }

  /**
   * Get a single property from baufi.properties
   * 
   * @param key
   *          the property key
   * @param pFilename
   *          Property File in your warc.homepath/conf folder
   * @return null if property dose not exist
   */
  public static String getPropertyByKey(String key, String pFilename) {
    return getPropertyByKey(key, null, pFilename);
  }

}
