package com.zoyi.channel.plugin.android.view.dock;

import android.content.Context;
import android.content.res.TypedArray;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.util.Initializer;

public class DockLayout extends ViewGroup {

  public DockLayout(Context context) {
    super(context);

    init(context, null);
  }

  public DockLayout(Context context, AttributeSet attrs) {
    super(context, attrs);

    init(context, attrs);
  }

  public DockLayout(Context context, AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);

    init(context, attrs);
  }

  private int spaceWidth = 0;
  private int baseItemCount = 1;

  @Initializer
  private void init(Context context, @Nullable AttributeSet attrs) {
    if (attrs != null) {
      TypedArray typedArray = context.getTheme().obtainStyledAttributes(attrs, R.styleable.DockLayout, 0, 0);

      try {
        spaceWidth = (int) typedArray.getDimension(R.styleable.DockLayout_ch_dl_spaceWidth, (float) spaceWidth);
        baseItemCount = Math.max(1, typedArray.getInt(R.styleable.DockLayout_ch_dl_baseItemCount, baseItemCount));
      } finally {
        typedArray.recycle();
      }
    }
  }

  @Override
  protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
    int visibleChildCount = 0;

    for (int i = 0; i < getChildCount(); i++) {
      View view = getChildAt(i);

      if (view != null && view.getVisibility() != View.GONE) {
        visibleChildCount++;
      }
    }

    int containerWidth = MeasureSpec.getSize(widthMeasureSpec);

    if (containerWidth == 0) {
      containerWidth = Math.max(0, ((ViewGroup) getParent()).getMeasuredWidth());
    }

    int itemSize;

    if (visibleChildCount <= baseItemCount) {
      itemSize = (containerWidth - spaceWidth * (baseItemCount - 1)) / baseItemCount;
    } else {
      itemSize = (int) ((containerWidth - spaceWidth * baseItemCount) / (baseItemCount + 0.5f));
    }

    for (int i = 0; i < getChildCount(); i++) {
      View view = getChildAt(i);

      if (view != null && view.getVisibility() != View.GONE) {
        view.measure(
            MeasureSpec.makeMeasureSpec(itemSize, MeasureSpec.EXACTLY),
            MeasureSpec.makeMeasureSpec(itemSize, MeasureSpec.EXACTLY)
        );
      }
    }

    setMeasuredDimension(Math.max(containerWidth, (itemSize + spaceWidth) * visibleChildCount - spaceWidth), itemSize);
  }

  @Override
  protected void onLayout(boolean changed, int l, int t, int r, int b) {
    int itemSize = b - t;

    int visibleChildCount = 0;

    for (int i = 0; i < getChildCount(); i++) {
      View view = getChildAt(i);

      if (view != null && view.getVisibility() != View.GONE) {
        visibleChildCount++;
      }
    }

    int horizontalStartPosition = Math.max(0, ((r - l) - visibleChildCount * itemSize - spaceWidth * (visibleChildCount - 1)) / 2);

    for (int i = 0; i < getChildCount(); i++) {
      View view = getChildAt(i);

      if (view != null && view.getVisibility() != View.GONE) {
        view.layout(
            horizontalStartPosition + (itemSize + spaceWidth) * i,
            t,
            horizontalStartPosition + (itemSize + spaceWidth) * i + itemSize,
            b
        );
      }
    }
  }
}
