package com.zoyi.channel.plugin.android.network;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.bind.Binder;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.concurrent.TimeUnit;

public class ApiCaller<E> extends Binder {

  private Observable<E> observable;

  @Nullable
  private Subscription subscription;

  @Nullable
  private ErrorFunction onErrorFunc;

  @Nullable
  private CompleteFunction onBeforeCompleteFunc;

  @Nullable
  private CompleteFunction onCompleteFunc;

  @Nullable
  private SuccessFunction<E> callFunc;

  public ApiCaller(Observable<E> observable) {
    this.observable = observable;
  }

  public ApiCaller<E> delay(long milliseconds) {
    this.observable = this.observable.delay(milliseconds, TimeUnit.MILLISECONDS);

    return this;
  }

  public ApiCaller<E> onError(ErrorFunction onErrorFunc) {
    this.onErrorFunc = onErrorFunc;

    return this;
  }

  public ApiCaller<E> onBeforeComplete(CompleteFunction onBeforeCompleteFunc) {
    this.onBeforeCompleteFunc = onBeforeCompleteFunc;

    return this;
  }

  public ApiCaller<E> onComplete(CompleteFunction onCompleteFunc) {
    this.onCompleteFunc = onCompleteFunc;

    return this;
  }

  public ApiCaller<E> call(SuccessFunction<E> callFunc) {
    this.callFunc = callFunc;

    return call();
  }

  public ApiCaller<E> call() {
    this.subscription = observable.onBackpressureBuffer()
        .subscribeOn(Schedulers.io())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<E>() {
          @Override
          public void onError(RetrofitException error) {
            if (onBeforeCompleteFunc != null) {
              onBeforeCompleteFunc.call();
            }
            if (onErrorFunc != null) {
              onErrorFunc.call(error);
            }
            if (onCompleteFunc != null) {
              onCompleteFunc.call();
            }
          }

          @Override
          public void onSuccess(@NonNull E e) {
            if (onBeforeCompleteFunc != null) {
              onBeforeCompleteFunc.call();
            }
            if (callFunc != null) {
              callFunc.call(e);
            }
            if (onCompleteFunc != null) {
              onCompleteFunc.call();
            }
          }

          @Override
          public void onSuccessWithNull() {
            if (onBeforeCompleteFunc != null) {
              onBeforeCompleteFunc.call();
            }
            if (onCompleteFunc != null) {
              onCompleteFunc.call();
            }
          }
        });

    return this;
  }

  @Override
  protected boolean isRunning() {
    return this.subscription != null && !this.subscription.isUnsubscribed();
  }

  @Override
  public void unbind() {
    if (this.subscription != null && !this.subscription.isUnsubscribed()) {
      this.subscription.unsubscribe();
    }
  }

  // interface

  public interface CompleteFunction {
    void call();
  }

  public interface ErrorFunction {
    void call(RetrofitException ex);
  }

  public interface SuccessFunction<T> {
    void call(T repo);
  }
}
