package com.zoyi.channel.plugin.android.activity.chat2;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.action.ChatAction;
import com.zoyi.channel.plugin.android.activity.chat.binder.ChatHeaderBinder;
import com.zoyi.channel.plugin.android.activity.chat.binder.ChatInteractionBinder;
import com.zoyi.channel.plugin.android.activity.chat.contract.ChatAdapterContract;
import com.zoyi.channel.plugin.android.activity.chat.enumerate.ChatState;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManager;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManagerInterface;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.base.AbstractAdapterPresenter;
import com.zoyi.channel.plugin.android.bind.BindAction;
import com.zoyi.channel.plugin.android.enumerate.*;
import com.zoyi.channel.plugin.android.global.*;
import com.zoyi.channel.plugin.android.model.repo.MessagesRepo;
import com.zoyi.channel.plugin.android.model.repo.UserChatRepo;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.source.photopicker.PhotoItem;
import com.zoyi.channel.plugin.android.selector.ChatSelector;
import com.zoyi.channel.plugin.android.selector.SocketSelector;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.com.annimon.stream.Stream;
import com.zoyi.com.annimon.stream.function.IndexedFunction;

import java.util.*;

import static com.zoyi.channel.plugin.android.global.Const.ACTION_TYPE_CLOSE;
import static com.zoyi.channel.plugin.android.global.Const.ACTION_TYPE_SOLVE;

public class ChatPresenter2
    extends AbstractAdapterPresenter<ChatContract2.View, ChatAdapterContract.View, ChatAdapterContract.Model>
    implements ChatContract2.Presenter {

  @Nullable
  private String chatId;

  @Nullable
  private String presetMessage;

  @Nullable
  private Session lastReadSession;

  @Nullable
  private String prev;

  private ChatState chatState = ChatState.DISCONNECTED;

  // queue

  @Nullable
  private ArrayList<Message> tempQueue;

  // binders

  @Nullable
  private ChatHeaderBinder chatHeaderBinder;

  @Nullable
  private ChatInteractionBinder chatInteractionBinder;

  // last message for set input state
  @Nullable
  private Message lastMessage;

  public ChatPresenter2(
      ChatContract2.View view,
      ChatAdapterContract.View adapterView,
      ChatAdapterContract.Model adapterModel,
      @Nullable String chatId,
      @Nullable String presetMessage
  ) {
    super(view, adapterView, adapterModel);

    this.chatId = chatId;
    this.presetMessage = presetMessage;
  }

  private ChatManagerInterface getChatManager() {
    return ChatManager.get(this.chatId);
  }

  @Override
  public void init() {
    view.onFetchStateChange(FetchState.LOADING);

    if (this.chatId != null) {
      initUserChat();
    } else {
      // TODO : Set local message logic
    }
  }

  @Override
  public void createMarketingSupportBotUserChat() {
    if (this.chatInteractionBinder != null) {
      Message marketingMessage = chatInteractionBinder.getMarketingMessage();

      if (this.chatId != null && marketingMessage != null && marketingMessage.getMarketing() != null && !isRunning(BindAction.CREATE_CHAT)) {
        view.showProgress();

        Api.createMarketingSupportBotUserChat(this.chatId, marketingMessage.getMarketing().getType(), marketingMessage.getMarketing().getId())
            .onComplete(() -> view.hideProgress())
            .call(repo -> {
              adapterModel.addMessage(repo.getMessage());

              if (this.chatInteractionBinder != null) {
                chatInteractionBinder.setMarketingMessage(repo.getMessage());
              }
            })
            .bind(this, BindAction.CREATE_CHAT);
      }
    }
  }

  // user chat logic

  private void initUserChat() {
    initUserChat(null);
  }

  private void initUserChat(@Nullable Message initMessage) {
    // must call only once

    SocketSelector.bindSocket(socketStatus -> {
      if (socketStatus == SocketStatus.READY) {
        fetchMessages();
      }
    }).bind(this, BindAction.BIND_SOCKET);

    if (this.chatId != null) {
      ChatSelector.bindHostTyping(this.chatId, typings -> adapterModel.setTypings(typings)).bind(this);
    }

    this.chatHeaderBinder = new ChatHeaderBinder(showAssignee -> view.switchHeader(showAssignee)).bind(this);

    this.chatInteractionBinder = new ChatInteractionBinder(initMessage, chatInteractionState ->
        view.onChatInteractionStateChange(chatInteractionState)
    ).bind(this);

    RxBus.bind(this::onSocketReceived, this);
  }

  // data fetch

  private void fetchMessages() {
    if (this.chatId != null) {
      this.chatState = ChatState.FETCHING;

      initQueue();

      Api.getMessages(this.chatId)
          .onError(ex -> {
            this.chatState = ChatState.DISCONNECTED;

            if (ex.is4xxClientError()) {
              view.finish();
            } else {
              view.onFetchStateChange(FetchState.FAILED);
            }

            clearQueue();
          })
          .call(repo -> {
            this.chatState = ChatState.NEWEST;

            UserChatRepo userChatRepo = repo.getUserChatRepo();
            MessagesRepo messagesRepo = repo.getMessagesRepo();

            onUserChatChanged(userChatRepo.getUserChat());

            this.lastReadSession = userChatRepo.getSession();
            this.prev = messagesRepo.getNext();

            // resolve messages
            ArrayList<Message> appendMessages = new ArrayList<>(messagesRepo.getMessages());

            if (this.tempQueue != null) {
              appendMessages.addAll(this.tempQueue);

              clearQueue();
            }

            // sort append messages
            Collections.sort(appendMessages, (o1, o2) -> CompareUtils.compare(o1.getCreatedAt(), o2.getCreatedAt()));

            adapterModel.setMessages(appendMessages, getChatManager().getUnsentItems(), lastReadSession, messagesRepo.getNext());

            // set last message when repo.prev (next) is null and not empty for show support bot button
            Message lastMessage = messagesRepo.getPrev() != null || appendMessages.isEmpty() ? null : appendMessages.get(appendMessages.size() - 1);
            if (this.chatInteractionBinder != null) {
              this.chatInteractionBinder.setMarketingMessage(lastMessage);
            }

            view.scrollToBottom();
            view.onFetchStateChange(FetchState.COMPLETE);

            ChatAction.read(chatId);

            // need to change postDelayed
            if (messagesRepo.getNext() != null && !view.isScrollable()) {
              fetchPrevMessages();
            }
          })
          .bind(this, BindAction.FETCH_MESSAGES);
    }
  }

  @Override
  public void fetchPrevMessages() {
    if (this.chatId != null && !isRunning(BindAction.FETCH_MESSAGES) && this.prev != null)
      Api.getMessages2(this.chatId, this.prev, Const.MESSAGE_FETCH_LIMIT, Const.DESC)
          .call(repo -> {
            this.prev = repo.getNext();

            adapterModel.addMessages(repo.getMessages(), this.lastReadSession, repo.getNext());

            if (repo.getNext() != null && !view.isScrollable()) {
              unbind(BindAction.FETCH_MESSAGES);

              fetchPrevMessages();
            }
          })
          .bind(this, BindAction.FETCH_MESSAGES);
  }

  // handle action, socket

  private void onSocketReceived(Object object) {
    if (object instanceof UserChat) {
      UserChat userChat = (UserChat) object;

      // TODO : check user chat version
      if (this.chatId != null && this.chatId.equals(userChat.getId())) {
        onUserChatChanged(userChat);
      }
    }

    if (object instanceof Message) {
      Message message = (Message) object;

      if (this.chatId != null && this.chatId.equals(message.getChatId())) {
        onReceiveMessage(message);
      }
    }
  }

  private void onUserChatChanged(UserChat userChat) {
    if (userChat.isStateRemoved()) {
      view.finish(Transition.NONE);
      return;
    }

    if (this.chatHeaderBinder != null) {
      this.chatHeaderBinder.setUserChat(userChat);
    }

    if (this.chatInteractionBinder != null) {
      this.chatInteractionBinder.setUserChat(userChat);
    }

    view.onChatStateChange(userChat);
  }

  @Override
  protected void handleAction(ActionType actionType) {
    switch (actionType) {
      case SOCKET_DISCONNECTED:
      case SHUTDOWN:
        if (isRunning(BindAction.FETCH_MESSAGES)) {
          view.onFetchStateChange(FetchState.FAILED);

          unbind(BindAction.FETCH_MESSAGES);
        }

        clearQueue();

        this.chatState = ChatState.DISCONNECTED;
        this.prev = null;
        break;
    }
  }

  private void onReceiveMessage(Message message) {
    switch (this.chatState) {
      case NEWEST:
        boolean isScrollOnBottom = view.isScrollOnBottom();

        adapterModel.addMessage(message);

        if (isScrollOnBottom) {
          view.scrollToBottom();
        } else if (!CompareUtils.isSame(message.getPersonType(), Const.USER)) {
          // TODO : Show new message alert in bottom
        }

        // to refresh chat interactive state
        if (this.chatInteractionBinder != null) {
          this.chatInteractionBinder.setMarketingMessage(message);
        }
        break;

      case FETCHING:
        if (this.tempQueue != null) {
          this.tempQueue.add(message);
        }
        break;
    }
  }

  // message actions

  @Override
  public void onActionClick(@NonNull String actionType, @NonNull ActionButton actionButton) {
    if (CompareUtils.exists(actionType, ACTION_TYPE_SOLVE, ACTION_TYPE_CLOSE) &&
        CompareUtils.isSame(actionButton.getKey(), Const.ACTION_KEY_REOPEN)
    ) {
      reopenChat(actionButton);
      return;
    }

    if (this.chatId != null) {
      getChatManager().sendMessage(new SendActionItem(this.chatId, actionType, actionButton));
    }

    view.scrollToBottom();
  }

  private void reopenChat(@NonNull ActionButton actionButton) {
    Message message = actionButton.getMessage();
    message.clearAction();
    adapterModel.addMessage(message);

    if (this.chatInteractionBinder != null) {
      this.chatInteractionBinder.setTemporaryInputOpened(true);
    }

    view.scrollToBottom();
  }

  @Override
  public void deleteMessage(String messageId) {
    if (this.chatId != null) {
      Api.deleteMessage2(this.chatId, messageId)
          .call(repo -> {
            if (repo.getMessage() != null) {
              adapterModel.updateItem(new ChatMessageItem(repo.getMessage()));
            }
          })
          .bind(this);
    }
  }

  // send actions

  @Override
  public void sendText(String message) {
    if (this.chatId == null) {
      // TODO : add send item process in local chat
    } else {
      getChatManager().sendMessage(new SendTextItem(this.chatId, message));
    }

    view.scrollToBottom();
  }

  @Override
  public void resend(SendItem item) {
    if (this.chatId == null) {
      // TODO : add resend item process in local chat
    } else {
      getChatManager().resend(item);
    }

    view.scrollToBottom();
  }

  @Override
  public void removeFailedItem(SendItem item) {
    // TODO : add remove failed item process in local chat
    getChatManager().remove(item);
  }

  @Override
  public void uploadFiles(List<PhotoItem> photoItems) {
    if (photoItems != null) {
      List<SendItem> items = Stream.of(photoItems)
          .mapIndexed((IndexedFunction<PhotoItem, SendItem>) (index, item) ->
              new SendFileItem(this.chatId, index, item.getUri(), item.getName(), item.getSize())
          )
          .toList();

      if (this.chatId == null) {
        // TODO : add upload files item process in local chat
      } else {
        getChatManager().sendMessages(items);
      }

      view.scrollToBottom();
    }
  }

  @Override
  public void cancelSendingFile(SendFileItem item) {
    // TODO : add remove failed file item process in local chat
    if (item.getType() == MessageType.FAILED_FILE) {
      removeFailedItem(item);
    } else {
      getChatManager().cancelRecentSendingFile();
    }
  }

  // util functions

  private void initQueue() {
    clearQueue();

    this.tempQueue = new ArrayList<>();
  }

  private void clearQueue() {
    if (this.tempQueue != null) {
      this.tempQueue.clear();
      this.tempQueue = null;
    }
  }
}
