package com.zoyi.channel.plugin.android.activity.chat;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v7.util.SortedList;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.activity.chat.contract.ChatAdapterContract;
import com.zoyi.channel.plugin.android.activity.chat.listener.*;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.*;
import com.zoyi.channel.plugin.android.base.adapter.BaseAdapter;
import com.zoyi.channel.plugin.android.base.adapter.BaseViewHolder;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.com.annimon.stream.Stream;

import java.util.*;

public class ChatAdapter
    extends BaseAdapter<BaseViewHolder>
    implements ChatAdapterContract.View, ChatAdapterContract.Model {

  private SortedList<MessageItem> items;

  @Nullable
  private OnProfileUpdateRequestListener onProfileUpdateRequestListener;
  @Nullable
  private OnMessageContentClickListener onMessageContentClickListener;

  @Nullable
  private OnChatActionListener listener;

  @NonNull
  private TypingItem typingItem;

  @Nullable
  private InitMessageItem initMessageItem = null;

  @Nullable
  private String backwardId = null;

  public ChatAdapter(@Nullable OnChatActionListener listener) {
    this.listener = listener;

    items = new SortedList<>(MessageItem.class, new SortedListCallback<>(this, false));

    // Typing layout must always exists
    typingItem = new TypingItem();
    items.add(typingItem);
  }

  public void setOnProfileUpdateRequestListener(
      @Nullable OnProfileUpdateRequestListener onProfileUpdateRequestListener
  ) {
    this.onProfileUpdateRequestListener = onProfileUpdateRequestListener;
  }

  public void setOnMessageContentClickListener(@Nullable OnMessageContentClickListener onMessageContentClickListener) {
    this.onMessageContentClickListener = onMessageContentClickListener;
  }

  @Override
  public void setInitMessage(InitMessageItem item) {
    if (item != null && initMessageItem == null) {
      initMessageItem = item;
      items.add(item);
    }
  }

  @Override
  public void setMessages(
      Collection<Message> messages,
      Collection<SendItem> unsentItems,
      @Nullable Session lastReadSession,
      @Nullable String backwardId
  ) {
    this.backwardId = backwardId;

    List<MessageItem> newItems = purifyMessages(messages, lastReadSession);
    newItems.add(typingItem);
    newItems.addAll(unsentItems);

    items.replaceAll(newItems);
  }

  @Override
  public void replaceInitMessageItem(Message message) {
    if (initMessageItem != null) {
      items.remove(initMessageItem);
    }
    if (message != null) {
      addMessages(Collections.singleton(message), null, null);
    }
  }

  @Override
  public void addMessages(
      Collection<Message> messages,
      @Nullable Session lastReadSession,
      @Nullable String backwardId
  ) {
    this.backwardId = backwardId;
    addMessages(messages, lastReadSession);
  }

  private void addMessages(Collection<Message> messages, @Nullable Session lastReadSession) {
    List<MessageItem> newItems = purifyMessages(messages, lastReadSession);

    items.beginBatchedUpdates();

    for (MessageItem item : newItems) {
      items.add(item);
    }

    for (Message message : messages) {
      if (message == null) {
        continue;
      }

      if (CompareUtils.exists(message.getPersonType(), User.CLASSNAME)) {
        DummyItem item = DummyItem.createItem(message.getRequestId());

        if (item != null) {
          items.remove(item);
        }
      }
    }

    items.endBatchedUpdates();
  }

  @Override
  public void addMessage(Message message) {
    addMessages(Collections.singleton(message), null);
  }

  @NonNull
  private List<MessageItem> purifyMessages(Collection<Message> messages, @Nullable Session session) {
    List<MessageItem> newItems = new ArrayList<>();

    long minTimestamp = Long.MAX_VALUE;
    long maxTimestamp = Long.MIN_VALUE;

    for (Message message : messages) {
      if (message == null) {
        continue;
      }

      minTimestamp = Math.min(minTimestamp, message.getCreatedAt());
      maxTimestamp = Math.max(maxTimestamp, message.getCreatedAt());

      if (message.getLog() != null) {
        continue;
      }

      newItems.add(new ChatMessageItem(message));

      if (message.getProfileBot() != null) {
        newItems.add(new ProfileBotMessageItem(message));
      }
    }

    if (session != null && session.getReadAt() != null && minTimestamp < session.getReadAt() && session.getReadAt() < maxTimestamp) {
      newItems.add(new NewMessageItem(session.getReadAt()));
    }

    return newItems;
  }

  @Override
  public void addMessageItem(MessageItem item) {
    items.add(item);
  }

  @Override
  public void addMessageItems(Collection<? extends MessageItem> items) {
    this.items.addAll(Stream.ofNullable(items).map(it -> (MessageItem) it).toList());
  }

  @Override
  public void updateItem(MessageItem item) {
    int index = this.items.indexOf(item);

    if (index >= 0) {
      this.items.updateItemAt(index, item);
    }
  }

  @Override
  public void removeMessageItem(MessageItem item) {
    items.remove(item);
  }

  @Override
  public void setTypings(List<Typing> typings) {
    addMessageItem(typingItem.setTypings(typings));
  }

  @Nullable
  @Override
  public BaseViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
    MessageType type = MessageType.fromId(viewType);

    switch (type) {
      case NEW_MESSAGE_DIVIDER:
        return NewMessageHolder.newInstance(parent);

      case TYPING:
        return TypingHolder.newInstance(parent);

      case LOG:
        return LogMessageHolder.newInstance(parent);

      case SENDING:
        return SendingTextMessageHolder.newInstance(parent, listener);

      case SENDING_ACTION:
        return SendingActionMessageHolder.newInstance(parent, listener);

      case HOST:
        return HostMessageHolder.newInstance(parent, listener);

      case USER:
        return UserMessageHolder.newInstance(parent, listener);

      case PROFILE_BOT:
        return ProfileBotMessageHolder.newInstance(parent, onProfileUpdateRequestListener);

      case WELCOME:
        return WelcomeMessageHolder.newInstance(parent, listener);

      case SUPPORT_BOT:
        return SupportBotMessageHolder.newInstance(parent, listener);

      case FAILED_FILE:
      case SENDING_FILE:
        return UploadStateHolder.newInstance(parent, onMessageContentClickListener);
      default:
        return null;
    }
  }

  @Override
  public void onBindViewHolder(BaseViewHolder holder, int position) {
    MessageItem item = items.get(position);
    ChatMessageItem chatMessageItem;

    switch (item.getType()) {
      case TYPING:
        TypingHolder typingHolder = (TypingHolder) holder;
        typingHolder.bind((TypingItem) item);
        break;

      case LOG:
        LogMessageHolder logMessageHolder = (LogMessageHolder) holder;
        logMessageHolder.bind((LogMessageItem) item);
        break;

      case SENDING:
        SendTextItem textItem = (SendTextItem) item;
        SendingTextMessageHolder sendingTextHolder = (SendingTextMessageHolder) holder;
        sendingTextHolder.bind(
            textItem,
            !textItem.isConnected(getItem(position + 1))
        );
        break;

      case SENDING_ACTION:
        SendActionItem actionItem = (SendActionItem) item;
        SendingActionMessageHolder sendingActionHolder = (SendingActionMessageHolder) holder;
        sendingActionHolder.bind(actionItem);
        break;

      case HOST:
        chatMessageItem = (ChatMessageItem) item;
        HostMessageHolder hostMessageHolder = (HostMessageHolder) holder;

        boolean isHead = chatMessageItem.getMessage().hasContents()
            ? !chatMessageItem.isConnected(getItem(position - 1))
            : !chatMessageItem.isSamePerson(getItem(position - 1)) && isLastPosition(position);

        hostMessageHolder.bind(
            chatMessageItem,
            !chatMessageItem.isSameDate(getItem(position - 1)) || isFirstMessage(position),
            isHead,
            !chatMessageItem.isConnected(getItem(position + 1)),
            isLastPosition(position)
        );
        break;

      case USER:
        chatMessageItem = (ChatMessageItem) item;
        UserMessageHolder userMessageHolder = (UserMessageHolder) holder;
        userMessageHolder.bind(
            chatMessageItem,
            !chatMessageItem.isSameDate(getItem(position - 1)) || isFirstMessage(position),
            !chatMessageItem.isConnected(getItem(position - 1)),
            !chatMessageItem.isConnected(getItem(position + 1))
        );
        break;

      case PROFILE_BOT:
        ProfileBotMessageItem profileItem = (ProfileBotMessageItem) item;
        ProfileBotMessageHolder profileBotMessageHolder = (ProfileBotMessageHolder) holder;
        profileBotMessageHolder.bind(profileItem);
        break;

      case WELCOME:
        WelcomeMessageItem welcomeMessageItem = (WelcomeMessageItem) item;
        WelcomeMessageHolder welcomeMessageHolder = (WelcomeMessageHolder) holder;
        welcomeMessageHolder.bind(welcomeMessageItem);
        break;

      case SUPPORT_BOT:
        SupportBotMessageItem supportBotMessageItem = (SupportBotMessageItem) item;
        SupportBotMessageHolder supportBotMessageHolder = (SupportBotMessageHolder) holder;
        supportBotMessageHolder.bind(supportBotMessageItem, isLastPosition(position));
        break;

      case FAILED_FILE:
      case SENDING_FILE:
        SendFileItem sendFileItem = (SendFileItem) item;
        UploadStateHolder uploadStateHolder = (UploadStateHolder) holder;
        uploadStateHolder.bind(sendFileItem);
        break;
    }
  }

  @Nullable
  public MessageItem getItem(int position) {
    if (position >= 0 && position < items.size()) {
      return items.get(position);
    }
    return null;
  }

  private boolean isLastPosition(int position) {
    MessageItem item = getItem(position + 1);

    return item == null || (item.getPrimaryKey() != null && item.getPrimaryKey() > Const.MESSAGE_PRIMARY_KEY_MAX);
  }

  // on top message when backward id is non null, show date raise layout flick
  private boolean isFirstMessage(int position) {
    return position == 0 && backwardId == null;
  }

  @Override
  public int getItemViewType(int position) {
    return items.get(position).getType().toInt();
  }

  @Override
  public int getItemCount() {
    return items.size();
  }
}
