package com.zoyi.channel.plugin.android.view.youtube.player;

import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.webkit.JavascriptInterface;

import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerListener;

import java.util.Collection;

/**
 * Bridge used for Javascript-Java communication.
 */
public class YouTubePlayerBridge {

  // these constants correspond to the values in the Javascript player
  private static final String STATE_UNSTARTED = "UNSTARTED";
  private static final String STATE_ENDED = "ENDED";
  private static final String STATE_PLAYING = "PLAYING";
  private static final String STATE_PAUSED = "PAUSED";
  private static final String STATE_BUFFERING = "BUFFERING";
  private static final String STATE_CUED = "CUED";

  private static final String QUALITY_SMALL = "small";
  private static final String QUALITY_MEDIUM = "medium";
  private static final String QUALITY_LARGE = "large";
  private static final String QUALITY_HD720 = "hd720";
  private static final String QUALITY_HD1080 = "hd1080";
  private static final String QUALITY_HIGH_RES = "highres";
  private static final String QUALITY_DEFAULT = "default";

  private static final String RATE_0_25 = "0.25";
  private static final String RATE_0_5 = "0.5";
  private static final String RATE_1 = "1";
  private static final String RATE_1_5 = "1.5";
  private static final String RATE_2 = "2";

  private static final String ERROR_INVALID_PARAMETER_IN_REQUEST = "2";
  private static final String ERROR_HTML_5_PLAYER = "5";
  private static final String ERROR_VIDEO_NOT_FOUND = "100";
  private static final String ERROR_VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER1 = "101";
  private static final String ERROR_VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER2 = "150";

  private YouTubePlayerBridgeCallbacks youTubePlayerOwner;

  private Handler mainThreadHandler = new Handler(Looper.getMainLooper());

  public YouTubePlayerBridge(YouTubePlayerBridgeCallbacks youTubePlayerOwner) {
    this.youTubePlayerOwner = youTubePlayerOwner;
  }

  public interface YouTubePlayerBridgeCallbacks {

    YouTubePlayer getInstance();

    Collection<YouTubePlayerListener> getListeners();

    void onYouTubeIFrameAPIReady();
  }

  @JavascriptInterface
  public void sendYouTubeIFrameAPIReady() {
    mainThreadHandler.post(() -> youTubePlayerOwner.onYouTubeIFrameAPIReady());
  }


  @JavascriptInterface
  public void sendReady() {
    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onReady(youTubePlayerOwner.getInstance());
      }
    });
  }

  @JavascriptInterface
  public void sendStateChange(String state) {
    PlayerConstants.PlayerState playerState = parsePlayerState(state);

    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onStateChange(youTubePlayerOwner.getInstance(), playerState);
      }
    });
  }

  @JavascriptInterface
  public void sendPlaybackQualityChange(String quality) {
    PlayerConstants.PlaybackQuality playbackQuality = parsePlaybackQuality(quality);

    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onPlaybackQualityChange(youTubePlayerOwner.getInstance(), playbackQuality);
      }
    });
  }

  @JavascriptInterface
  public void sendPlaybackRateChange(String rate) {
    PlayerConstants.PlaybackRate playbackRate = parsePlaybackRate(rate);

    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onPlaybackRateChange(youTubePlayerOwner.getInstance(), playbackRate);
      }
    });
  }

  @JavascriptInterface
  public void sendError(String error) {
    PlayerConstants.PlayerError playerError = parsePlayerError(error);

    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onError(youTubePlayerOwner.getInstance(), playerError);
      }
    });
  }

  @JavascriptInterface
  public void sendApiChange() {
    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onApiChange(youTubePlayerOwner.getInstance());
      }
    });
  }

  @JavascriptInterface
  public void sendVideoCurrentTime(String seconds) {
    float currentTimeSeconds;
    try {
      currentTimeSeconds = Float.valueOf(seconds);
    } catch (NumberFormatException e) {
      e.printStackTrace();
      return;
    }

    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onCurrentSecond(youTubePlayerOwner.getInstance(), currentTimeSeconds);
      }
    });
  }

  @JavascriptInterface
  public void sendVideoDuration(String seconds) {
    float videoDuration;

    try {
      String finalSeconds = TextUtils.isEmpty(seconds) ? "0" : seconds;
      videoDuration = Float.valueOf(finalSeconds);
    } catch (NumberFormatException e) {
      e.printStackTrace();
      return;
    }

    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onVideoDuration(youTubePlayerOwner.getInstance(), videoDuration);
      }
    });
  }

  @JavascriptInterface
  public void sendVideoLoadedFraction(String fraction) {
    float loadedFraction;
    try {
      loadedFraction = Float.valueOf(fraction);
    } catch (NumberFormatException e) {
      e.printStackTrace();
      return;
    }

    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onVideoLoadedFraction(youTubePlayerOwner.getInstance(), loadedFraction);
      }
    });
  }

  @JavascriptInterface
  public void sendVideoId(String videoId) {
    mainThreadHandler.post(() -> {
      for (YouTubePlayerListener listener : youTubePlayerOwner.getListeners()) {
        listener.onVideoId(youTubePlayerOwner.getInstance(), videoId);
      }
    });
  }

  private PlayerConstants.PlayerState parsePlayerState(String state) {
    switch (state) {
      case STATE_UNSTARTED:
        return PlayerConstants.PlayerState.UNSTARTED;
      case STATE_ENDED:
        return PlayerConstants.PlayerState.ENDED;
      case STATE_PLAYING:
        return PlayerConstants.PlayerState.PLAYING;
      case STATE_PAUSED:
        return PlayerConstants.PlayerState.PAUSED;
      case STATE_BUFFERING:
        return PlayerConstants.PlayerState.BUFFERING;
      case STATE_CUED:
        return PlayerConstants.PlayerState.VIDEO_CUED;
      default:
        return PlayerConstants.PlayerState.UNKNOWN;
    }
  }

  private PlayerConstants.PlaybackQuality parsePlaybackQuality(String quality) {
    switch (quality) {
      case QUALITY_SMALL:
        return PlayerConstants.PlaybackQuality.SMALL;
      case QUALITY_MEDIUM:
        return PlayerConstants.PlaybackQuality.MEDIUM;
      case QUALITY_LARGE:
        return PlayerConstants.PlaybackQuality.LARGE;
      case QUALITY_HD720:
        return PlayerConstants.PlaybackQuality.HD720;
      case QUALITY_HD1080:
        return PlayerConstants.PlaybackQuality.HD1080;
      case QUALITY_HIGH_RES:
        return PlayerConstants.PlaybackQuality.HIGH_RES;
      case QUALITY_DEFAULT:
        return PlayerConstants.PlaybackQuality.DEFAULT;
      default:
        return PlayerConstants.PlaybackQuality.UNKNOWN;
    }
  }

  private PlayerConstants.PlaybackRate parsePlaybackRate(String rate) {
    switch (rate) {
      case RATE_0_25:
        return PlayerConstants.PlaybackRate.RATE_0_25;
      case RATE_0_5:
        return PlayerConstants.PlaybackRate.RATE_0_5;
      case RATE_1:
        return PlayerConstants.PlaybackRate.RATE_1;
      case RATE_1_5:
        return PlayerConstants.PlaybackRate.RATE_1_5;
      case RATE_2:
        return PlayerConstants.PlaybackRate.RATE_2;
      default:
        return PlayerConstants.PlaybackRate.UNKNOWN;
    }
  }

  private PlayerConstants.PlayerError parsePlayerError(String error) {
    switch (error) {
      case ERROR_INVALID_PARAMETER_IN_REQUEST:
        return PlayerConstants.PlayerError.INVALID_PARAMETER_IN_REQUEST;
      case ERROR_HTML_5_PLAYER:
        return PlayerConstants.PlayerError.HTML_5_PLAYER;
      case ERROR_VIDEO_NOT_FOUND:
        return PlayerConstants.PlayerError.VIDEO_NOT_FOUND;
      case ERROR_VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER1:
      case ERROR_VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER2:
        return PlayerConstants.PlayerError.VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER;
      default:
        return PlayerConstants.PlayerError.UNKNOWN;
    }
  }
}
