package com.zoyi.channel.plugin.android.util;

import android.content.Context;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.activity.base.BaseActivity2;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.TimeRange;
import com.zoyi.com.annimon.stream.Stream;

import java.util.Calendar;
import java.util.SimpleTimeZone;

public class ChannelUtils {

  private final static long MINUTES_TO_MILLISECONDS = 60L * 1000L;

  private final static int WEEK_COUNT = 7;

  private final static int DAY_IN_MINUTES = 24 * 60;

  private final static int WEEK_IN_MINUTES = WEEK_COUNT * DAY_IN_MINUTES;

  public static boolean isChannelPluginActivity(Context context) {
    return context instanceof BaseActivity2;
  }

  public static boolean isWorking(@Nullable Channel channel, @Nullable Long timestamp) {
    return getMinutesLeftToWork(channel, timestamp) == 0;
  }

  /**
   * Get remaining minutes to working time for channel
   *
   * @param channel   channel information
   * @param timestamp timestamp in millseconds for now without timezone
   * @return {@code -1} if not working and can not calculate remaining time {@code 0}  if channel is now working else, remaining time in
   * minutes to working time
   */
  public static int getMinutesLeftToWork(@Nullable Channel channel, @Nullable Long timestamp) {
    if (channel == null || timestamp == null || timestamp == 0L) {
      return -1;
    }
    if (Const.WORKING_TYPE_ALWAYS.equals(channel.getWorkingType())) {
      return 0;
    }
    if (Const.WORKING_TYPE_NEVER.equals(channel.getWorkingType())) {
      return -1;
    }

    if (!channel.hasValidWorkingTime()) {
      return 0;
    }

    long utcOffsetMilliseconds = channel.getUtcOffsetMinutes() * MINUTES_TO_MILLISECONDS;
    long now = timestamp + utcOffsetMilliseconds;

    Calendar calendar = Calendar.getInstance(new SimpleTimeZone(0, "UTC"));
    calendar.setTimeInMillis(now);

    int todayWeekday = (calendar.get(Calendar.DAY_OF_WEEK) + 5) % WEEK_COUNT; // monday is 0, sunday is 6
    int hours = calendar.get(Calendar.HOUR_OF_DAY);
    int minutes = calendar.get(Calendar.MINUTE);
    int nowMinutes = hours * 60 + minutes;

    return Stream.ofNullable(channel.getWorkingTimeRanges())
        .map(timeRange -> getRemainingMinuteForTimeRange(timeRange, todayWeekday, nowMinutes))
        .min(Integer::compareTo)
        .map(it -> it == Integer.MAX_VALUE ? -1 : it)
        .orElse(-1);
  }

  /**
   * Get remaining minutes to working time for channel
   *
   * @param timeRange  time range for weekday
   * @param nowWeekDay current weekDay
   * @param nowMinutes target timestamp
   * @return {@code Integer.MAX_VALUE} If this weekday did not working and must see next weekday. Return 0 or positive number for remaining
   * minutes. That weekday is working.
   */
  private static int getRemainingMinuteForTimeRange(TimeRange timeRange, int nowWeekDay, int nowMinutes) {
    return Stream.ofNullable(timeRange.getDayOfWeeks())
        .map(weekDay -> {
          int weekDayDiff = (weekDay.getIndex() - nowWeekDay + WEEK_COUNT) % WEEK_COUNT;

          if (weekDayDiff == 0 && timeRange.getFrom() <= nowMinutes && nowMinutes <= timeRange.getTo()) {
            return 0;
          }
          return (weekDayDiff * DAY_IN_MINUTES + timeRange.getFrom() - nowMinutes + WEEK_IN_MINUTES) % WEEK_IN_MINUTES;
        })
        .min(Integer::compareTo)
        .orElse(Integer.MAX_VALUE);
  }
}
