package com.zoyi.channel.plugin.android.view.media_thumbnail;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.util.FormatUtils;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.view.layout.ChBorderLayout;
import com.zoyi.channel.plugin.android.view.video_player.AbsVideoPlayerView;
import com.zoyi.com.bumptech.glide.Glide;
import com.zoyi.com.bumptech.glide.load.engine.DiskCacheStrategy;

public abstract class MediaThumbnailView extends ChBorderLayout {

  public MediaThumbnailView(@NonNull Context context) {
    super(context);

    init(context);
  }

  public MediaThumbnailView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);

    init(context);
  }

  public MediaThumbnailView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);

    init(context);
  }

  private Context context;

  private View viewPlayerController;
  private ImageView imageMediaThumbnail;
  private FrameLayout layoutMediaThumbnailVideos;
  private View viewPlayerDuration;
  private TextView textPlayerDuration;

  @Initializer
  private void init(@NonNull Context context) {
    this.context = context;

    LayoutInflater.from(context).inflate(R.layout.ch_view_media_thumbnail, this);

    viewPlayerController = findViewById(R.id.ch_viewPlayerController);
    imageMediaThumbnail = findViewById(R.id.ch_imageMediaThumbnail);
    layoutMediaThumbnailVideos = findViewById(R.id.ch_layoutMediaThumbnailVideos);
    viewPlayerDuration = findViewById(R.id.ch_viewPlayerDuration);
    textPlayerDuration = findViewById(R.id.ch_textPlayerDuration);

    viewPlayerController.setOnClickListener(v -> onPlayClick());
  }

  @Override
  public void setOnClickListener(@Nullable View.OnClickListener l) {
    imageMediaThumbnail.setOnClickListener(l);
  }

  @Override
  public void setOnLongClickListener(@Nullable View.OnLongClickListener l) {
    viewPlayerController.setOnLongClickListener(l);
    imageMediaThumbnail.setOnLongClickListener(l);
  }

  protected void setImage(@Nullable String url) {
    setImage(url, ImageView.ScaleType.CENTER_CROP);
  }

  protected void setImage(@Nullable String url, ImageView.ScaleType scaleType) {
    try {
      imageMediaThumbnail.setScaleType(scaleType);
      if (scaleType == ImageView.ScaleType.FIT_CENTER) {
        Glide.with(context)
            .load(url)
            .fitCenter()
            .diskCacheStrategy(DiskCacheStrategy.SOURCE)
            .into(imageMediaThumbnail);
      } else {
        Glide.with(context)
            .load(url)
            .centerCrop()
            .diskCacheStrategy(DiskCacheStrategy.SOURCE)
            .into(imageMediaThumbnail);
      }
    } catch (Exception e) {
    }
  }

  protected void setImage(@Nullable String url, int width, int height) {
    try {
      Glide.with(context)
          .load(url)
          .centerCrop()
          .override(width, height)
          .diskCacheStrategy(DiskCacheStrategy.SOURCE)
          .into(imageMediaThumbnail);
    } catch (Exception e) {
    }
  }

  protected <E extends AbsVideoPlayerView> E appendVideoView(E view) {
    if (layoutMediaThumbnailVideos != null) {
      layoutMediaThumbnailVideos.setVisibility(View.VISIBLE);
      layoutMediaThumbnailVideos.addView(view);
    }

    return view;
  }

  protected void activatePlayButton() {
    viewPlayerController.setVisibility(View.VISIBLE);
  }

  protected void activateDuration(@Nullable Double duration) {
    Long durationMilliseconds = duration != null ? duration.longValue() : null;
    String formattedDuration = FormatUtils.formatDuration(durationMilliseconds);

    if (formattedDuration != null && viewPlayerDuration != null) {
      viewPlayerDuration.setVisibility(View.VISIBLE);
      textPlayerDuration.setText(formattedDuration);
    } else if (viewPlayerDuration != null) {
      viewPlayerDuration.setVisibility(View.GONE);
    }
  }

  public void onPlayClick() {
    imageMediaThumbnail.setVisibility(View.GONE);
    viewPlayerController.setVisibility(View.GONE);
    viewPlayerDuration.setVisibility(View.GONE);
  }

  public void clear() {
    // image: visible
    if (imageMediaThumbnail != null) {
      imageMediaThumbnail.setVisibility(View.VISIBLE);

      Glide.clear(imageMediaThumbnail);
    }

    if (layoutMediaThumbnailVideos != null) {
      for (int i = 0; i < layoutMediaThumbnailVideos.getChildCount(); i++) {
        View child = layoutMediaThumbnailVideos.getChildAt(i);

        if (child instanceof AbsVideoPlayerView) {
          ((AbsVideoPlayerView) child).clear();
        }
      }

      layoutMediaThumbnailVideos.removeAllViews();
      layoutMediaThumbnailVideos.setVisibility(View.GONE);
    }

    // video controller: gone
    if (viewPlayerController != null) {
      viewPlayerController.setVisibility(View.GONE);
    }

    // video duration: gone
    if (viewPlayerDuration != null) {
      viewPlayerDuration.setVisibility(View.GONE);
    }
  }
}
