package com.zoyi.channel.plugin.android.activity.chat.viewholder;

import android.graphics.Point;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageContentClickListener;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.chat.view.action.ActionButtonLayout;
import com.zoyi.channel.plugin.android.activity.chat.view.action.ActionButtonView;
import com.zoyi.channel.plugin.android.enumerate.TranslationState;
import com.zoyi.channel.plugin.android.model.ActionButton;
import com.zoyi.channel.plugin.android.model.MessageButton;
import com.zoyi.channel.plugin.android.model.TranslationInfo;
import com.zoyi.channel.plugin.android.model.entity.Previewable;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.WebPage;
import com.zoyi.channel.plugin.android.store2.TranslationStore;
import com.zoyi.channel.plugin.android.util.ListUtils;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.util.Utils;
import com.zoyi.channel.plugin.android.util.Views;
import com.zoyi.channel.plugin.android.util.draw.Resizer;
import com.zoyi.channel.plugin.android.view.button.MessageButtonView;
import com.zoyi.channel.plugin.android.view.external.cpv.CircularProgressView;
import com.zoyi.channel.plugin.android.view.layout.MaskableFrameLayout;
import com.zoyi.com.bumptech.glide.Glide;
import com.zoyi.com.bumptech.glide.load.engine.DiskCacheStrategy;
import com.zoyi.com.bumptech.glide.load.resource.drawable.GlideDrawable;
import com.zoyi.com.bumptech.glide.request.RequestListener;
import com.zoyi.com.bumptech.glide.request.target.Target;

import java.util.List;

/**
 * Created by jerry on 2018. 10. 22..
 */

public abstract class ContentMessageHolder extends TextMessageHolder {

  private View rootContentMessageFile;
  private ImageView imageContentMessageFile;
  private TextView textContentMessageFileName;
  private TextView textContentMessageFileInfo;

  private MaskableFrameLayout rootContentMessageImage;
  private MaskableFrameLayout frameContentMessageImage;
  private ImageView imageContentMessageImage;
  private ImageView imageContentMessagePlaceHolder;

  private ViewGroup rootContentMessageWebPage;
  private View layoutContentMessageWebDescription;
  private TextView textContentMessageWebDescriptionTitle;
  private TextView textContentMessageWebDescriptionUrl;
  private View layoutContentMessageWebPreview;
  private ImageView imageContentMessageWebPreview;

  private View rootContentMessageTranslation;
  private LinearLayout buttonContentMessageTranslation;
  private TextView textContentMessageTranslation;
  private ImageView viewContentMessageTranslationButtonArrow;
  private CircularProgressView progressContentMessageTranslation;

  private LinearLayout rootExtraContentMessageAction;
  private ActionButtonLayout layoutExtraContentMessageActionButton;

  private ViewGroup rootContentsMessageButton;
  private ViewGroup layoutContentsMessageButton;

  private MaskableFrameLayout rootContentsMessageLinkableImage;
  private MaskableFrameLayout frameContentsMessageLinkableImage;
  private ImageView imageContentsMessageLinkableImage;
  private ImageView imageContentsMessageLinkableImagePlaceHolder;

  @Nullable
  protected OnMessageContentClickListener onMessageContentClickListener;

  @Nullable
  protected Message message;

  protected ContentMessageHolder(
      View itemView,
      MessageType messageType,
      @Nullable final OnMessageContentClickListener onMessageContentClickListener
  ) {
    super(itemView, messageType);

    this.onMessageContentClickListener = onMessageContentClickListener;

    rootContentMessageFile = itemView.findViewById(R.id.ch_root_message_content_file);
    imageContentMessageFile = itemView.findViewById(R.id.ch_image_message_content_file_icon);
    textContentMessageFileName = itemView.findViewById(R.id.ch_text_message_content_file_name);
    textContentMessageFileInfo = itemView.findViewById(R.id.ch_text_message_content_file_info);

    rootContentMessageImage = itemView.findViewById(R.id.ch_root_message_content_image);
    imageContentMessageImage = itemView.findViewById(R.id.ch_image_message_content);
    imageContentMessagePlaceHolder = itemView.findViewById(R.id.ch_image_message_content_placeholder);
    frameContentMessageImage = itemView.findViewById(R.id.ch_frame_message_content_image);

    rootContentMessageWebPage = itemView.findViewById(R.id.ch_root_message_content_web_page);

    layoutContentMessageWebPreview = itemView.findViewById(R.id.ch_layout_message_content_web_preview);
    imageContentMessageWebPreview = itemView.findViewById(R.id.ch_image_message_content_web_preview);

    layoutContentMessageWebDescription = itemView.findViewById(R.id.ch_layout_message_content_web_description);
    textContentMessageWebDescriptionTitle = itemView.findViewById(R.id.ch_text_message_content_web_description_title);
    textContentMessageWebDescriptionUrl = itemView.findViewById(R.id.ch_text_message_content_web_description_url);

    rootContentMessageTranslation = itemView.findViewById(R.id.ch_root_message_content_translation);
    buttonContentMessageTranslation = itemView.findViewById(R.id.ch_layout_message_content_translation_button);
    textContentMessageTranslation = itemView.findViewById(R.id.ch_text_content_message_translation);
    viewContentMessageTranslationButtonArrow = itemView.findViewById(R.id.ch_image_content_message_translation_arrow);
    progressContentMessageTranslation = itemView.findViewById(R.id.ch_progress_content_message_translation);

    rootExtraContentMessageAction = itemView.findViewById(R.id.ch_root_extra_content_form);
    layoutExtraContentMessageActionButton = itemView.findViewById(R.id.ch_layout_extra_content_form_button);

    rootContentsMessageButton = itemView.findViewById(R.id.ch_root_message_contents_button);
    layoutContentsMessageButton = itemView.findViewById(R.id.ch_layout_message_contents_button);

    rootContentsMessageLinkableImage = itemView.findViewById(R.id.ch_root_message_contents_linkable_image);
    frameContentsMessageLinkableImage = itemView.findViewById(R.id.ch_linkable_frame_message_contents_image);
    imageContentsMessageLinkableImage = itemView.findViewById(R.id.ch_linkable_image_message_contents);
    imageContentsMessageLinkableImagePlaceHolder = itemView.findViewById(R.id.ch_linkable_image_message_contents_placeholder);

    // Set file description click event
    rootContentMessageFile.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        onFileClick();
      }
    });

    // Set image click event
    imageContentMessageImage.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        onImageClick();
      }
    });

    // Set web page description click event
    rootContentMessageWebPage.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        onWebPageDescriptionClick();
      }
    });

    // Set translation button click event
    textContentMessageTranslation.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        onTranslationButtonClick();
      }
    });

    // Set linkable image click event
    rootContentsMessageLinkableImage.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        onLinkableImageClick();
      }
    });
  }

  public void bind(@Nullable Message message, MessageType messageType, boolean isConnectedMessage) {
    super.bind(message, messageType, isConnectedMessage);
    this.message = message;
  }

  @Override
  protected void resetViews() {
    super.resetViews();

    imageContentMessagePlaceHolder.setVisibility(View.GONE);

    layoutContentMessageWebPreview.setVisibility(View.GONE);
    layoutContentMessageWebDescription.setVisibility(View.GONE);
  }

  // Set click events

  protected void onFileClick() {
    if (onMessageContentClickListener != null && message != null) {
      onMessageContentClickListener.onFileClick(message.getFile());
    }
  }

  protected void onImageClick() {
    if (onMessageContentClickListener != null && message != null && message.getFile() != null) {
      onMessageContentClickListener.onImageClick(message.getFile());
    }
  }

  protected void onWebPageDescriptionClick() {
    if (onMessageContentClickListener != null && message != null && message.getWebPage() != null) {
      onMessageContentClickListener.onWebPageClick(message.getWebPage().getUrl());
    }
  }

  protected void onTranslationButtonClick() {
    if (onMessageContentClickListener != null && message != null) {
      TranslationInfo translationInfo = TranslationStore.get().translation.get(message.getId());

      if (translationInfo == null || translationInfo.getState() != TranslationState.PROGRESS) {
        onMessageContentClickListener.onTranslationButtonClick(message);
      }
    }
  }

  protected void onLinkableImageClick() {
  }

  // File

  protected void setFileDescription(File file) {
    String fileSize = Utils.getProperBytes(file.getSize());
    String fileType = "";
    String extension = file.getExtension();

    if (!"default".equals(extension) && !TextUtils.isEmpty(extension)) {
      fileType = String.format(" · %s", extension.substring(0, 1).toUpperCase() + extension.substring(1));
    }

    rootContentMessageFile.setVisibility(View.VISIBLE);

    textContentMessageFileName.setText(file.getFilename());
    textContentMessageFileInfo.setText(String.format("%s%s", fileSize, fileType));
    imageContentMessageFile.setImageResource(ResUtils.getFileIconResourceId(file));
  }

  // Image

  protected void setImage(@Nullable Previewable previewable) {
    if (previewable == null) {
      return;
    }

    rootContentMessageImage.setVisibility(View.VISIBLE);
    imageContentMessagePlaceHolder.setVisibility(View.VISIBLE);

    Point previewImageSize = Utils.resizeImage(context, new Point(previewable.getWidth(), previewable.getHeight()), false);
    Resizer.size(frameContentMessageImage, previewImageSize.x, previewImageSize.y);

    Glide.with(itemView.getContext())
        .load(previewable.getUrl())
        .diskCacheStrategy(DiskCacheStrategy.SOURCE)
        .override(previewImageSize.x, previewImageSize.y)
        .centerCrop()
        .listener(new RequestListener<String, GlideDrawable>() {
          @Override
          public boolean onException(Exception e, String model, Target<GlideDrawable> target, boolean isFirstResource) {
            return false;
          }

          @Override
          public boolean onResourceReady(
              GlideDrawable resource,
              String model,
              Target<GlideDrawable> target,
              boolean isFromMemoryCache,
              boolean isFirstResource
          ) {
            imageContentMessagePlaceHolder.setVisibility(View.GONE);
            return false;
          }
        })
        .into(imageContentMessageImage);
  }

  // Web page

  protected void setWebPage(final WebPage webPage) {

    rootContentMessageWebPage.setVisibility(View.VISIBLE);

    // Web description
    layoutContentMessageWebDescription.setVisibility(View.VISIBLE);

    if (!TextUtils.isEmpty(webPage.getTitle())) {
      textContentMessageWebDescriptionTitle.setText(webPage.getTitle());
    } else {
      textContentMessageWebDescriptionTitle.setText(webPage.getUrl() != null ? webPage.getUrl() : "");
    }

    if (!TextUtils.isEmpty(webPage.getDescription())) {
      textContentMessageWebDescriptionUrl.setText(webPage.getTitle());
    } else {
      textContentMessageWebDescriptionUrl.setText(webPage.getUrl() != null ? webPage.getUrl() : "");
    }

    // Web preview
    if (webPage.getPreviewThumb() != null) {
      layoutContentMessageWebPreview.setVisibility(View.VISIBLE);

      Point previewImageSize = Utils.resizeImage(
          context,
          new Point(webPage.getPreviewThumb().getWidth(), webPage.getPreviewThumb().getHeight()),
          true
      );
      Resizer.size(layoutContentMessageWebPreview, previewImageSize.x, previewImageSize.y);

      Glide.with(itemView.getContext())
          .load(webPage.getPreviewThumb().getUrl())
          .diskCacheStrategy(DiskCacheStrategy.SOURCE)
          .override(previewImageSize.x, previewImageSize.y)
          .centerCrop()
          .into(imageContentMessageWebPreview);
    }
  }

  // Translation

  protected void setTranslationButton(final Message message, boolean isConnectedMessage) {
    rootContentMessageTranslation.setVisibility(View.VISIBLE);

    CharSequence messageContent = message.getFormattedSpanMessage();

    TranslationInfo translationInfo = TranslationStore.get().translation.get(message.getId());

    TranslationState translationState = translationInfo == null ? TranslationState.ORIGIN : translationInfo.getState();

    switch (translationState) {
      case ORIGIN:
        setTranslateButtonWithoutProgress(ResUtils.getString("show_translate"), false);
        break;

      case TRANSLATED:
        setTranslateButtonWithoutProgress(ResUtils.getString("undo_translate"), true);
        if (translationInfo != null) {
          messageContent = translationInfo.getTranslatedMessage();
        }
        break;

      case PROGRESS:
        buttonContentMessageTranslation.setVisibility(View.INVISIBLE);
        progressContentMessageTranslation.setVisibility(View.VISIBLE);
        break;
    }

    setHostMessageText(messageContent, isConnectedMessage);
  }

  private void setTranslateButtonWithoutProgress(String text, boolean isTranslated) {
    buttonContentMessageTranslation.setVisibility(View.VISIBLE);
    progressContentMessageTranslation.setVisibility(View.GONE);
    textContentMessageTranslation.setText(text);
    Views.setVisibility(viewContentMessageTranslationButtonArrow, isTranslated);
  }

  // Form

  protected void setAction(@Nullable String actionType, @Nullable List<ActionButton> buttons) {
    if (actionType != null && ListUtils.hasItems(buttons)) {
      rootExtraContentMessageAction.setVisibility(View.VISIBLE);

      layoutExtraContentMessageActionButton.removeAllViews();
      layoutExtraContentMessageActionButton.refresh();

      // ch_plugin_item_message_extra_content_form paddingStart 값과 일치해야함
      layoutExtraContentMessageActionButton.setWidth(Utils.getWindowSize(context).x - ((int) Utils.dpToPx(context, 40f)));

      for (ActionButton actionButton : buttons) {
        ActionButtonView actionButtonView = new ActionButtonView(context, actionType, actionButton, onMessageContentClickListener);

        layoutExtraContentMessageActionButton.add(actionButtonView);
      }
    }
  }

  protected void setMessageButtons(@NonNull List<MessageButton> messageButtons) {
    rootContentsMessageButton.setVisibility(View.VISIBLE);
    layoutContentsMessageButton.removeAllViews();

    for (MessageButton messageButton : messageButtons) {
      if (messageButton.getUrl() != null) {
        layoutContentsMessageButton.addView(createMessageButton(messageButton.getTitle(), messageButton.getUrl()));
      }
    }
  }

  private MessageButtonView createMessageButton(@NonNull String buttonTitle, final @NonNull String buttonRedirectUrl) {
    MessageButtonView messageButtonView = new MessageButtonView(context);
    messageButtonView.setButtonTitle(buttonTitle);
    messageButtonView.setButtonClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        if (onMessageContentClickListener != null) {
          onMessageContentClickListener.onRedirectContentsClick(buttonRedirectUrl);
        }
      }
    });

    return messageButtonView;
  }

  protected void setLinkableImage(@NonNull Previewable previewable) {
    rootContentsMessageLinkableImage.setVisibility(View.VISIBLE);
    imageContentsMessageLinkableImagePlaceHolder.setVisibility(View.VISIBLE);

    Point previewImageSize = Utils.resizeImage(context, new Point(previewable.getWidth(), previewable.getHeight()), false);
    Resizer.size(frameContentsMessageLinkableImage, previewImageSize.x, previewImageSize.y);

    Glide.with(itemView.getContext())
        .load(previewable.getUrl())
        .diskCacheStrategy(DiskCacheStrategy.SOURCE)
        .override(previewImageSize.x, previewImageSize.y)
        .centerCrop()
        .listener(new RequestListener<String, GlideDrawable>() {
          @Override
          public boolean onException(Exception e, String model, Target<GlideDrawable> target, boolean isFirstResource) {
            return false;
          }

          @Override
          public boolean onResourceReady(
              GlideDrawable resource,
              String model,
              Target<GlideDrawable> target,
              boolean isFromMemoryCache,
              boolean isFirstResource
          ) {
            imageContentsMessageLinkableImagePlaceHolder.setVisibility(View.GONE);
            return false;
          }
        })
        .into(imageContentsMessageLinkableImage);
  }
}
