package com.zoyi.channel.plugin.android;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.FrameLayout;

import com.zoyi.channel.plugin.android.activity.common.chat.ChatContentType;
import com.zoyi.channel.plugin.android.activity.lounge.LoungeActivity;
import com.zoyi.channel.plugin.android.enumerate.ExposureType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.push_bot.OnPushClickListener;
import com.zoyi.channel.plugin.android.selector2.GlobalSelector;
import com.zoyi.channel.plugin.android.selector2.PushSelector;
import com.zoyi.channel.plugin.android.store2.binder.Binder;
import com.zoyi.channel.plugin.android.util.Executor;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.view.popup.BannerPushView;
import com.zoyi.channel.plugin.android.view.popup.PopupPushView;

/**
 * Control launcher, push
 */
class ChannelView extends FrameLayout implements OnPushClickListener {

  private Context context;

  private ChannelLauncherView launcherView;
  private PopupPushView popupPushView;
  private BannerPushView bannerPushView;

  @Nullable
  private Binder visibilityBinder;

  @Nullable
  private Binder launcherVisibilityBinder;

  @Nullable
  private Binder popupBinder;

  public ChannelView(Context context) {
    super(context);
    init(context);
  }

  public ChannelView(Context context, AttributeSet attrs) {
    super(context, attrs);
    init(context);
  }

  public ChannelView(Context context, AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context);
  }

  @Initializer
  private void init(Context context) {
    this.context = context;

    View view = LayoutInflater.from(context).inflate(R.layout.ch_plugin_layout_channel_view, this, true);

    launcherView = view.findViewById(R.id.ch_view_launcher);
    popupPushView = view.findViewById(R.id.ch_view_push_popup);
    bannerPushView = view.findViewById(R.id.ch_view_push_banner);
    popupPushView.setPushBotClickListener(this);
    bannerPushView.setPushBotClickListener(this);
  }

  @Override
  protected void onAttachedToWindow() {
    super.onAttachedToWindow();

    visibilityBinder = GlobalSelector.bindBootState(booted -> setVisibility(booted ? View.VISIBLE : View.GONE));

    launcherVisibilityBinder = GlobalSelector.bindLauncherVisibility(showLauncher -> {
      if (showLauncher) {
        launcherView.show();
      } else {
        launcherView.hide();
      }
    });

    popupBinder = PushSelector.bindPopup(context, popupItem -> {
      if (popupItem != null) {

        switch (popupItem.getMobileExposureType()) {
          case IN_APP_PUSH:
          case PUSH_BOT_BOTTOM_BANNER:
            popupPushView.hideView();
            bannerPushView.show(popupItem);
            break;
          case PUSH_BOT_POPUP:
            bannerPushView.hideView();
            popupPushView.show(popupItem);
            break;
        }
      } else {
        popupPushView.hideView();
        bannerPushView.hideView();
      }
    });
  }

  @Override
  protected void onDetachedFromWindow() {
    super.onDetachedFromWindow();

    if (visibilityBinder != null) {
      visibilityBinder.unbind();
    }

    if (launcherVisibilityBinder != null) {
      launcherVisibilityBinder.unbind();
    }

    if (popupBinder != null) {
      popupBinder.unbind();
    }
  }

  @Override
  public void onPushClick(ExposureType exposureType, @NonNull String pushId) {
    switch (exposureType) {
      case PUSH_BOT_BOTTOM_BANNER:
      case PUSH_BOT_POPUP:
        startPushBotChat(pushId);
        break;
      case IN_APP_PUSH:
        ChannelIO.openChat(context, pushId);
        break;
    }
  }

  @Override
  public void onPushContentsClick(@NonNull String pushId, @Nullable String redirectUrl) {
    if (redirectUrl != null) {
      Executor.executeRedirectUrlAction(context, redirectUrl);
    } else {
      startPushBotChat(pushId);
    }
  }

  private void startPushBotChat(@NonNull String pushBotId) {
    IntentUtils.setNextActivity(context, LoungeActivity.class)
        .setTransition(Transition.SLIDE_FROM_BOTTOM)
        .putExtra(Const.EXTRA_CHAT_CONTENT_TYPE, ChatContentType.PUSH_BOT_CHAT.toString())
        .putExtra(Const.EXTRA_CHAT_CONTENT_ID, pushBotId)
        .startActivity();
  }
}
