package com.zoyi.channel.plugin.android.activity.chat3.manager.chat;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.action.ChatAction;
import com.zoyi.channel.plugin.android.action.PushBotAction;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.type.ContentType;
import com.zoyi.channel.plugin.android.activity.chat3.listener.MessageSendListener;
import com.zoyi.channel.plugin.android.enumerate.SendingState;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.ActionButton;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.wrapper.MessageWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.store2.ChatStore;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

class RealChatManager implements ChatManagerImpl {

  private String chatId;

  private Map<ContentType, Queue<SendingMessageItem>> queue;

  private Map<ContentType, SendingMessageItem> sendingItems;

  private Map<String, SendingMessageItem> failedItems;

  @Nullable
  private MessageSendListener listener;

  RealChatManager(String chatId) {
    this.chatId = chatId;
    this.queue = new HashMap<>();
    this.sendingItems = new HashMap<>();
    this.failedItems = new HashMap<>();

    for (ContentType contentType : ContentType.values()) {
      queue.put(contentType, new LinkedList<>());
    }
  }

  @Override
  public void attachListener(@Nullable MessageSendListener listener) {
    this.listener = listener;
  }

  @Override
  public void clearListener() {
    this.listener = null;
  }

  @Override
  public void send(@Nullable SendingMessageItem item) {
    send(Collections.singleton(item));
  }

  @Override
  public void send(@Nullable Collection<SendingMessageItem> items) {
    if (items != null) {
      for (SendingMessageItem item : items) {
        if (item == null) {
          continue;
        }
        if (item.getSendingState() == SendingState.WAITING) {
          Queue<SendingMessageItem> itemQueue = queue.get(item.getContentType());

          if (itemQueue != null) {
            itemQueue.add(item);
          }
        } else {
          failedItems.put(item.getRequestId(), item);
        }
      }
    }

    for (ContentType contentType : ContentType.values()) {
      send(contentType);
    }
  }

  private void send(ContentType contentType) {
    if (sendingItems.get(contentType) != null) {
      return;
    }

    Queue<SendingMessageItem> itemQueue = queue.get(contentType);
    SendingMessageItem newItem = itemQueue != null && itemQueue.size() > 0 ? itemQueue.poll() : null;

    if (newItem != null) {
      sendingItems.put(contentType, newItem);

      switch (contentType) {
        case TEXT:
          sendText(newItem);
          break;

        case FILE:
          uploadFile(newItem);
          break;

        case ACTION:
          handleAction(newItem);
          break;
      }
    }
  }

  // api call

  private void sendText(SendingMessageItem item) {
    ChatAction.sendTextMessage(chatId, item, createCommonSubscriber(ContentType.TEXT, item));
  }

  private void uploadFile(SendingMessageItem item) {
    ChatAction.uploadFile(chatId, item, createCommonSubscriber(ContentType.FILE, item));
  }

  private void handleAction(SendingMessageItem item) {
    String actionType = item.getActionType();
    ActionButton actionButton = item.getActionButton();

    if (actionType != null && actionButton != null) {
      switch (actionType) {
        case Const.ACTION_TYPE_SOLVE:
        case Const.ACTION_TYPE_CLOSE:
          switch (actionButton.getKey()) {
            case Const.ACTION_KEY_CLOSE:
              closeChat(item);
              break;

            case Const.ACTION_KEY_LIKE:
              reviewChat(item, Const.REVIEW_LIKE);
              break;

            case Const.ACTION_KEY_DISLIKE:
              reviewChat(item, Const.REVIEW_DISLIKE);
              break;

            default:
              doAction(item);
              break;
          }
          break;

        case Const.ACTION_TYPE_KEEP_PUSH_BOT:
          keepPushBot(item);
          break;

        case Const.ACTION_TYPE_SUPPORT_BOT:
          selectSupportBotStep(item);
          break;

        default:
          doAction(item);
          break;
      }
    }
  }

  private void keepPushBot(SendingMessageItem item) {
    PushBotAction.keepPushBot(chatId, createCommonSubscriber(ContentType.ACTION, item));
  }

  private void closeChat(SendingMessageItem item) {
    ChatAction.closeChat(chatId, item, createCommonSubscriber(ContentType.ACTION, item));
  }

  private void reviewChat(SendingMessageItem item, @NonNull String review) {
    ChatAction.reviewChat(chatId, review, item, createCommonSubscriber(ContentType.ACTION, item));
  }

  private void selectSupportBotStep(SendingMessageItem item) {
    ChatAction.selectSupportBotStep(chatId, item, createCommonSubscriber(ContentType.ACTION, item));
  }

  private void doAction(SendingMessageItem item) {
    ChatAction.selectForm(chatId, item, createCommonSubscriber(ContentType.ACTION, item));
  }

  private RestSubscriber<MessageWrapper> createCommonSubscriber(ContentType contentType, SendingMessageItem item) {
    return new RestSubscriber<MessageWrapper>() {
      @Override
      public void onError(RetrofitException error) {
        sendingItems.remove(contentType);

        handleFailedMessage(contentType, item);
      }

      @Override
      public void onSuccess(@NonNull MessageWrapper repo) {
        if (repo.getMessage() != null) {
          handleSucceedMessage(contentType, repo.getMessage());
        } else {
          handleFailedMessage(contentType, item);
        }
      }
    };
  }

  private void handleFailedMessage(ContentType contentType, SendingMessageItem item) {
    item.setSendingState(SendingState.FAIL);

    sendingItems.remove(contentType);

    failedItems.put(item.getRequestId(), item);

    if (listener != null) {
      listener.onFail(item);
    }

    send(contentType);
  }

  private void handleSucceedMessage(ContentType contentType, Message message) {
    sendingItems.remove(contentType);

    ChatStore.get().messages.add(message);

    send(contentType);
  }

  // resend

  @Override
  public void resend(@Nullable SendingMessageItem item) {
    if (item != null && failedItems.containsKey(item.getRequestId())) {
      SendingMessageItem newItem = SendingMessageItem.copyContentFrom(item);

      if (listener != null) {
        listener.onResend(newItem, item);
      }

      failedItems.remove(item.getRequestId());
      send(newItem);
    }
  }

  @NonNull
  @Override
  public List<SendingMessageItem> getUnsentItems() {
    List<SendingMessageItem> items = new ArrayList<>();

    for (ContentType contentType : ContentType.values()) {
      Queue<SendingMessageItem> itemQueue = queue.get(contentType);

      if (itemQueue != null) {
        items.addAll(itemQueue);
      }

      SendingMessageItem sendingItem = sendingItems.get(contentType);

      if (sendingItem != null) {
        items.add(sendingItem);
      }
    }

    items.addAll(failedItems.values());

    return items;
  }

  @Override
  public void removeFailedItem(SendingMessageItem item) {
    if (item != null) {
      failedItems.remove(item.getRequestId());
    }
  }

  @Override
  public void release() {
    this.queue.clear();
    this.sendingItems.clear();
    this.failedItems.clear();
    this.listener = null;
  }
}
