package com.zoyi.channel.plugin.android.model;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.SpannableStringBuilder;
import com.zoyi.channel.plugin.android.enumerate.ExposureType;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.entity.Entity;
import com.zoyi.channel.plugin.android.model.entity.ProfileEntity;
import com.zoyi.channel.plugin.android.model.etc.PushItem;
import com.zoyi.channel.plugin.android.model.rest.Bot;
import com.zoyi.channel.plugin.android.model.rest.PreviewThumb;
import com.zoyi.channel.plugin.android.model.rest.PushBot;
import com.zoyi.channel.plugin.android.model.rest.PushBotVariant;
import com.zoyi.channel.plugin.android.selector2.BotSelector;
import com.zoyi.channel.plugin.android.util.ChannelUtils;
import com.zoyi.channel.plugin.android.util.ParseUtils;
import com.zoyi.channel.plugin.android.util.TimeUtils;
import com.zoyi.com.annimon.stream.Optional;

/**
 * Created by jerry on 2018. 11. 23..
 */

public class PushBotItem implements Entity, PushItem {

  @NonNull
  private PushBot pushBot;
  @Nullable
  private PushBotVariant pushBotVariant;
  private long createdAt;
  private int alertCount;
  private boolean removed;
  private SpannableStringBuilder formattedSpanMessage;
  private SpannableStringBuilder formattedSpanMessageWithTitle;

  @Nullable
  private String targetActivityClassName;
  private int targetActivityHashCode;

  public static PushBotItem newInstance(@NonNull PushBot pushBot, @Nullable PushBotVariant pushBotVariant, @Nullable Activity activity) {
    return new PushBotItem(pushBot, pushBotVariant, activity);
  }

  private PushBotItem(@NonNull PushBot pushBot, @Nullable PushBotVariant pushBotVariant, @Nullable Activity activity) {
    this.pushBot = pushBot;
    this.pushBotVariant = pushBotVariant;
    this.createdAt = TimeUtils.getCurrentTime();
    this.alertCount = pushBotVariant != null ? 1 : 0;
    this.removed = false;
    this.formattedSpanMessage = pushBotVariant != null
        ? ParseUtils.parseMessage(pushBotVariant.getMessage())
        : new SpannableStringBuilder();
    this.formattedSpanMessageWithTitle = pushBotVariant != null
        ? ParseUtils.parseMessage(this.getTitle(), pushBotVariant.getMessage())
        : new SpannableStringBuilder();

    this.targetActivityClassName = Optional.ofNullable(activity).map(it -> it.getClass().getCanonicalName()).orElse(null);
    this.targetActivityHashCode = Optional.ofNullable(activity).map(Object::hashCode).orElse(0);
  }

  // PushBot variant id
  @NonNull
  @Override
  public String getId() {
    return pushBot.getId();
  }

  @Override
  public long getCreatedAt() {
    return createdAt;
  }

  @Nullable
  @Override
  public ProfileEntity getProfile() {
    if (pushBotVariant != null) {
      return BotSelector.getBotByName(pushBotVariant.getBotName());
    }
    return null;
  }

  @Nullable
  public String getBotId() {
    if (pushBotVariant != null) {
      Bot bot = BotSelector.getBotByName(pushBotVariant.getBotName());
      if (bot != null) {
        return bot.getId();
      }
    }
    return null;
  }

  @Nullable
  @Override
  public String getTitle() {
    if (pushBotVariant != null) {
      return pushBotVariant.getTitle();
    }
    return null;
  }

  @NonNull
  public String getMessage() {
    if (pushBotVariant != null) {
      return pushBotVariant.getMessage();
    }
    return "";
  }

  @NonNull
  @Override
  public SpannableStringBuilder getFormattedSpanMessage() {
    return formattedSpanMessage;
  }

  @NonNull
  @Override
  public SpannableStringBuilder getFormattedSpanMessageWithTitle() {
    return formattedSpanMessageWithTitle;
  }

  @Nullable
  public String getButtonTitle() {
    if (pushBotVariant != null) {
      return pushBotVariant.getButtonTitle();
    }
    return null;
  }

  @Nullable
  public String getButtonRedirectUrl() {
    if (pushBotVariant != null) {
      return pushBotVariant.getButtonRedirectUrl();
    }
    return null;
  }

  @Nullable
  public PreviewThumb getImageThumb() {
    if (pushBotVariant != null) {
      return pushBotVariant.getImageThumb();
    }
    return null;
  }

  @Nullable
  public String getAttachment() {
    if (pushBotVariant != null) {
      return pushBotVariant.getAttachment();
    }
    return null;
  }

  @Nullable
  public String getImageRedirectUrl() {
    if (pushBotVariant != null) {
      return pushBotVariant.getImageRedirectUrl();
    }
    return null;
  }

  @Override
  public ExposureType getMobileExposureType() {
    if (pushBotVariant != null) {
      switch (pushBotVariant.getMobileExposureType()) {
        case "popup":
          return ExposureType.PUSH_BOT_POPUP;
        default:
          return ExposureType.PUSH_BOT_BOTTOM_BANNER;
      }
    }

    return null;
  }

  @Override
  public String getContentUrl() {
    if (getAttachment() != null) {
      switch (getAttachment()) {
        case Const.PUSH_BOT_ATTACHMENT_BUTTON:
          return getButtonRedirectUrl();
        case Const.PUSH_BOT_ATTACHMENT_IMAGE:
          return getImageRedirectUrl();
      }
    }
    return null;
  }

  public int getAlertCount() {
    return alertCount;
  }

  public boolean isActivatedIn(Context context) {
    return ChannelUtils.isSameActivity(context, targetActivityClassName, targetActivityHashCode);
  }

  public void read() {
    alertCount = 0;
  }

  public void remove() {
    this.alertCount = 0;
    this.removed = true;
  }

  public boolean isActive() {
    return pushBotVariant != null && !removed;
  }
}
