package com.zoyi.channel.plugin.android.activity.chat;

import android.support.annotation.Nullable;
import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.text.TextUtils;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.activity.chat.listener.*;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.*;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.SupportBotEntry;
import com.zoyi.channel.plugin.android.store.ImageFileStore;
import com.zoyi.channel.plugin.android.store.Store;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.ListUtils;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Created by mika on 2016. 12. 8..
 */
public class ChatAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder>
    implements ChatAdapterContract.Model, ChatAdapterContract.View {

  private SortedList<MessageItem> items;

  @Nullable
  private OnProfileUpdateListener onProfileUpdateListener;
  @Nullable
  private OnMessageContentClickListener onMessageContentClickListener;

  @Nullable
  private WelcomeMessageItem welcomeMessageItem = null;

  @Nullable
  private SupportBotMessageItem supportBotMessageItem = null;

  private boolean isNewChat;
  private boolean hasChatEditTextFocus = false;

  public ChatAdapter(boolean isNewChat) {
    this.isNewChat = isNewChat;
    items = new SortedList<>(MessageItem.class, new SortedListCallback<MessageItem>(this, false));

    // Typing layout must always exists
    items.add(new TypingItem());
  }

  @Override
  public void setOnProfileUpdateListener(@Nullable OnProfileUpdateListener onProfileUpdateListener) {
    this.onProfileUpdateListener = onProfileUpdateListener;
  }

  @Override
  public void setOnMessageContentClickListener(@Nullable OnMessageContentClickListener onMessageContentClickListener) {
    this.onMessageContentClickListener = onMessageContentClickListener;
  }

  @Override
  public void addMessage(Message message, @Nullable String chatId) {
    addMessages(Collections.singletonList(message), chatId);
  }

  @Override
  public void addMessages(@Nullable List<Message> messages, @Nullable String chatId) {
    if (!ListUtils.hasItems(messages) || TextUtils.isEmpty(chatId)) {
      return;
    }

    items.beginBatchedUpdates();

    for (Message message : messages) {
      if (message == null || !CompareUtils.isSame(chatId, message.getChatId())) {
        continue;
      }

      if (isNewChat &&
          message.getBotOption() != null &&
          message.getBotOption().isWelcome() &&
          welcomeMessageItem != null) {
        items.remove(welcomeMessageItem);
        items.add(new DateItem(message.getCreatedAt()));
        items.add(new ChatMessageItem(message));

        welcomeMessageItem = null;

        continue;
      }

      if (message.getLog() != null) {
        items.add(new LogMessageItem(message));
        continue;
      }

      if (message.getFile() != null && message.getFile().isImageFile()) {
        Store.getInstance(ImageFileStore.class).add(message.getFile());
      }

      items.add(new ChatMessageItem(message));
      items.add(new DateItem(message.getCreatedAt()));

      if (message.getProfileBot() != null) {
        items.add(new ProfileBotMessageItem(message));
        continue;
      }

      if (CompareUtils.exists(message.getPersonType(), User.CLASSNAME, Veil.CLASSNAME)) {
        SendingMessageItem item = SendingMessageItem.create(message.getRequestId());
        if (item != null) {
          items.remove(item);
        }
      }
    }

    items.endBatchedUpdates();
  }

  @Override
  public void addMessageItem(MessageItem item) {
    items.add(item);
  }

  @Override
  public void addMessageItems(Collection<MessageItem> items) {
    this.items.addAll(items);
  }

  @Override
  public void removeMessageItem(@Nullable MessageItem item) {
    if (item == null) {
      return;
    }
    items.remove(item);
  }

  @Override
  public void addOrUpdateMessageItem(int position, MessageItem item) {
    if (position >= 0) {
      items.updateItemAt(position, item);
    } else {
      addMessageItem(item);
    }
  }

  @Nullable
  @Override
  public MessageItem getItem(int index) {
    if (index < 0 || index >= items.size()) {
      return null;
    }
    return items.get(index);
  }

  @Nullable
  @Override
  public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
    MessageType type = MessageType.fromId(viewType);

    switch (type) {

      case NEW_MESSAGE_DIVIDER:
        return NewMessageHolder.newInstance(parent);

      case TYPING:
        return TypingHolder.newInstance(parent);

      case DATE:
        return DateDividerHolder.newInstance(parent);

      case LOG:
        return LogMessageHolder.newInstance(parent);

      case SENDING:
        return SendingMessageHolder.newInstance(parent, type, onMessageContentClickListener);

      case HOST:
      case GUEST:
        return ChatMessageHolder.newInstance(parent, type, onMessageContentClickListener);

      case PROFILE_BOT:
        return ProfileBotMessageHolder.newInstance(parent, onProfileUpdateListener);

      case WELCOME:
        return WelcomeMessageHolder.newInstance(parent);

      case SUPPORT_BOT:
        return SupportBotMessageHolder.newInstance(parent, onMessageContentClickListener);

      default:
        return null;
    }
  }

  @Override
  public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {
    MessageItem item = items.get(position);
    ChatMessageItem chatMessageItem;

    switch (item.getType()) {
      case TYPING:
        TypingHolder typingHolder = (TypingHolder) holder;
        typingHolder.bind((TypingItem) item);
        break;

      case DATE:
        DateDividerHolder dateDividerHolder = (DateDividerHolder) holder;
        dateDividerHolder.bind((DateItem) item);
        break;

      case LOG:
        LogMessageHolder logMessageHolder = (LogMessageHolder) holder;
        logMessageHolder.bind((LogMessageItem) item);
        break;

      case SENDING:
        SendingMessageItem sendingMessageItem = (SendingMessageItem) item;
        SendingMessageHolder sendingHolder = (SendingMessageHolder) holder;
        sendingHolder.bind((SendingMessageItem) item, sendingMessageItem.isConnected(getItem(position - 1)));
        break;

      case HOST:
      case GUEST:
        chatMessageItem = (ChatMessageItem) item;
        ChatMessageHolder chatMessageHolder = (ChatMessageHolder) holder;
        chatMessageHolder.bind(
            chatMessageItem,
            chatMessageItem.isConnected(getItem(position - 1)),
            isLastPosition(position)
        );
        break;

      case PROFILE_BOT:
        ProfileBotMessageItem profileItem = (ProfileBotMessageItem) item;
        ProfileBotMessageHolder profileBotMessageHolder = (ProfileBotMessageHolder) holder;
        profileBotMessageHolder.bind(profileItem, getChatEditTextFocus());
        break;

      case WELCOME:
        WelcomeMessageItem welcomeMessageItem = (WelcomeMessageItem) item;
        WelcomeMessageHolder welcomeMessageHolder = (WelcomeMessageHolder) holder;
        welcomeMessageHolder.bind(welcomeMessageItem);
        break;

      case SUPPORT_BOT:
        SupportBotMessageItem supportBotMessageItem = (SupportBotMessageItem) item;
        SupportBotMessageHolder supportBotMessageHolder = (SupportBotMessageHolder) holder;
        supportBotMessageHolder.bind(supportBotMessageItem, isLastPosition(position));
        break;
    }
  }

  public boolean isLastPosition(int position) {
    return position == getItemCount() - 2;
  }

  @Override
  public int getItemViewType(int position) {
    return items.get(position).getType().toInt();
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  @Override
  public int getIndex(MessageItem item) {
    return items.indexOf(item);
  }

  @Nullable
  @Override
  public String getLastMessageItemId() {
    if (items.size() > 1) {
      MessageItem messageItem = items.get(items.size() - 2);
      return String.format("%s:%s", messageItem.getPrimaryKey(), messageItem.getSecondaryKey());
    }
    return null;
  }

  @Nullable
  @Override
  public WelcomeMessageItem getWelcomeMessageItem() {
    return welcomeMessageItem;
  }

  @Nullable
  @Override
  public SupportBotMessageItem getSupportBotItem() {
    return supportBotMessageItem;
  }

  @Override
  public void setWelcomeMessage(String message, Long timestamp) {
    if (this.welcomeMessageItem != null) {
      items.remove(this.welcomeMessageItem);
    }
    this.welcomeMessageItem = new WelcomeMessageItem(message, timestamp);
    addMessageItem(this.welcomeMessageItem);
  }

  @Override
  public void setSupportBotMessage(SupportBotEntry supportBotMessage, Long timestamp) {
    if (this.supportBotMessageItem != null) {
      items.remove(this.supportBotMessageItem);
    }
    this.supportBotMessageItem = new SupportBotMessageItem(supportBotMessage, timestamp);
    addMessageItem(this.supportBotMessageItem);
  }

  public void setChatEditTextFocus(boolean hasFocus) {
    hasChatEditTextFocus = hasFocus;
  }

  private boolean getChatEditTextFocus() {
    return hasChatEditTextFocus;
  }

  @Nullable
  public MessageItem get(int position) {
    if (position < 0 || position >= items.size()) {
      return null;
    }
    return items.get(position);
  }
}
