package com.zoyi.channel.plugin.android;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.os.AsyncTask;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.Size;
import android.text.TextUtils;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.splunk.mint.Mint;
import com.zoyi.channel.plugin.android.activity.chat.ChatManager;
import com.zoyi.channel.plugin.android.activity.userchat_list.UserChatListActivity;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.ApiTag;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Event;
import com.zoyi.channel.plugin.android.model.wrapper.PackageWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.push.ChannelPushClient;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.selector.ChannelIOSelector;
import com.zoyi.channel.plugin.android.selector.ChannelSelector;
import com.zoyi.channel.plugin.android.selector.GuestSelector;
import com.zoyi.channel.plugin.android.selector.PluginSelector;
import com.zoyi.channel.plugin.android.selector.TokenSelector;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.ChannelIOSettingsStore;
import com.zoyi.channel.plugin.android.store.DataStore;
import com.zoyi.channel.plugin.android.store.UiStateStore;
import com.zoyi.channel.plugin.android.store.Store;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.wrapper.CHDefaultEvent;
import com.zoyi.rx.Observable;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Func1;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.Map;
import java.util.concurrent.RejectedExecutionException;

public class ChannelIO {
  private static final String MINT_DEBUG_KEY = "e72f17f4";
  private static final String MINT_PROD_KEY = "22464da6";

  private static boolean isDebugMode = false;
  private static boolean isEnabledTrackDefaultEvent = true;

  @Nullable
  private static ChannelIO channelIO;

  private Application application;
  private Thread.UncaughtExceptionHandler uncaughtExceptionHandler;
  @Nullable
  private ActivityLifecycleManager activityLifecycleManager;

  private ChannelIO(@NonNull Application application) {
    this.application = application;

    this.activityLifecycleManager = new ActivityLifecycleManager();
    application.registerActivityLifecycleCallbacks(activityLifecycleManager);

    setUncaughtExceptionHandler();
    ChannelIOManager.create(application);
    SocketManager.create(application);

    new GoogleAdIdTask().execute();
  }

  // Main Method

  public static void initialize(@Nullable Application application) {
    if (application == null) {
      L.e("Fail to 'initialize', Application can't be NULL");
    } else if (isInitializedChannelIO()) {
      L.e("Fail to 'initialize', Channel plugin already initialized");
    } else {
      channelIO = new ChannelIO(application);
    }
  }

  public static void boot(@Nullable ChannelPluginSettings pluginSettings) {
    boot(pluginSettings, null, null);
  }

  public static void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable OnBootListener completion) {
    boot(pluginSettings, null, completion);
  }

  public static void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable Profile profile) {
    boot(pluginSettings, profile, null);
  }

  public static void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable Profile profile, @Nullable OnBootListener completion) {
    if (channelIO != null && pluginSettings != null && pluginSettings.getPluginKey() != null) {
      channelIO.setBootSettings(pluginSettings);
      channelIO.bootProcess(pluginSettings, profile, completion);
    } else {
      L.e("Fail to 'boot', Check plugin information");
      if (completion != null) {
        completion.onCompletion(ChannelPluginCompletionStatus.NOT_INITIALIZED, null);
      }
    }
  }

  public static void shutdown() {
    if (channelIO != null) {
      channelIO.shutdownProcess();
    }
  }

  public static boolean open(@Nullable Context context) {
    return open(context, true);
  }

  public static boolean open(@Nullable Context context, boolean animate) {
    if (!isInitializedChannelIO()) {
      L.e("Fail to 'open', please initialize ChannelIO first");
      return false;
    }
    if (!isBooted()) {
      L.e("Fail to 'open', please boot first");
      return false;
    }

    if (context == null) {
      L.e("Fail to 'open', Context can't be NULL");
      return false;
    }

    ChannelPluginListener channelPluginListener = ChannelIOManager.getChannelPluginListener();
    if (channelPluginListener != null) {
      channelPluginListener.willShowMessenger();
    }

    if (ChannelIO.isEnabledTrackDefaultEvent()) {
      track(context, PluginSelector.getPluginKey(), CHDefaultEvent.Name.CHANNEL_OPEN, null);
    }
    IntentUtils.setNextActivity(context, UserChatListActivity.class)
        .putExtra(Const.EXTRA_REDIRECT_ANIMATED, animate)
        .setTransition(animate ? Transition.SLIDE_FROM_BOTTOM : Transition.NONE)
        .startActivity();
    return true;
  }

  public static void close() {
    close(true);
  }

  public static void close(boolean animated) {
    if (channelIO != null) {
      channelIO.finishAll(animated);
    }
  }

  public static void show() {
    Store.getInstance(UiStateStore.class).setLauncherVisible(true);
    RxBus.post(new CommandBus(Command.CHECK_CHANNEL_VIEW_VISIBILITY));
  }

  public static void hide() {
    Store.getInstance(UiStateStore.class).setLauncherVisible(false);
    RxBus.post(new CommandBus(Command.CHECK_CHANNEL_VIEW_VISIBILITY));
  }

  public static boolean openChat(@Nullable Context context) {
    return openChat(context, null);
  }

  public static boolean openChat(@Nullable Context context, @Nullable String chatId) {
    return openChat(context, chatId, true);
  }

  public static boolean openChat(@Nullable Context context, @Nullable String chatId, boolean animate) {
    if (!isInitializedChannelIO()) {
      L.e("Fail to 'openChat', please initialize ChannelIO first");
      return false;
    }
    if (!isBooted()) {
      L.e("Fail to 'openChat', please Boot first");
      return false;
    }

    if (context == null) {
      L.e("Fail to 'openChat', Context can't be NULL");
      return false;
    }

    if (ChannelIO.isEnabledTrackDefaultEvent()) {
      track(context, PluginSelector.getPluginKey(), CHDefaultEvent.Name.CHANNEL_OPEN, null);
    }

    IntentUtils.setNextActivity(context, UserChatListActivity.class)
        .putExtra(Const.EXTRA_CHAT_ID, chatId)
        .putExtra(Const.EXTRA_REDIRECT_ANIMATED, animate)
        .setTransition(animate ? Transition.SLIDE_FROM_BOTTOM : Transition.NONE)
        .startActivityForResult(Const.REQUEST_CHAT);
    return true;
  }

  public static void initPushToken(String token) {
    Context context = getAppContext();
    if (context != null) {
      PrefSupervisor.setDeviceToken(context, token);
    }
  }

  public static void handlePushNotification() {
    Context context = getAppContext();
    if (context != null) {
      ChannelPushClient.handlePushNotification(context);
    }
  }

  public static void showPushNotification(Map<String, String> message) {
    Context context = getAppContext();
    if (context != null) {
      ChannelPushManager.showPushNotification(context, message);
    }
  }

  public static boolean isChannelPushNotification(Map<String, String> message) {
    return ChannelPushManager.isChannelPushNotification(message);
  }

  private void setUncaughtExceptionHandler() {
    uncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
    Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
      @Override
      public void uncaughtException(Thread thread, Throwable ex) {
        if (Looper.getMainLooper().getThread() == thread) {
          uncaughtExceptionHandler.uncaughtException(thread, ex);
        } else if (ex instanceof RejectedExecutionException) {
          SocketManager.reconnect();
        }
      }
    });
  }

  private void setBootSettings(ChannelPluginSettings pluginSettings) {
    isDebugMode = pluginSettings.isDebugMode();
    isEnabledTrackDefaultEvent = pluginSettings.isEnabledTrackDefaultEvent();

    if (application != null) {
      PrefSupervisor.setPluginLanguage(application, pluginSettings.getLocale());
      PrefSupervisor.setPluginSetting(application, pluginSettings);
    }
  }

  private void bootProcess(
      final ChannelPluginSettings pluginSettings,
      final @Nullable Profile profile,
      final @Nullable OnBootListener listener
  ) {
    if (isBooted()) {
      shutdownProcess();
    }

    ChannelApiManager.callOnce(
        ChannelApiManager.get()
            .getLastestPackage(Const.PACKAGE_NAME, BuildConfig.VERSION_NAME)
            .filter(new Func1<PackageWrapper, Boolean>() {
              @Override
              public Boolean call(PackageWrapper packageWrapper) {
                return BootManager.isValidVersion(packageWrapper, listener);
              }
            })
            .flatMap(new Func1<PackageWrapper, Observable<PluginWrapper>>() {
              @Override
              public Observable<PluginWrapper> call(PackageWrapper packageWrapper) {
                if (!CompareUtils.isSame(packageWrapper.getVersionString(), BuildConfig.VERSION_NAME)) {
                  L.i("Newest version is: " + packageWrapper.getVersionString());
                }

                return ChannelApiManager.get().bootV2(
                    BootManager.createBootHeader(),
                    BootManager.createBootBody(pluginSettings, profile),
                    pluginSettings.getPluginKey()
                );
              }
            })
            .filter(new Func1<PluginWrapper, Boolean>() {
              @Override
              public Boolean call(PluginWrapper pluginWrapper) {
                return BootManager.isValidBootResult(pluginWrapper, listener);
              }
            }),

        new RestSubscriber<PluginWrapper>() {
          @Override
          public void onError(RetrofitException e) {
            BootManager.sendNetworkError(listener, e);
          }

          @Override
          public void onNext(PluginWrapper repo) {
            repo.set();

            PrefSupervisor.setBootData(application, repo.getGuest());

            Store.getInstance(ChannelIOSettingsStore.class).setPluginSettings(pluginSettings);
            Store.getInstance(UiStateStore.class).setClosedChatVisible(PrefSupervisor.isShownClosedChat(application));

            ChannelPushManager.sendTokenToChannelIO(application);

            SocketManager.setChannelId(ChannelSelector.getCurrentChannelId());
            SocketManager.connect();
            ChannelIOManager.subscribeSocket();

            RxBus.post(new CommandBus(Command.BOOT));

            if (pluginSettings.isEnabledTrackDefaultEvent()) {
              track(application, pluginSettings.getPluginKey(), CHDefaultEvent.Name.BOOT, null);
            }

            if (listener != null) {
              listener.onCompletion(ChannelPluginCompletionStatus.SUCCESS, GuestSelector.get());
            }
          }
        },

        ApiTag.BOOT_V2
    );
  }

  private void shutdownProcess() {

    PrefSupervisor.clearBootData(application);

    String jwt = TokenSelector.getJwt();
    ChannelPushManager.deleteToken(application, jwt);
    PrefSupervisor.setJwtToken(application, null);
    
    Store.destroy();
    RxBus.post(new CommandBus(Command.SHUTDOWN));

    SocketManager.setChannelId(null);
    SocketManager.disconnect();

    ChatManager.release();

    releaseBugTracking();
  }

  private void finishAll(boolean animated) {
    if (activityLifecycleManager != null) {
      activityLifecycleManager.finish(animated);
    }
  }

  @Nullable
  static Activity getTopActivity() {
    if (channelIO != null && channelIO.activityLifecycleManager != null) {
      return channelIO.activityLifecycleManager.getTopActivity();
    }
    return null;
  }

  private void initBugTracking(Channel channel) {
    if (application != null) {
      Mint.disableNetworkMonitoring();

      Mint.initAndStartSession(application, MINT_PROD_KEY);
      Mint.addExtraData("plugin_version", BuildConfig.VERSION_NAME);

      if (channel != null) {
        Mint.addExtraData("channel_id", channel.getId());
        Mint.addExtraData("channel_name", channel.getName());
      }
    }
  }

  private void releaseBugTracking() {
    if (application != null) {
      Mint.clearExtraData();
      Mint.closeSession(application);
      Mint.flush();
    }
  }

  public static void track(
      @Nullable Context context,
      @Nullable String pluginKey,
      @NonNull @Size(min = 1L, max = 30L) String eventName,
      @Nullable Map<String, Object> eventProperty) {
    if (TextUtils.isEmpty(eventName) || eventName.length() > 30) {
      return;
    }
    track(new Event(context, pluginKey, eventName, eventProperty));
  }

  private static void track(@NonNull Event event) {
    ChannelApiManager.get()
        .trackEvent(event)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<Void>());
  }

  public static boolean isInitializedChannelIO() {
    if (channelIO == null) {
      return false;
    }
    return true;
  }

  @Nullable
  public static Context getAppContext() {
    if (channelIO != null) {
      return channelIO.application.getApplicationContext();
    }
    return null;
  }

  public static boolean isDebugMode() {
    return isDebugMode;
  }

  public static boolean isEnabledTrackDefaultEvent() {
    return isEnabledTrackDefaultEvent;
  }

  @Deprecated
  public static boolean isDataStored() {
    return isBooted();
  }

  public static boolean isBooted() {
    return ChannelIOSelector.isBooted();
  }

  public static void updateGuest() {
    updateGuest(null);
  }

  public static void updateGuest(@Nullable OnGuestUpdatedListener listener) {
    ChannelIOManager.fetchMe(listener);
  }

  public static void setChannelPluginListener(ChannelPluginListener channelPluginListener) {
    ChannelIOManager.setChannelPluginListener(channelPluginListener);
  }

  public static void clearChannelPluginListener() {
    ChannelIOManager.clearChannelPluginListener();
  }

  // Google ad id

  private class GoogleAdIdTask extends AsyncTask<Void, Void, String> {
    @Nullable
    protected String doInBackground(final Void... params) {
      try {
        if (application != null) {
          AdvertisingIdClient.Info advertisingIdInfo = AdvertisingIdClient.getAdvertisingIdInfo(application);
          if (advertisingIdInfo != null && !advertisingIdInfo.isLimitAdTrackingEnabled()) {
            return advertisingIdInfo.getId();
          }
        }
      } catch (Exception ignored) {
      }
      return null;
    }

    protected void onPostExecute(String adId) {
      if (adId != null) {
        Store.getInstance(DataStore.class).setAdId(adId);
      }
    }
  }
}
