package com.zoyi.channel.plugin.android.activity.chat;

import android.content.Context;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.activity.chat.listener.*;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.enumerate.ChatState;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.event.ChannelModelBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.ActionInput;
import com.zoyi.channel.plugin.android.model.entity.Entity;
import com.zoyi.channel.plugin.android.model.entity.Person;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.*;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.selector.GuestSelector;
import com.zoyi.channel.plugin.android.selector.PluginSelector;
import com.zoyi.channel.plugin.android.selector.ProfileSelector;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.view.dialog.CountryCodeDialog;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.Observable;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Func2;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by mika on 2017. 2. 27..
 */

public class ChatPresenter implements
    ChatContract.Presenter,
    OnSendingMessageClickListener,
    OnMessageClickListener,
    OnMessageSendListener,
    OnMessageReviewListener,
    OnProfileUpdateListener,
    OnFormMessageClickListener {

  private ChatContract.View view;
  private ChatAdapterContract.View adapterView;
  private ChatAdapterContract.Model adapterModel;

  private Context context;

  @Nullable
  private UserChat userChat;
  @Nullable
  private Session session;

  @Nullable
  private String chatId;

  private ChatState state = ChatState.IDLE;

  private String backwardId = Const.MESSAGE_ID_MAX;
  private String forwardId = Const.MESSAGE_ID_MIN;
  private String forwardTempId = Const.MESSAGE_ID_MIN;
  private Long forwardTimestamp = Long.MIN_VALUE;

  private int fetchCounter = 0;
  private boolean backwardLoading = false;
  private boolean isStateCompleted = false;
  private boolean isNewUserChat = false;

  @Nullable
  private Long lastReadAt;
  private long welcomedAt;

  private List<Country> countries;

  private List<SendingMessageItem> tempQueue;

  private final Object messageObject = new Object();

  public ChatPresenter(Context context) {
    this.context = context;

    TimeUtils.refreshOffset();

    tempQueue = new ArrayList<>();
    countries = new ArrayList<>();
  }

  @Initializer
  @Override
  public void setView(ChatContract.View view) {
    this.view = view;
  }

  @Nullable
  @Override
  public UserChat getUserChat() {
    return userChat;
  }

  @Nullable
  @Override
  public String getChatId() {
    return chatId;
  }

  @Override
  public boolean isStateCompleted() {
    return isStateCompleted;
  }

  @Override
  public void setChatId(@Nullable String chatId) {
    this.chatId = chatId;
  }

  @Initializer
  @Override
  public void setAdapterView(ChatAdapterContract.View adapterView) {
    this.adapterView = adapterView;
    this.adapterView.setOnMessageClickListener(this);
    this.adapterView.setOnSendingMessageClickListener(this);
    this.adapterView.setOnMessageReviewListener(this);
    this.adapterView.setOnProfileUpdateListener(this);
    this.adapterView.setOnFormMessageClickListener(this);
  }

  @Initializer
  @Override
  public void setAdapterModel(ChatAdapterContract.Model adapterModel) {
    this.adapterModel = adapterModel;
  }

  @Override
  public void requestPreloadData(@Nullable final String chatId) {
    Plugin plugin = PluginSelector.getPlugin();
    if (plugin == null) {
      return;
    }
    view.showProgress(ResUtils.getString(context, "ch.loading_information"));

    Observable.zip(
        ChannelIO.getApi().getFollowingManagers(),
        ChannelIO.getApi().getPlugin(plugin.getId()),
        new Func2<ManagersWrapper, PluginWrapper, ChatPreloadWrapper>() {
          @Override
          public ChatPreloadWrapper call(ManagersWrapper managersWrapper, PluginWrapper pluginWrapper) {
            return new ChatPreloadWrapper(managersWrapper, pluginWrapper);
          }
        })
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<ChatPreloadWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            L.e(error.getMessage());
            view.hideProgress();
          }

          @Override
          public void onNext(ChatPreloadWrapper repo) {
            view.hideProgress();

            if (repo != null) {
              repo.update();

              if (!TextUtils.isEmpty(chatId)) {
                fetchUserChat(chatId);
              } else {
                addWelcomeMessage();

                view.onLoadUserChat(null, false);
                view.updateBackwardId(null);
              }
            }
          }
        });
  }

  private void addWelcomeMessage() {
    String message;
    String channelLanguage = PrefSupervisor.getPluginLanguage(context);
    Plugin plugin = PluginSelector.getPlugin();

    if (plugin != null) {
      if (GuestSelector.isNamed()) {
        message = plugin
            .getWelcomeNamedI18n()
            .get(channelLanguage)
            .replace("${name}", GuestSelector.getName());
      } else {
        message = plugin.getWelcomeI18n().get(channelLanguage);
      }
    } else {
      message = ResUtils.getString(context, Const.UNKNOWN_KEY);
    }

    if (!TextUtils.isEmpty(message)) {
      welcomedAt = System.currentTimeMillis();
      adapterModel.setWelcomeMessage(message, welcomedAt);
    }
  }

  private void fetchUserChat(String chatId) {
    this.chatId = chatId;

    adapterModel.addMessageItems(getChatManager().getFailedItems(chatId));

    if (userChat == null) {
      view.showProgress(ResUtils.getString(context, "ch.loading_information"));
    }

    setState(ChatState.USER_CHAT_LOADING, false);

    ChannelIO.getApi().getUserChat(chatId)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<UserChatWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            view.hideProgress();
            handleUserChatFetchError(error);
          }

          @Override
          public void onNext(UserChatWrapper repo) {
            if (repo != null) {
              repo.update();
            }

            setUserChat(repo, false);
          }
        });
  }

  private void createUserChat() {
    view.showProgress(ResUtils.getString(context, "ch.loading_information"));

    setState(ChatState.USER_CHAT_LOADING, false);

    Plugin plugin = PluginSelector.getPlugin();

    if (plugin != null && plugin.getKey() != null) {
      ChannelIO.getApi().createUserChat(plugin.getKey(), plugin.getId(), welcomedAt)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              view.hideProgress();
              handleUserChatFetchError(error);
            }

            @Override
            public void onNext(UserChatWrapper repo) {
              if (repo != null) {
                repo.update();
              }

              RxBus.post(new ChannelModelBus(repo.getUserChat(), true));
              RxBus.post(new ChannelModelBus(repo.getSession(), true));

              setUserChat(repo, true);
            }
          });
    } else {
      view.hideProgress();
    }
  }

  private void setUserChat(UserChatWrapper wrapper, boolean create) {
    view.hideProgress();

    if (isStateEquals(ChatState.USER_CHAT_LOADING)) {
      if (wrapper == null) {
        handleUserChatFetchError(new Exception("UserChatWrapper cannot be null"));
      } else {
        userChat = wrapper.getUserChat();
        chatId = userChat.getId();
        isStateCompleted = userChat.isStateCompleted();
        session = wrapper.getSession();

        adapterModel.setUserChat(userChat);

        view.onLoadUserChat(userChat, create);
        view.setInputLayoutVisibility();
        view.setReconnectVisibility(false);

        if (lastReadAt == null) {
          lastReadAt = create ? Long.MAX_VALUE : session.getLastReadAt();
        }

        if (tempQueue.size() > 0) {
          getChatManager().send(chatId, tempQueue, this);
          tempQueue.clear();
        }

        joinChat();
      }
    }
  }

  private void handleUserChatFetchError(Throwable throwable) {
    if (throwable != null) {
      L.e(throwable.getMessage());
    }
    if (isStateEquals(ChatState.USER_CHAT_LOADING)) {
      setState(ChatState.USER_CHAT_NOT_LOADED, true);
    }
  }

  private void joinChat() {
    if (SocketManager.isReady()) {
      setState(ChatState.CHAT_JOINING, false);
      SocketManager.joinChat(chatId);
    } else {
      setState(ChatState.WAITING_SOCKET, false);
      SocketManager.reconnect();
    }
  }

  private boolean isAvailableRequest(int count) {
    return isStateEquals(ChatState.MESSAGES_LOADING) && fetchCounter == count;
  }

  private void fetchMessages() {
    if (SocketManager.isReady()) {
      fetchCounter++;
      setState(ChatState.MESSAGES_LOADING, false);
      if (!isMessageExists()) {
        fetchInitMessages(fetchCounter);
      } else {
        fetchForwardMessages(forwardId, fetchCounter);
      }
    } else {
      setState(ChatState.WAITING_SOCKET, false);
      SocketManager.reconnect();
    }
  }

  @Override
  public boolean isMessageExists() {
    return CompareUtils.compare(backwardId, forwardId) < 0;
  }

  private void fetchInitMessages(final int counter) {
    if (!TextUtils.isEmpty(chatId)) {
      ChannelIO.getApi()
          .getMessages(
              chatId,
              Const.MESSAGE_ID_MAX,
              Const.MESSAGE_FETCH_LIMIT,
              Const.DESC)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<MessagesWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              if (isAvailableRequest(counter)) {
                handleMessagesFetchError(error, counter);
              }
            }

            @Override
            public void onNext(MessagesWrapper repo) {
              if (isAvailableRequest(counter)) {
                if (repo != null) {
                  repo.update();
                }

                if (repo == null) {
                  handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), counter);
                } else {
                  backwardId = repo.getNext();
                  view.updateBackwardId(backwardId);

                  addMessages(repo.getMessages());
                  calculateForwardId(repo.getMessages());
                  calculateForwardId();

                  if (lastReadAt >= forwardTimestamp) {
                    lastReadAt = Long.MAX_VALUE;
                  }

                  addNewMessageItem(repo.getMessages());

                  setState(ChatState.CHAT_READY, false);
                  readAll();
                  if (isNewUserChat) {
                    requestProfileBot();
                    isNewUserChat = false;
                  }

                  view.scrollToBottom(true);
                }
              }
            }
          });
    }
  }

  @Override
  public void fetchBackwardMessages() {
    if (backwardLoading || !isMessageExists() || chatId == null) {
      return;
    }
    backwardLoading = true;

    ChannelIO.getApi()
        .getMessages(
            chatId,
            backwardId,
            Const.MESSAGE_FETCH_LIMIT,
            Const.DESC)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessagesWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            backwardLoading = false;

            handleMessagesFetchError(error, fetchCounter);
          }

          @Override
          public void onNext(MessagesWrapper repo) {
            if (repo != null) {
              repo.update();
            }

            if (repo == null) {
              handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), fetchCounter);
            } else {
              backwardId = repo.getNext();
              view.updateBackwardId(backwardId);

              adapterModel.addMessages(repo.getMessages(), chatId);

              addNewMessageItem(repo.getMessages());
            }
            backwardLoading = false;
          }
        });
  }

  @Override
  public void fetchForwardMessages(String since, final int counter) {
    if (!TextUtils.isEmpty(chatId)) {
      ChannelIO.getApi()
          .getMessages(
              chatId,
              since,
              Const.MESSAGE_FETCH_LIMIT,
              Const.ASC)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<MessagesWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              if (isAvailableRequest(counter)) {
                handleMessagesFetchError(error, counter);
              }
            }

            @Override
            public void onNext(MessagesWrapper repo) {
              if (isAvailableRequest(counter)) {
                if (repo == null) {
                  handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), counter);
                } else {
                  repo.update();

                  addMessages(repo.getMessages());
                  calculateForwardId(repo.getMessages());

                  if (repo.getNext() == null) {
                    calculateForwardId();
                    setState(ChatState.CHAT_READY, false);
                    readAll();
                  } else {
                    fetchForwardMessages(forwardId, counter);
                  }
                }
              }
            }
          });
    }
  }

  private void handleMessagesFetchError(Throwable throwable, int counter) {
    if (throwable != null) {
      L.e(throwable.getMessage());
    }
    if (isStateEquals(ChatState.MESSAGES_LOADING) && fetchCounter == counter) {
      setState(ChatState.MESSAGES_NOT_LOADED, true);
    }
  }

  private void addNewMessageItem(List<Message> messages) {
    if (lastReadAt == null || isStateCompleted) {
      return;
    }

    long min = Long.MAX_VALUE;
    long max = Long.MIN_VALUE;

    if (messages != null) {
      for (Message message : messages) {
        if (message.getLog() != null) {
          continue;
        }
        min = Math.min(min, message.getCreatedAt());
        max = Math.max(max, message.getCreatedAt());
      }
    }

    if (min <= lastReadAt && lastReadAt < max) {
      adapterModel.addMessageItem(new NewMessageItem(lastReadAt));
    }
  }

  private void addMessage(Message message) {
    if (!TextUtils.isEmpty(chatId) && CompareUtils.isSame(message.getChatId(), chatId)) {
      String oldMessageId = adapterModel.getLastMessageItemId();
      adapterModel.addMessage(message, chatId);
      String newMessageId = adapterModel.getLastMessageItemId();

      if (isStateEquals(ChatState.CHAT_READY)) {
        calculateForwardId(message);
        readAll();
      } else {
        forwardTempId = CompareUtils.max(forwardTempId, message.getId());
      }

      forwardTimestamp = Math.max(forwardTimestamp, message.getCreatedAt());

      view.scrollToBottom(view.isMovableToBottomPosition());

      if (!CompareUtils.isSame(oldMessageId, newMessageId)) {
        view.processNewMessage(
            ProfileSelector.getHostProfile(message.getPersonType(), message.getPersonId()),
            message);
      }
    }
  }

  private void addMessages(List<Message> messages) {
    if (messages != null) {
      for (Message message : messages) {
        forwardTimestamp = Math.max(forwardTimestamp, message.getCreatedAt());
        forwardId = CompareUtils.max(forwardId, message.getId());
      }
    }
    adapterModel.addMessages(messages, chatId);
  }

  private void calculateForwardId(Message message) {
    if (message != null) {
      forwardId = CompareUtils.max(forwardId, message.getId());
    }
  }

  private void calculateForwardId(List<Message> messages) {
    if (messages != null) {
      for (Message message : messages) {
        calculateForwardId(message);
      }
    }
  }

  private void calculateForwardId() {
    forwardId = CompareUtils.max(forwardId, forwardTempId);
  }

  @Override
  public void sendTextMessage(String message) {
    if (!TextUtils.isEmpty(message)) {
      SendingMessageItem item = new SendingMessageItem(chatId, message, false);
      adapterModel.addMessageItem(item);

      sendMessages(Collections.singletonList(item));
    }
  }

  @Override
  public void sendImageFiles(ArrayList<String> imagePaths) {
    List<SendingMessageItem> items = new ArrayList<>();
    if (imagePaths != null && !TextUtils.isEmpty(chatId)) {
      for (String p : imagePaths) {
        SendingMessageItem item = new SendingMessageItem(chatId, p, true);
        adapterModel.addMessageItem(item);
        items.add(item);
      }
    }
    sendMessages(items);
  }

  private void sendSelectedFormMessage(SendingMessageItem item) {
    adapterModel.addMessageItem(item);
    sendMessages(Collections.singletonList(item));
  }

  private void sendMessages(List<SendingMessageItem> items) {
    view.scrollToBottom(true);

    if (chatId == null) {
      tempQueue.addAll(items);
      isNewUserChat = true;
      createUserChat();
    } else {
      getChatManager().send(items, this);
    }
  }

  @Override
  public void receiveData(Entity entity, boolean upsert) {
    if (entity != null) {
      if (!TextUtils.isEmpty(chatId)) {
        synchronized (messageObject) {
          switch (entity.getClass().getSimpleName()) {
            case Message.CLASSNAME:
              Message message = (Message) entity;

              if (message.getBotOption() != null &&
                  message.getBotOption().isWelcome() &&
                  adapterModel.getWelcomeMessage() != null) {
                return;
              }

              handleMessageLog(message);
              addMessage(message);
              break;

            case Bot.CLASSNAME:
            case Manager.CLASSNAME:
              if (userChat != null && !(userChat.isStateReady() || userChat.isStateOpen())) {
                view.setPerson((Person) entity);
              }
              break;

            case UserChat.CLASSNAME:
              handleUserChatData((UserChat) entity, chatId);
              view.changeBigBar();
              break;
          }
        }
      }

      if (Channel.CLASSNAME.equals(entity.getClass().getSimpleName())) {
        // Exception for Channel model
        view.setChannel((Channel) entity);
      }
    }
  }

  private void handleUserChatData(UserChat userChat, String chatId) {
    if (userChat == null || !chatId.equals(userChat.getId())) {
      return;
    }

    this.userChat = userChat;

    if (userChat.isStateRemoved()) {
      leaveChat();
      view.onRemovedChat();
    } else if (userChat.isStateCompleted()) {
      adapterModel.addMessageItem(userChat);
      view.scrollToBottom(false);
    } else if (userChat.isStateFollowing()) {
      view.setPerson(ProfileSelector.getPerson(userChat.getHostType(), userChat.getHostId()));
    }
  }

  private void handleMessageLog(Message message) {
    if (CompareUtils.isSame(message.getChatId(), chatId) && userChat != null) {
      if (message.getLog() == null) {
        return;
      }
      if (message.getLog().isResolve()) {
        isStateCompleted = true;
        userChat.setState(Const.USER_CHAT_STATE_RESOLVED);
      } else if (message.getLog().isClose()) {
        isStateCompleted = true;
        userChat.setState(Const.USER_CHAT_STATE_CLOSED);
      } else {
        isStateCompleted = false;
      }
      adapterModel.setUserChat(userChat);

      view.setInputLayoutVisibility();
    }
  }

  private void requestProfileBot() {
    Plugin plugin = PluginSelector.getPlugin();

    if (plugin != null && !TextUtils.isEmpty(chatId)) {
      ChannelIO.getApi().requestProfileBot(chatId, plugin.getId())
          .onBackpressureBuffer()
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe();
    }
  }

  private void readAll() {
    if (!TextUtils.isEmpty(chatId)) {
      ChannelIO.getApi().readAll(chatId)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<Void>());
    }
  }

  @Override
  public void receiveCommand(Command command, @Nullable Object object) {
    switch (command) {
      case APP_STARTED:
        refresh();
        break;

      case SOCKET_DISCONNECTED:
        fetchCounter++;
        if (!isStateEquals(ChatState.IDLE)) {
          setState(ChatState.USER_CHAT_NOT_LOADED, true);
        }
        break;

      case SOCKET_ERROR:
        view.setReconnectVisibility(true);
        break;

      case READY:
        view.setReconnectVisibility(false);

        if (isStateEquals(ChatState.WAITING_SOCKET)) {
          joinChat();
        } else {
          refresh();
        }
        break;

      case LEAVED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && isStateEquals(ChatState.CHAT_READY)) {
          setState(ChatState.NOT_JOINED, false);  // true?
        }
        break;

      case JOINED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && isStateEquals(ChatState.CHAT_JOINING)) {

          view.setReconnectVisibility(false);

          fetchMessages();
        }
        break;
    }
  }

  @Override
  public void receiveTyping(Typing typing) {
    synchronized (messageObject) {
//      boolean isBottom = view.isBottomPosition();

      TypingItem item = new TypingItem();
      int position = adapterModel.getIndex(item);

      if (position >= 0) {
        item = (TypingItem) adapterModel.getItem(position);
      }
      if (item == null) {
        return;
      }

      switch (typing.getAction()) {
        case "start":
          item.addOrUpdateTyping(typing);
          break;

        case "stop":
          item.removeTyping(typing);
          break;
      }

      adapterModel.addOrUpdateMessageItem(position, item);
//      if (item.hasTyping()) {
//        adapterModel.addOrUpdateMessageItem(position, item);
//      } else {
//        adapterModel.removeMessageItem(item);
//      }
//
//      if (isBottom) {
//        view.scrollToBottom(true);
//      }
    }
  }

  @Override
  public void receiveLanguageEvent() {
    if (chatId == null) {
      addWelcomeMessage();
    }
  }

  @Override
  public void messageClicked(Message message) {
    if (message.getFile() != null) {
      if (message.getFile().getPreviewThumb() == null || !message.getFile().isImage()) {
        view.onFileDownload(message);
      } else {
        adapterModel.setImageFilesToStorage();
        view.onShowPhotoAlbum(message.getFile());
      }
    } else if (message.getWebPage() != null) {
      // web preview click event
      urlClicked(message.getWebPage().getUrl());
    }
  }

  @Override
  public void urlClicked(String url) {
    view.onUrlClicked(url);
  }

  @Override
  public void requestResendClicked(SendingMessageItem item) {
    view.sendingMessageClicked(item);
  }

  @Override
  public void resend(SendingMessageItem item) {
    if (item != null) {
      removeFailedItem(item);

      item.refreshCreatedAt();

      adapterModel.addMessageItem(item);
      getChatManager().send(item, this);
    }
  }

  @Override
  public void removeFailedItem(SendingMessageItem item) {
    if (item != null) {
      adapterModel.removeMessageItem(item);
      getChatManager().removeFailedItem(item.getCreatedAt());
    }
  }

  @Override
  public void sendSuccessed(SendingMessageItem item, Message message) {
    adapterModel.removeMessageItem(item);
    receiveData(message, true);
  }

  @Override
  public void sendFailed(SendingMessageItem item) {
    adapterModel.addMessageItem(item);
  }

  @Override
  public void refresh() {
    switch (state) {
      case USER_CHAT_NOT_LOADED:
        if (!TextUtils.isEmpty(chatId)) {
          fetchUserChat(chatId);
        } else {
          createUserChat();
        }
        break;

      case WAITING_SOCKET:
        SocketManager.connect();
        break;

      case NOT_JOINED:
        setState(ChatState.CHAT_JOINING, true);
        SocketManager.joinChat(chatId);
        break;

      case MESSAGES_NOT_LOADED:
        fetchMessages();
        break;

      case IDLE:
        requestPreloadData(chatId);
        break;
    }
  }

  private void setState(ChatState state, boolean showRefresh) {
    this.state = state;
    view.setRefreshVisibility(showRefresh);
  }

  private boolean isStateEquals(ChatState targetState) {
    return targetState.equals(state);
  }

  @Override
  public void onMessageReview(@Nullable String review) {
    if (!TextUtils.isEmpty(chatId)) {
      ChannelIO.getApi().reviewUserChat(chatId, review)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              L.e(error.getMessage());
            }

            @Override
            public void onNext(UserChatWrapper repo) {
              if (repo != null) {
                repo.update();
              }

              if (repo != null) {
                UserChat userChat = repo.getUserChat();

                if (userChat != null) {
                  adapterModel.addMessageItem(userChat);
                }
              }
            }
          });
    }
  }

  @Override
  public void leaveChat() {
    SocketManager.leaveChat(chatId);
  }

  private ChatManager getChatManager() {
    return ChatManager.get();
  }

  @Override
  public void getCountries(final ProfileBotMessageItem item, final String key, final MobileNumber mobileNumber) {
    if (countries == null || countries.isEmpty()) {
      view.showProgress(ResUtils.getString(context, "ch.loading_information"));

      ChannelIO.getApi().getCountries()
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<CountryWrapper>() {
            @Override
            public void onError(RetrofitException e) {
              view.hideProgress();
            }

            @Override
            public void onNext(CountryWrapper repo) {
              view.hideProgress();

              if (repo != null) {
                repo.set();
              }

              if (repo != null) {
                countries = repo.getCountries();
                showCountryDialog(item, key, mobileNumber);
              }
            }
          });
    } else {
      showCountryDialog(item, key, mobileNumber);
    }
  }

  private void showCountryDialog(final ProfileBotMessageItem item, final String key, final MobileNumber mobileNumber) {
    String countryName;

    switch (PrefSupervisor.getPluginLanguage(context)) {
      case "ko":
        countryName = "South Korea";
        break;
      case "ja":
        countryName = "Japan";
        break;
      default:
        countryName = "United States";
        break;
    }
    new CountryCodeDialog(
        context,
        countryName,
        countries,
        new CountryCodeDialog.OnCountryCodeSelectListener() {
          @Override
          public void onCountryCodeSelected(int position, String countryCode) {
            mobileNumber.setCountryCode(Integer.valueOf(countryCode));
            item.setInputState(key, mobileNumber, false, null);
            adapterModel.addMessageItem(item);
          }
        }).show();
  }

  @Override
  public void onProfileUpdate(final ProfileBotMessageItem item, final String key, final Object value) {
    item.setInputState(key, item.getInputState(), true, null);
    adapterModel.addMessageItem(item);

    RequestBody requestBody = RequestUtils.form().set(key, value).create();
    ChannelIO
        .getApi()
        .updateProfileBot(item.getMessage().getId(), requestBody)
        .onBackpressureBuffer()
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessageWrapper>() {

          @Override
          public void onError(RetrofitException error) {
            item.setInputState(key, item.getInputState(), false, ResUtils.getString(context, "ch.profile_form.error"));
            adapterModel.addMessageItem(item);
            view.scrollToBottom(view.isMovableToBottomPosition());
          }

          @Override
          public void onNext(MessageWrapper repo) {
            if (repo != null) {
              repo.update();
            }

            item.setMessage(repo.getMessage());
            item.resetInputState();
            adapterModel.addMessageItem(item);
            view.scrollToBottom(view.isMovableToBottomPosition());
          }
        });
  }

  @Override
  public void onCompleteWriteProfile() {
    if (userChat != null && !userChat.isStateCompleted() && !userChat.isStateRemoved()) {
      view.requestFocusEditText();
    }
  }

  @Override
  public void onChangeProfileFocus(boolean hasFocus) {
    view.onChangeProfileMessageFocus(hasFocus);
  }

  @Override
  public void onFormMessageClick(ActionInput actionInput) {
    Map<String, String> submit = new HashMap<>();
    submit.put("id", actionInput.getMessage().getId());
    submit.put("key", actionInput.getKey());

    SendingMessageItem item = new SendingMessageItem(
        actionInput.getMessage().getChatId(),
        actionInput.getValue(PrefSupervisor.getPluginLanguage(context)),
        false);
    item.setSubmit(submit);

    sendSelectedFormMessage(item);
  }
}
