package com.zoyi.channel.plugin.android;

import android.app.Application;
import android.content.Context;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.Size;
import android.text.TextUtils;

import com.splunk.mint.Mint;
import com.zoyi.channel.plugin.android.activity.chat.ChatActivity;
import com.zoyi.channel.plugin.android.activity.chat.ChatManager;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnChatListener;
import com.zoyi.channel.plugin.android.activity.userchat_list.UserChatListActivity;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.LauncherBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.CheckInPrefSupervisor;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Event;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.wrapper.PackageWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.ChannelApi;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.network.ServiceFactory;
import com.zoyi.channel.plugin.android.push.ChannelPushClient;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.channel.plugin.android.wrapper.CHDefaultEvent;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicInteger;

public class ChannelIO {
  private static final String MINT_DEBUG_KEY = "e72f17f4";
  private static final String MINT_PROD_KEY = "22464da6";

  private static boolean isDebugMode = false;
  private static boolean isEnabledTrackDefaultEvent = true;
  private static boolean isHideDefaultInAppPush = true;

  private static ChannelIO channelIO;
  private static ChannelApi channelApi;
  private static OnChatListener onChatListener = null;

  private Application application;
  private Thread.UncaughtExceptionHandler uncaughtExceptionHandler;

  private AtomicInteger bootCounter = new AtomicInteger(0);

  private ActivityLifecycleManager activityLifecycleManager;

  private ChannelIO(Application application) {
    if (isInitializedChannelIO()) {
      L.e("Channel plugin already initialized");
      return;
    }
    if (application == null) {
      L.e("Application cannot be null");
      return;
    }

    this.application = application;

    activityLifecycleManager = new ActivityLifecycleManager();
    application.registerActivityLifecycleCallbacks(activityLifecycleManager);

    setUncaughtExceptionHandler();
    ChannelStore.create(application);
    SocketManager.create(application);
  }

  // Main Method

  public static void initialize(@NonNull Application application) {
    channelIO = new ChannelIO(application);
  }

  public static void boot(ChannelPluginSettings pluginSettings) {
    boot(pluginSettings, null, null);
  }

  public static void boot(ChannelPluginSettings pluginSettings, OnBootListener completion) {
    boot(pluginSettings, null, completion);
  }

  public static void boot(ChannelPluginSettings pluginSettings, Profile profile) {
    boot(pluginSettings, profile, null);
  }

  public static void boot(ChannelPluginSettings pluginSettings, Profile profile, OnBootListener completion) {
    if (isInitializedChannelIO() && pluginSettings != null && pluginSettings.getPluginKey() != null) {
      channelIO.setChannelInfo(pluginSettings);
      channelIO.bootProcess(pluginSettings, profile, completion);
    } else {
      L.e("Check plugin information");
    }
  }

  public static void shutdown() {
    if (isInitializedChannelIO()) {
      channelIO.shutdownProcess();
    }
  }

  public static boolean open(@NonNull Context context) {
    return open(context, true);
  }

  public static boolean open(@NonNull Context context, boolean animated) {
    if (!isInitializedChannelIO() || !isDataStored()) {
      L.e("Please check in first");
      return false;
    }

    for (ChannelPluginListener l : ChannelStore.getChannelPluginListeners()) {
      if (l != null) {
        l.willShowMessenger();
      }
    }

    if (ChannelIO.isEnabledTrackDefaultEvent() && ChannelStore.getPluginKey() != null) {
      track(context, ChannelStore.getPluginKey(), CHDefaultEvent.Name.CHANNEL_OPEN, null);
    }
    IntentUtils.setNextActivity(context, UserChatListActivity.class)
        .putExtra(Const.EXTRA_ANIMATED, animated)
        .startActivity();
    return true;
  }

  public static void close() {
    close(true);
  }

  public static void close(boolean animated) {
    if (isInitializedChannelIO()) {
      channelIO.finishAll(animated);
    }
  }

  public static void show() {
    RxBus.post(new LauncherBus(true));
  }

  public static void hide() {
    RxBus.post(new LauncherBus(false));
  }

  public static boolean openChat(@NonNull Context context) {
    return openChat(context, null);
  }

  public static boolean openChat(@NonNull Context context, @Nullable String chatId) {
    if (!isInitializedChannelIO() || !isDataStored()) {
      L.e("Please Boot first");
      return false;
    }

    if (ChannelIO.isEnabledTrackDefaultEvent() && ChannelStore.getPluginKey() != null) {
      track(context, ChannelStore.getPluginKey(), CHDefaultEvent.Name.CHANNEL_OPEN, null);
    }

    IntentUtils.setNextActivity(context, ChatActivity.class)
        .putExtra(Const.EXTRA_CHAT_COUNT, 0)
        .putExtra(Const.EXTRA_CHAT_FIRST, chatId == null)
        .putExtra(Const.EXTRA_CHAT_ID, chatId)
        .startActivityForResult(Const.REQUEST_CHAT);
    return true;
  }

  public static void handlePushNotification(Context context) {
    ChannelPushClient.handlePushNotification(context);
  }

  public static void showPushNotification(Context context, Map<String, String> message) {
    ChannelPushManager.showPushNotification(context, message);
  }

  public static boolean isChannelPushNotification(Map<String, String> message) {
    return ChannelPushManager.isChannelPushNotification(message);
  }

  private void setUncaughtExceptionHandler() {
    uncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
    Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
      @Override
      public void uncaughtException(Thread thread, Throwable ex) {
        if (Looper.getMainLooper().getThread() == thread) {
          uncaughtExceptionHandler.uncaughtException(thread, ex);
        } else if (ex instanceof RejectedExecutionException) {
          SocketManager.reconnect();
        }
      }
    });
  }

  private void setChannelInfo(ChannelPluginSettings pluginSettings) {
    isDebugMode = pluginSettings.isDebugMode();
    isEnabledTrackDefaultEvent = pluginSettings.isEnabledTrackDefaultEvent();
    isHideDefaultInAppPush = pluginSettings.isHideDefaultInAppPush();

    ChannelStore.setPluginKey(pluginSettings.getPluginKey());
    PrefSupervisor.setPluginLanguage(channelIO.application, pluginSettings.getLocale());
    PrefSupervisor.setPluginSetting(channelIO.application, pluginSettings);
  }

  private void bootProcess(ChannelPluginSettings pluginSettings, Profile profile, OnBootListener listener) {
    if (isDataStored()) {
      shutdownProcess();
    }
    channelIO.checkVersion(pluginSettings, profile, listener);
  }

  private void checkVersion(
      final ChannelPluginSettings pluginSettings,
      final Profile profile,
      final OnBootListener listener) {
    final int counter = bootCounter.incrementAndGet();

    getApi().getLastestPackage("com.zoyi.channel.plugin.android", BuildConfig.VERSION_NAME)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<PackageWrapper>() {
          @Override
          public void onError(Throwable error) {
            if (counter != bootCounter.get()) {
              return;
            }
            if (listener != null) {
              listener.onCompletion(ChannelPluginCompletionStatus.NETWORK_TIMEOUT, null);
            }
          }

          @Override
          public void onNext(PackageWrapper wrapper) {
            if (counter != bootCounter.get()) {
              // show error
              return;
            }

            if (!wrapper.isNeedToUpgrade()) {
              if (!CompareUtils.isSame(wrapper.getVersionString(), BuildConfig.VERSION_NAME)) {
                L.i("Newest version is: " + wrapper.getVersionString());
              }
              bootChannelIO(pluginSettings, profile, listener, counter);
            } else {
              if (listener != null) {
                listener.onCompletion(ChannelPluginCompletionStatus.NOT_AVAILABLE_VERSION, null);
              }
            }
          }
        });
  }

  private void bootChannelIO(
      final ChannelPluginSettings pluginSettings,
      final Profile profile,
      final OnBootListener listener,
      final int counter) {
    Map<String, Object> form = new HashMap<>();
    if (profile != null) {
      form.put("name", profile.getName());
      form.put("avatarUrl", profile.getAvatarUrl());
      form.put("mobileNumber", profile.getMobileNumber());

      for (Map.Entry<String, Object> entry : profile.getProperty().entrySet()) {
        form.put(entry.getKey(), entry.getValue());
      }
    }

    RequestBody body = RequestUtils.form(form).create();

    Map<String, String> headers = new HashMap<>();

    if (!TextUtils.isEmpty(pluginSettings.getUserId())) {
      headers.put("X-User-Id", pluginSettings.getUserId());
    }

    getApi().boot(headers, pluginSettings.getPluginKey(), body)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<PluginWrapper>() {
          @Override
          public void onError(Throwable e) {
            if (counter != bootCounter.get()) {
              return;
            }

            L.e(e.getMessage());
            if (listener != null) {
              ChannelPluginCompletionStatus status = ChannelPluginCompletionStatus.UNKNOWN_ERROR;

              if (e instanceof RetrofitException) {
                switch (((RetrofitException) e).getKind()) {
                  case HTTP:
                    status = ChannelPluginCompletionStatus.NOT_INITIALIZED;
                    break;

                  case NETWORK:
                    status = ChannelPluginCompletionStatus.NETWORK_TIMEOUT;
                    break;

                  default:
                    status = ChannelPluginCompletionStatus.UNKNOWN_ERROR;
                    break;
                }
              }

              listener.onCompletion(status, null);
            }
          }

          @Override
          public void onNext(PluginWrapper pluginWrapper) {
            if (counter == bootCounter.get() && !pluginWrapper.getChannel().isBlockedPlugin()) {
              Channel channel = pluginWrapper.getChannel();
              Plugin plugin = pluginWrapper.getPlugin();

              if (pluginSettings.getUserId() != null) {
                CheckInPrefSupervisor.setProfile(application, pluginSettings.getUserId(), profile);
              }
              CheckInPrefSupervisor.setVeil(application, pluginWrapper.getVeil());

              ChannelStore.bootPlugin(pluginWrapper);
              initBugTracking(channel);
              ChannelPushManager.sendTokenToChannelIO(application);

              SocketManager.setChannelId(channel.getId());
              SocketManager.connect();

              if (!pluginSettings.isHideDefaultLauncher() &&
                  !plugin.isHideLauncher() &&
                  !channel.isHideLauncher()) {
                RxBus.post(new CommandBus(Command.SHOW_CHANNEL_VIEW, plugin));
              }

              if (pluginSettings.isEnabledTrackDefaultEvent()) {
                track(application, pluginSettings.getPluginKey(), CHDefaultEvent.Name.BOOT, null);
              }

              if (listener != null) {
                listener.onCompletion(
                    ChannelPluginCompletionStatus.SUCCESS,
                    pluginWrapper.getUser() != null
                        ? pluginWrapper.getUser()
                        : pluginWrapper.getVeil());
              }
            }
          }
        });
  }

  private void shutdownProcess() {
    bootCounter.incrementAndGet();

    ChannelPushManager.deleteToken(application);
    CheckInPrefSupervisor.clear(application);
    ChannelStore.clear();
    RxBus.post(new CommandBus(Command.CHECKED_OUT));

    SocketManager.setChannelId(null);
    SocketManager.disconnect();

    ChatManager.release();

    releaseBugTracking();
  }

  private void finishAll(boolean animated) {
    if (activityLifecycleManager != null) {
      activityLifecycleManager.finish(animated);
    }
  }

  public static ChannelApi getApi() {
    if (channelApi == null) {
      channelApi = ServiceFactory.create();
    }
    return channelApi;
  }

  private void initBugTracking(Channel channel) {
    if (application != null) {
      Mint.disableNetworkMonitoring();

      Mint.initAndStartSession(application, MINT_PROD_KEY);
      Mint.addExtraData("plugin_version", BuildConfig.VERSION_NAME);

      if (channel != null) {
        Mint.addExtraData("channel_id", channel.getId());
        Mint.addExtraData("channel_name", channel.getName());
      }
    }
  }

  private void releaseBugTracking() {
    if (application != null) {
      Mint.clearExtraData();
      Mint.closeSession(application);
      Mint.flush();
    }
  }

  public static void track(
      @NonNull Context context,
      @NonNull String pluginKey,
      @NonNull @Size(min = 1L, max = 30L) String eventName,
      @Nullable Map<String, Object> eventProperty) {
    if (TextUtils.isEmpty(eventName) || eventName.length() > 30) {
      return;
    }
    track(new Event(context, pluginKey, eventName, eventProperty));
  }

  private static void track(@NonNull Event event) {
    getApi().trackEvent(event)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<Void>());
  }

  public static boolean isInitializedChannelIO() {
    if (channelIO == null) {
      return false;
    }
    return true;
  }

  public static Context getAppContext() {
    if (channelIO != null) {
      return channelIO.application.getApplicationContext();
    }
    return null;
  }

  public static boolean isDebugMode() {
    return isDebugMode;
  }

  public static boolean isEnabledTrackDefaultEvent() {
    return isEnabledTrackDefaultEvent;
  }

  public static boolean isHideInAppPush() {
    return isHideDefaultInAppPush;
  }

  public static boolean isDataStored() {
    return ChannelStore.isDataStored();
  }

  public static void updateGuest() {
    updateGuest(null);
  }

  public static void updateGuest(OnGuestUpdatedListener listener) {
    ChannelStore.fetchMe(listener);
  }

  public static void addChannelPluginListener(ChannelPluginListener listener) {
    ChannelStore.addChannelPluginListener(listener);
  }

  public static void removeChannelPluginListener(ChannelPluginListener listener) {
    ChannelStore.removeChannelPluginListener(listener);
  }

  public static List<ChannelPluginListener> getChannelPluginListeners() {
    return ChannelStore.getChannelPluginListeners();
  }
}
