package com.zoyi.channel.plugin.android.activity.chat.view;

import android.content.Context;
import android.text.TextUtils;
import android.view.KeyEvent;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.widget.TextView;

import com.google.i18n.phonenumbers.NumberParseException;
import com.google.i18n.phonenumbers.PhoneNumberUtil;
import com.google.i18n.phonenumbers.PhoneNumberUtil.PhoneNumberFormat;
import com.google.i18n.phonenumbers.Phonenumber.PhoneNumber;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnChangeInputStateListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnCountryCodeSelectorClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnProfileEventListener;
import com.zoyi.channel.plugin.android.activity.chat.model.MobileNumber;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.CountryUtils;
import com.zoyi.channel.plugin.android.view.handler.EditTextChangedListener;

/**
 * Created by mika on 2018. 4. 17..
 */

public class MobileNumberProfileMessageView extends ProfileMessageView {

  private MobileNumberProfileInputView mobileNumberProfileInputView;

  public MobileNumberProfileMessageView(
      Context context,
      OnProfileEventListener onProfileEventListener,
      OnChangeInputStateListener onChangeInputStateListener) {
    super(context, onProfileEventListener, onChangeInputStateListener);
  }

  @Override
  protected void initInputView(Context context) {
    mobileNumberProfileInputView = new MobileNumberProfileInputView(context);
    overrideInputView(mobileNumberProfileInputView);
  }

  @Override
  protected void initInputViewEventListener() {
    mobileNumberProfileInputView.setEditTextChangedListener(new EditTextChangedListener() {
      @Override
      public void onWatchedTextChanged(String text) {
        if (text.contains(" ")) {
          text = text.replace(" ", "");
          mobileNumberProfileInputView.setText(text);
        }
        if (hasEditTextFocus) {
          onChangeInputStateListener.onUpdateInputState(key, mobileNumberProfileInputView.getMobileNumber(), !text.isEmpty());
        }
        setStyle(false, mobileNumberProfileInputView.isInputFocused(), isValidInputState(mobileNumberProfileInputView.getMobileNumber()), isLoading);

      }
    });

    mobileNumberProfileInputView.setOnFocusChangeListener(new OnFocusChangeListener() {
      @Override
      public void onFocusChange(View v, boolean hasFocus) {
        hasEditTextFocus = hasFocus;
        setStyle(hasError, hasFocus, isValidInputState(mobileNumberProfileInputView.getMobileNumber()), isLoading);
        onProfileEventListener.onChangeFocus(hasFocus);
      }
    });

    mobileNumberProfileInputView.setOnCountryCodeSelectorClickListener(new OnCountryCodeSelectorClickListener() {
      @Override
      public void onClick(MobileNumber mobileNumber) {
        if (!isLoading && onProfileEventListener != null) {
          onProfileEventListener.onCountryCodeClick(key, mobileNumber);
        }
      }
    });

    mobileNumberProfileInputView.setOnEditorActionListener(new TextView.OnEditorActionListener() {
      @Override
      public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
        if (actionId == EditorInfo.IME_ACTION_DONE || actionId == EditorInfo.IME_ACTION_NEXT) {
          if (isValidInputState(getValue())) {
            onProfileEventListener.onSendButtonClick(key, getValue(), getInputState());
          }
          return true;
        }
        return false;
      }
    });

  }

  @Override
  protected boolean isInputViewFocused() {
    return mobileNumberProfileInputView.isInputFocused();
  }

  @Override
  protected void requestFocusInputView() {
    mobileNumberProfileInputView.requestFocusInputView();
  }

  @Override
  protected void setInputState(Object inputState, boolean isLoading, boolean isLastProfile) {
    if (inputState != null) {
      if (inputState instanceof MobileNumber) {
        MobileNumber mobileNumber = (MobileNumber) inputState;
        mobileNumberProfileInputView.setMobileNumber(mobileNumber.getCountryCode(), mobileNumber.getPhoneNumber());
        mobileNumberProfileInputView.setSelection(mobileNumber.getPhoneNumber().length());
      }
      if (inputState instanceof String) {
        MobileNumber mobileNumber = parsePhoneNumber((String) inputState);
        mobileNumberProfileInputView.setMobileNumber(mobileNumber.getCountryCode(), mobileNumber.getPhoneNumber());
        mobileNumberProfileInputView.setSelection(mobileNumber.getPhoneNumber().length());
      }
      mobileNumberProfileInputView.setEnabled(!isLoading);
    }
    mobileNumberProfileInputView.setImeOption(isLastProfile ? EditorInfo.IME_ACTION_DONE : EditorInfo.IME_ACTION_NEXT);
  }

  private MobileNumber parsePhoneNumber(String value) {
    PhoneNumberUtil phoneUtil = PhoneNumberUtil.getInstance();
    try {
      PhoneNumber numberProto = phoneUtil.parse(value, "");

      return new MobileNumber(
          numberProto.getCountryCode(),
          phoneUtil.format(numberProto, PhoneNumberFormat.NATIONAL));
    } catch (NumberParseException e) {
      return new MobileNumber(CountryUtils.getDefaultCountryCodeInt(), "");
    }
  }

  @Override
  protected Object getValue() {
    MobileNumber mobileNumber = mobileNumberProfileInputView.getMobileNumber();
    return String.format("+%s%s", mobileNumber.getCountryCode(), mobileNumber.getPhoneNumber()).replace("-", "");
  }

  @Override
  protected Object getInputState() {
    return mobileNumberProfileInputView.getMobileNumber();
  }

  @Override
  protected boolean isValidInputState(Object value) {
    if (value != null) {
      if (value instanceof MobileNumber) {
        return !TextUtils.isEmpty(((MobileNumber) value).getPhoneNumber());
      }
      if (value instanceof String) {
        MobileNumber mobileNumber = parsePhoneNumber((String) value);
        return !TextUtils.isEmpty(mobileNumber.getPhoneNumber());
      }
    }
    return false;
  }

  @Override
  protected String getFormattedString(Object object) {
    PhoneNumberUtil phoneUtil = PhoneNumberUtil.getInstance();
    try {
      PhoneNumber numberProto = phoneUtil.parse(object.toString(), "");
      return phoneUtil.format(numberProto, PhoneNumberFormat.INTERNATIONAL);
    } catch (NumberParseException e) {
      return object == null ? "" : object.toString();
    }
  }
}
