package com.zoyi.channel.plugin.android;

import android.app.Application;
import android.support.annotation.NonNull;

import com.zoyi.channel.plugin.android.event.BadgeBus;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.PushBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.CheckInPrefSupervisor;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.etc.PushEvent;
import com.zoyi.channel.plugin.android.model.rest.Bot;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.model.wrapper.ChannelWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.GuestWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscriber;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Action1;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

public class ChannelStore {
  private static ChannelStore channelStore;

  private Application application;
  private String pluginKey;

  private Plugin plugin;
  private Channel channel;
  private User user;
  private Veil veil;
  private Bot bot;

  private int activityRunningCount = 0;

  private Subscription channelFetchSubscription;
  private Subscription socketSubscription;

  private List<ChannelPluginListener> listeners;

  static void create(Application application) {
    if (channelStore == null) {
      channelStore = new ChannelStore(application);
    }
  }

  private ChannelStore(Application application) {
    this.application = application;
    this.listeners = new ArrayList<>();
  }

  public static void setPluginKey(String key) {
    if (channelStore != null) {
      channelStore.pluginKey = key;
    }
  }

  static void checkIn(PluginWrapper pluginWrapper) {
    if (channelStore != null) {
      PrefSupervisor.setVeilId(channelStore.application, pluginWrapper.getVeilId());

      if (pluginWrapper.getUser() != null) {
        channelStore.user = pluginWrapper.getUser();
      }
      channelStore.plugin = pluginWrapper.getPlugin();
      channelStore.channel = pluginWrapper.getChannel();
      channelStore.veil = pluginWrapper.getVeil();

      channelStore.startReceiveSocket();
      channelStore.startFetchChannel();
    }
  }

  static void clear() {
    if (channelStore != null) {
      channelStore.stopFetchChannel();
      channelStore.stopReceiveSocket();
      channelStore.plugin = null;
      channelStore.channel = null;
      channelStore.user = null;
      channelStore.veil = null;
    }
  }

  static void fetchMe(final OnGuestUpdatedListener listener) {
    if (channelStore != null) {
      ChannelIO.getApi().getMe()
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<GuestWrapper>() {
            @Override
            public void onNext(GuestWrapper guestWrapper) {
              if (channelStore != null && guestWrapper != null) {
                setUserVeil(guestWrapper.getUser(), guestWrapper.getVeil());

                if (listener != null) {
                  listener.onGuestUpdated();
                }
              }
            }
          });
    }
  }

  private void startReceiveSocket() {
    if (socketSubscription == null || socketSubscription.isUnsubscribed()) {
      socketSubscription = RxBus.observable()
          .onBackpressureBuffer()
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new Action1<Object>() {
            @Override
            public void call(Object o) {
              if (o instanceof CommandBus) {
                CommandBus commandBus = (CommandBus) o;

                switch (commandBus.getCommand()) {
                  case READY:
                    fetchMe(null);
                    break;
                }
              }

              if (o instanceof BadgeBus) {
                for (ChannelPluginListener listener : listeners) {
                  if (listener != null) {
                    listener.onChangeBadge(((BadgeBus) o).getCount());
                  }
                }
              }

              if (o instanceof PushBus) {
                PushBus pushBus = (PushBus) o;
                for (ChannelPluginListener listener : listeners) {
                  if (listener != null) {
                    listener.onReceivePush(new PushEvent(pushBus));
                  }
                }
              }
            }
          });
    }
  }

  private void stopReceiveSocket() {
    if (socketSubscription != null && !socketSubscription.isUnsubscribed()) {
      socketSubscription.unsubscribe();
    }
  }

  private void startFetchChannel() {
    stopFetchChannel();
    channelFetchSubscription = Observable.interval(3, TimeUnit.MINUTES)
        .subscribe(new Subscriber<Long>() {
          @Override
          public void onCompleted() {
          }

          @Override
          public void onError(Throwable e) {
          }

          @Override
          public void onNext(Long aLong) {
            if (channelStore != null) {
              channelStore.fetchChannel();
            }
          }
        });
  }

  private void stopFetchChannel() {
    if (channelFetchSubscription != null && !channelFetchSubscription.isUnsubscribed()) {
      channelFetchSubscription.unsubscribe();
    }
  }

  private void fetchChannel() {
    ChannelIO.getApi()
        .getChannels()
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<ChannelWrapper>() {
          @Override
          public void onNext(ChannelWrapper channelWrapper) {
            if (channelWrapper != null) {
              channel = channelWrapper.getChannel();
            }
          }
        });
  }

  public static void setChannel(Channel channel) {
    if (channelStore != null && channel != null) {
      channelStore.channel = channel;
    }
  }

  public static Channel getChannel() {
    return channelStore != null ? channelStore.channel : null;
  }

  public static User getUser() {
    return channelStore != null ? channelStore.user : null;
  }

  public static Bot getBot() {
    return channelStore != null ? channelStore.bot : null;
  }

  public static String getName() {
    if (channelStore != null) {
      if (channelStore.user != null && !channelStore.user.isGhost()) {
        return channelStore.user.getName();
      }
      if (channelStore.veil != null && !channelStore.veil.isGhost()) {
        return channelStore.veil.getName();
      }
    }
    return null;
  }

  public static String getMobileNumber() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return channelStore.user.getMobileNumber();
      }
      if (channelStore.veil != null) {
        return channelStore.veil.getMobileNumber();
      }
    }
    return null;
  }

  public static String getPersonType() {
    if (channelStore != null && channelStore.user != null) {
      return User.CLASSNAME;
    }
    return Veil.CLASSNAME;
  }

  public static String getPersonId() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return channelStore.user.getId();
      }
      return PrefSupervisor.getVeilId(channelStore.application);
    }
    return null;
  }

  public static String getVeilId() {
    if (channelStore != null && channelStore.application != null) {
      return PrefSupervisor.getVeilId(channelStore.application);
    }
    return null;
  }

  public static String getUserId() {
    if (channelStore != null && channelStore.user != null) {
      return channelStore.user.getId();
    }
    return null;
  }

  public static boolean isLauncherHidden() {
    if (channelStore == null || channelStore.channel == null || channelStore.plugin == null) {
      return true;
    }

    if (channelStore.plugin.isHideButton()) {
      return true;
    } else {
      return !channelStore.channel.isWorking() && Const.AWAY_OPTION_HIDDEN.equals(channelStore.channel.getAwayOption());
    }
  }

  public static boolean isRequestGuestInfo() {
    return channelStore != null && channelStore.channel != null && channelStore.channel.isRequestGuestInfo();
  }

  public static boolean hasName() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return !channelStore.user.isGhost();
      }
      if (channelStore.veil != null) {
        return !channelStore.veil.isGhost();
      }
    }
    return true;
  }

  public static void setUserVeil(User user, Veil veil) {
    if (channelStore != null) {
      if (user != null) {
        channelStore.user = user;
      }
      if (veil != null) {
        channelStore.veil = veil;
      }
      if (channelStore.application != null) {
        CheckInPrefSupervisor.setName(channelStore.application, getName());
        CheckInPrefSupervisor.setMobileNumber(channelStore.application, getMobileNumber());
      }
      RxBus.post(new BadgeBus(getCount()));
    }
  }

  public static void setBot(Bot bot) {
    if (channelStore != null && bot != null) {
      channelStore.bot = bot;
    }
  }

  public static Plugin getPlugin() {
    if (channelStore != null) {
      return channelStore.plugin;
    }
    return null;
  }

  public static String getPluginKey() {
    if (channelStore != null) {
      return channelStore.pluginKey;
    }
    return null;
  }

  public static void increaseActivityRunnigCount() {
    if (channelStore != null) {
      channelStore.activityRunningCount++;
    }
  }

  public static void decreaseActivityRunnigCount() {
    if (channelStore != null) {
      channelStore.activityRunningCount--;
    }
  }

  public static boolean isMainRunning() {
    return channelStore != null && channelStore.activityRunningCount > 0;
  }

  static boolean isDataStored() {
    return channelStore != null && channelStore.channel != null && channelStore.plugin != null;
  }

  static int getCount() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return channelStore.user.getAlert();
      }
      if (channelStore.veil != null) {
        return channelStore.veil.getAlert();
      }
    }
    return 0;
  }

  @NonNull
  static List<ChannelPluginListener> getChannelPluginListeners() {
    if (channelStore == null) {
      return new ArrayList<>();
    }
    return channelStore.listeners;
  }

  static void addChannelPluginListener(ChannelPluginListener listener) {
    if (channelStore != null && channelStore.listeners != null && listener != null) {
      channelStore.listeners.add(listener);
    }
  }

  static void removeChannelPluginListener(ChannelPluginListener listener) {
    if (channelStore != null && channelStore.listeners != null && listener != null) {
      channelStore.listeners.remove(listener);
    }
  }
}
