package com.zoyi.channel.plugin.android;

import android.app.Application;
import android.content.Context;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.Size;
import android.text.TextUtils;

import com.splunk.mint.Mint;
import com.zoyi.channel.plugin.android.activity.chat.ChatActivity;
import com.zoyi.channel.plugin.android.activity.chat.ChatManager;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnChatListener;
import com.zoyi.channel.plugin.android.activity.userchat_list.UserChatListActivity;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.LauncherBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.CheckInPrefSupervisor;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Event;
import com.zoyi.channel.plugin.android.model.wrapper.PackageWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.ChannelApi;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.network.ServiceFactory;
import com.zoyi.channel.plugin.android.push.ChannelPushClient;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.channel.plugin.android.wrapper.Tracker;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicInteger;

public class ChannelIO {
  private static final String MINT_DEBUG_KEY = "e72f17f4";
  private static final String MINT_PROD_KEY = "22464da6";

  private static boolean isDebugMode = false;
  private static boolean isEnabledTrackDefaultEvent = true;
  private static boolean isHideDefaultInAppPush = true;

  private static ChannelIO channelIO;
  private static ChannelApi channelApi;
  private static OnChatListener onChatListener = null;

  private Application application;
  private Thread.UncaughtExceptionHandler uncaughtExceptionHandler;

  private AtomicInteger checkInCounter = new AtomicInteger(0);

  private ActivityLifecycleManager activityLifecycleManager;

  private ChannelIO(Application application) {
    if (isInitializedChannelIO()) {
      L.e("Channel plugin already initialized");
      return;
    }
    if (application == null) {
      L.e("Application cannot be null");
      return;
    }

    this.application = application;

    activityLifecycleManager = new ActivityLifecycleManager();
    application.registerActivityLifecycleCallbacks(activityLifecycleManager);

    setUncaughtExceptionHandler();
    ChannelStore.create(application);
    SocketManager.create(application);
  }

  // Main Method

  public static void initialize(@NonNull Application application) {
    channelIO = new ChannelIO(application);
  }

  public static void boot(ChannelPluginSettings pluginSettings) {
    boot(pluginSettings, null, null);
  }

  public static void boot(ChannelPluginSettings pluginSettings, OnBootListener completion) {
    boot(pluginSettings, null, completion);
  }

  public static void boot(ChannelPluginSettings pluginSettings, Guest guest) {
    boot(pluginSettings, guest, null);
  }

  public static void boot(ChannelPluginSettings pluginSettings, Guest guest, OnBootListener completion) {
    if (isInitializedChannelIO() && pluginSettings != null && pluginSettings.getPluginKey() != null) {
      channelIO.setChannelInfo(pluginSettings);
      channelIO.checkIn(pluginSettings, guest, completion);
    } else {
      L.e("Check plugin information");
    }
  }

  public static void shutdown() {
    if (isInitializedChannelIO()) {
      channelIO.checkOutProcess();
    }
  }

  public static boolean open(@NonNull Context context) {
    if (!isInitializedChannelIO() || !isDataStored()) {
      L.e("Please check in first");
      return false;
    }

    for (ChannelPluginListener l : ChannelStore.getChannelPluginListeners()) {
      if (l != null) {
        l.willShowMessenger();
      }
    }
    IntentUtils.setNextActivity(context, UserChatListActivity.class).startActivity();
    return true;
  }

  public static void close() {
    if (isInitializedChannelIO()) {
      channelIO.finishAll();
    }
  }

  public static void show() {
    RxBus.post(new LauncherBus(true));
  }

  public static void hide() {
    RxBus.post(new LauncherBus(false));
  }

  public static boolean openChat(@NonNull Context context) {
    return openChat(context, null);
  }

  public static boolean openChat(@NonNull Context context, @Nullable String chatId) {
    if (!isInitializedChannelIO() || !isDataStored()) {
      L.e("Please Boot first");
      return false;
    }

    IntentUtils.setNextActivity(context, ChatActivity.class)
        .putExtra(Const.EXTRA_CHAT_COUNT, 0)
        .putExtra(Const.EXTRA_CHAT_FIRST, chatId == null)
        .putExtra(Const.EXTRA_CHAT_ID, chatId)
        .startActivityForResult(Const.REQUEST_CHAT);
    return true;
  }

  public static void handlePushNotification(Context context) {
    ChannelPushClient.handlePushNotification(context);
  }

  public static void showPushNotification(Context context, Map<String, String> message) {
    ChannelPushManager.showPushNotification(context, message);
  }

  public static boolean isChannelPushNotification(Map<String, String> message) {
    return ChannelPushManager.isChannelPushNotification(message);
  }

  private void setUncaughtExceptionHandler() {
    uncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
    Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
      @Override
      public void uncaughtException(Thread thread, Throwable ex) {
        if (Looper.getMainLooper().getThread() == thread) {
          uncaughtExceptionHandler.uncaughtException(thread, ex);
        } else if (ex instanceof RejectedExecutionException) {
          SocketManager.reconnect();
        }
      }
    });
  }

  private void setChannelInfo(ChannelPluginSettings pluginSettings) {
    isDebugMode = pluginSettings.isDebugMode();
    isEnabledTrackDefaultEvent = pluginSettings.isEnabledTrackDefaultEvent();
    isHideDefaultInAppPush = pluginSettings.isHideDefaultInAppPush();

    ChannelStore.setPluginKey(pluginSettings.getPluginKey());
    PrefSupervisor.setPluginSetting(channelIO.application, pluginSettings);
  }

  private void checkIn(ChannelPluginSettings pluginSettings, Guest guest, OnBootListener listener) {
    if (isDataStored()) {
      checkOutProcess();
    }
    channelIO.checkVersion(pluginSettings, guest, listener);
  }

  private void checkVersion(
      final ChannelPluginSettings pluginSettings,
      final Guest guest,
      final OnBootListener listener) {
    final int counter = checkInCounter.incrementAndGet();

    getApi().getLastestPackage("com.zoyi.channel.plugin.android", BuildConfig.VERSION_NAME)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<PackageWrapper>() {
          @Override
          public void onError(Throwable error) {
            if (counter != checkInCounter.get()) {
              return;
            }
            if (listener != null) {
              listener.onCompletion(ChannelPluginCompletionStatus.NETWORK_TIMEOUT);
            }
          }

          @Override
          public void onNext(PackageWrapper wrapper) {
            if (counter != checkInCounter.get()) {
              // show error
              return;
            }

            if (!wrapper.isNeedToUpgrade()) {
              if (!CompareUtils.isSame(wrapper.getVersionString(), BuildConfig.VERSION_NAME)) {
                L.i("Newest version is: " + wrapper.getVersionString());
              }
              checkInProcess(pluginSettings, guest, listener, counter);
            } else {
              if (listener != null) {
                listener.onCompletion(ChannelPluginCompletionStatus.NOT_AVAILABLE_VERSION);
              }
            }
          }
        });
  }

  private void checkInProcess(
      final ChannelPluginSettings pluginSettings,
      final Guest guest,
      final OnBootListener listener,
      final int counter) {
    Map<String, Object> form = new HashMap<>();
    if (guest != null) {
      form.put("name", guest.getName());
      form.put("avatarUrl", guest.getAvatarUrl());
      form.put("mobileNumber", guest.getMobileNumber());
      form.put("meta", guest.getProperty());
    }

    RequestBody body = RequestUtils.form(form).create();
    Map<String, String> headers = new HashMap<>();

    if (guest != null && guest.getId() != null) {
      headers.put("X-User-Id", guest.getId());
    }
    if (PrefSupervisor.getVeilId(application) != null) {
      headers.put("X-Veil-Id", PrefSupervisor.getVeilId(application));
    }

    getApi().checkIn(headers, pluginSettings.getPluginKey(), null, body)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<PluginWrapper>() {
          @Override
          public void onError(Throwable e) {
            if (counter != checkInCounter.get()) {
              return;
            }

            L.e(e.getMessage());
            if (listener != null) {
              ChannelPluginCompletionStatus status = ChannelPluginCompletionStatus.UNKNOWN_ERROR;

              if (e instanceof RetrofitException) {
                switch (((RetrofitException) e).getKind()) {
                  case HTTP:
                    status = ChannelPluginCompletionStatus.NOT_INITIALIZED;
                    break;

                  case NETWORK:
                    status = ChannelPluginCompletionStatus.NETWORK_TIMEOUT;
                    break;

                  default:
                    status = ChannelPluginCompletionStatus.UNKNOWN_ERROR;
                    break;
                }
              }

              listener.onCompletion(status);
            }
          }

          @Override
          public void onNext(PluginWrapper pluginWrapper) {
            if (counter == checkInCounter.get() &&
                !pluginWrapper.getChannel().isBlockedPlugin() &&
                !pluginSettings.isHideDefaultLauncher()) {

              if (guest == null) {
                CheckInPrefSupervisor.setVeil(application, pluginWrapper.getVeil());
              } else {
                CheckInPrefSupervisor.setGuest(application, guest);
              }

              if (pluginSettings.isEnabledTrackDefaultEvent()) {
                track(application, pluginSettings.getPluginKey(), Tracker.Name.CHECK_IN, null);
              }

              initBugTracking(pluginWrapper.getChannel());
              ChannelStore.checkIn(pluginWrapper);
              ChannelPushManager.sendTokenToChannelIO(application);

              SocketManager.setChannelId(pluginWrapper.getChannel().getId());
              SocketManager.connect();

              RxBus.post(new CommandBus(Command.CHECKED_IN, pluginWrapper.getPlugin()));
            if (listener != null) {
              listener.onCompletion(ChannelPluginCompletionStatus.SUCCESS);
            }
            }
          }
        });
  }

  private void checkOutProcess() {
    checkInCounter.incrementAndGet();

    ChannelPushManager.deleteToken(application);
    CheckInPrefSupervisor.clear(application);
    ChannelStore.clear();
    RxBus.post(new CommandBus(Command.CHECKED_OUT));

    SocketManager.setChannelId(null);
    SocketManager.disconnect();

    ChatManager.release();

    releaseBugTracking();
  }

  private void finishAll() {
    if (activityLifecycleManager != null) {
      activityLifecycleManager.finish();
    }
  }

  public static ChannelApi getApi() {
    if (channelApi == null) {
      channelApi = ServiceFactory.create();
    }
    return channelApi;
  }

  private void initBugTracking(Channel channel) {
    if (application != null) {
      Mint.disableNetworkMonitoring();

      Mint.initAndStartSession(application, MINT_PROD_KEY);
      Mint.addExtraData("plugin_version", BuildConfig.VERSION_NAME);

      if (channel != null) {
        Mint.addExtraData("channel_id", channel.getId());
        Mint.addExtraData("channel_name", channel.getName());
      }
    }
  }

  private void releaseBugTracking() {
    if (application != null) {
      Mint.clearExtraData();
      Mint.closeSession(application);
      Mint.flush();
    }
  }

  public static void track(
      @NonNull Context context,
      @NonNull String pluginKey,
      @NonNull @Size(min = 1L, max = 30L) String eventName,
      @Nullable Map<String, Object> eventProperty) {
    if (TextUtils.isEmpty(eventName) || eventName.length() > 30) {
      return;
    }
    track(new Event(context, pluginKey, eventName, eventProperty));
  }

  private static void track(@NonNull Event event) {
    getApi().trackEvent(event)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<Void>());
  }

  public static boolean isInitializedChannelIO() {
    if (channelIO == null) {
      return false;
    }
    return true;
  }

  public static Context getAppContext() {
    if (channelIO != null) {
      return channelIO.application.getApplicationContext();
    }
    return null;
  }

  public static boolean isDebugMode() {
    return isDebugMode;
  }

  public static boolean isEnabledTrackDefaultEvent() {
    return isEnabledTrackDefaultEvent;
  }

  public static boolean isHideInAppPush() {
    return isHideDefaultInAppPush;
  }

  public static boolean isDataStored() {
    return ChannelStore.isDataStored();
  }

  public static void updateGuest() {
    updateGuest(null);
  }

  public static void updateGuest(OnGuestUpdatedListener listener) {
    ChannelStore.fetchMe(listener);
  }

  public static void addChannelPluginListener(ChannelPluginListener listener) {
    ChannelStore.addChannelPluginListener(listener);
  }

  public static void removeChannelPluginListener(ChannelPluginListener listener) {
    ChannelStore.removeChannelPluginListener(listener);
  }

  @Nullable
  public static OnChatListener getOnChatListener() {
    return onChatListener;
  }

  public static void setOnChatListener(@NonNull OnChatListener listener) {
    if (isInitializedChannelIO()) {
      onChatListener = listener;
    }
  }
}
