package com.zoyi.channel.plugin.android.activity.photo_picker;

import android.content.Intent;
import android.os.Bundle;
import android.support.v7.widget.GridLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.text.TextUtils;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.Toast;
import com.zoyi.channel.plugin.android.ChannelStore;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.source.photopicker.PhotoItem;
import com.zoyi.channel.plugin.android.presenter.photopicker.PhotoPickerContract;
import com.zoyi.channel.plugin.android.presenter.photopicker.PhotoPickerPresenter;
import com.zoyi.channel.plugin.android.util.Executor;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.view.layout.BigBar.MenuPosition;
import com.zoyi.channel.plugin.android.view.layout.MenuButton.MenuState;

import java.util.ArrayList;

import static com.zoyi.channel.plugin.android.view.layout.MenuButton.ActionType;

/**
 * Created by mika on 8/31/16.
 */
public class PhotoPickerActivity extends BaseActivity implements PhotoPickerContract.View {

  private LinearLayout emptyLayout;

  private PhotoPickerContract.Presenter presenter;

  @Override
  protected void onCreate(Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    init(R.layout.ch_plugin_activity_photo_picker);

    RecyclerView recyclerView = (RecyclerView) findViewById(R.id.recycler_view_photo_picker);
    emptyLayout = (LinearLayout) findViewById(R.id.no_album);

    getBigBar().withActivity(this)
        .setTitle(ResUtils.getString(this, "ch.photo.all_images"))
        .addMenu(ActionType.BACK, MenuPosition.LEFT)
        .addMenu(ActionType.CAMERA, MenuPosition.RIGHT)
        .addMenu(ActionType.DONE, MenuPosition.RIGHT, MenuState.DISABLED)
        .build();

    PhotoPickerAdapter adapter = new PhotoPickerAdapter(this);

    presenter = new PhotoPickerPresenter(this);
    presenter.setView(this);
    presenter.setAdapterView(adapter);
    presenter.setAdapterModel(adapter);

    recyclerView.setLayoutManager(new GridLayoutManager(this, Const.COLUMNS_PHOTO_PICKER));
    recyclerView.setAdapter(adapter);
    recyclerView.setItemAnimator(null);

    setStyle();
  }

  private void setStyle() {
    Plugin plugin = ChannelStore.getPlugin();

    if (plugin != null) {
      setStatusBarColor(plugin.getBackgroundColor());
      getBigBar().setTheme(plugin.getBackgroundColor(), plugin.getTextColor());
    }
  }

  @Override
  protected void onResume() {
    super.onResume();
    fetchPhotoItems();
  }

  @Override
  public void onBackPressed() {
    setResult(RESULT_CANCELED);
    super.onBackPressed();
  }

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    if (resultCode == RESULT_OK) {
      switch (requestCode) {
        case Const.REQUEST_TAKE_PHOTO:
          String photoPath = Executor.getPhotoPath();

          if (!TextUtils.isEmpty(photoPath)) {
            Executor.startFileMediaScan(this, photoPath);

            sendPhotoPath(Executor.getPhotoPath());
          }
          return;
      }
    }
    super.onActivityResult(requestCode, resultCode, data);
  }

  private void sendPhotoPath(String photoPath) {
    ArrayList<String> photoPaths = new ArrayList<>();

    photoPaths.add(photoPath);

    sendPhotoPaths(photoPaths);
  }

  @Override
  public void optionClicked(ActionType actionType) {
    super.optionClicked(actionType);
    switch (actionType) {
      case BACK:
        onBackPressed();
        break;

      case CAMERA:
        openCamera();
        break;

      case DONE:
        presenter.collectSelectedPhotoItem();
        break;
    }
  }

  @Override
  public void loadPhotoItems(ArrayList<PhotoItem> photoItems) {
    setEmptyLayoutVisibility(photoItems.isEmpty());
  }

  @Override
  public void sendPhotoPaths(ArrayList<String> imagePaths) {
    Intent i = new Intent();
    i.putStringArrayListExtra(Const.PHOTO_INTENT_KEY, imagePaths);

    setResult(Const.RESULT_PHOTO_CODE, i);
    finish();
  }

  @Override
  public void setMenuState(int selectedCount) {
    getBigBar().setMenuState(
        ActionType.DONE,
        selectedCount > 0 ? MenuState.VISIBLE : MenuState.DISABLED);
  }

  private void setEmptyLayoutVisibility(boolean visible) {
    emptyLayout.setVisibility(visible ? View.VISIBLE : View.GONE);
  }

  @Override
  protected void permissionAccepted(int requestCode) {
    switch (requestCode) {
      case PERMISSION_WRITE_STORAGE:
        fetchPhotoItemsPermissionAccepted();
        break;

      case PERMISSION_CAMERA:
        openCameraPermissionAccepted();
        break;
    }
  }

  @Override
  protected void permissionRejected(int requestCode) {
    if (requestCode == PERMISSION_WRITE_STORAGE) {
      finish();
    }
  }

  private void fetchPhotoItems() {
    if (checkPermission(PERMISSION_WRITE_STORAGE)) {
      fetchPhotoItemsPermissionAccepted();
    }
  }

  private void fetchPhotoItemsPermissionAccepted() {
    presenter.fetchPhotoItems();
  }

  private void openCamera() {
    if (checkPermission(PERMISSION_CAMERA)) {
      openCameraPermissionAccepted();
    }
  }

  private void openCameraPermissionAccepted() {
    if (!Executor.takePhoto(this)) {
      Toast.makeText(
          this,
          ResUtils.getString(this, "ch.permission.denied"),
          Toast.LENGTH_LONG)
          .show();
    }
  }
}
