package com.zoyi.channel.plugin.android.activity.chat;

import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;

import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.event.TypingBus;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.view.layout.WatchedEditText;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscriber;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;
import com.zoyi.rx.subscriptions.CompositeSubscription;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class TypingManager implements TextWatcher {

  private static final int TYPING_THROTTLE = 1000;

  private String chatId, chatType;

  private long typingStartTime = 0;
  private Map<String, Subscription> typingSubscriptionItems;
  private CompositeSubscription ttlSubscriptions;

  public TypingManager(String chatId, String chatType, WatchedEditText editText) {
    this.chatId = chatId;
    this.chatType = chatType;
    editText.addTextChangedListener(this);

    typingSubscriptionItems = new HashMap<>();
    ttlSubscriptions = new CompositeSubscription();
  }

  public void release() {
    if (ttlSubscriptions != null && !ttlSubscriptions.isUnsubscribed()) {
      ttlSubscriptions.unsubscribe();
    }
    typingSubscriptionItems.clear();
    stopTyping();
  }

  @Override
  public void beforeTextChanged(CharSequence s, int start, int count, int after) {}

  @Override
  public void onTextChanged(CharSequence s, int start, int before, int count) {
    String msg = s.toString();
    if (TextUtils.isEmpty(msg)) {
      stopTyping();
    } else {
      if (typingStartTime == 0) {
        startTyping();
      } else {
        if (System.currentTimeMillis() - typingStartTime > TYPING_THROTTLE) {
          startTyping();
        }
      }
    }
  }

  @Override
  public void afterTextChanged(Editable s) {}

  public void startTyping() {
    typingStartTime = System.currentTimeMillis();

    SocketManager.typing(new Typing("start", chatId, chatType));
  }

  public void stopTyping() {
    typingStartTime = 0;

    SocketManager.typing(new Typing("stop", chatId, chatType));
  }

  public void addTTLTimer(Typing typing) {
    Subscription subscription = typingSubscriptionItems.get(typing.getKey());
    if (subscription != null) {
      removeTTLSubscription(typing);
    }

    switch (typing.getAction()) {
      case "start":
        addTTLSubscription(typing);
        break;
    }
  }

  private void removeTTLSubscription(Typing typing) {
    Subscription subscription = typingSubscriptionItems.remove(typing.getKey());
    if (subscription != null) {
      ttlSubscriptions.remove(subscription);
    }
  }

  private void addTTLSubscription(final Typing typing) {
    Subscription subscription = Observable.timer(
        Typing.TTL,
        TimeUnit.SECONDS,
        Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new Subscriber<Long>() {
          @Override
          public void onCompleted() {}

          @Override
          public void onError(Throwable e) {}

          @Override
          public void onNext(Long aLong) {
            if (typing != null && typing.isExpired()) {
              typing.setAction("stop");
              RxBus.post(new TypingBus(typing));
            }
          }
        });

    typingSubscriptionItems.put(typing.getKey(), subscription);
    ttlSubscriptions.add(subscription);
  }
}
