package com.zoyi.channel.plugin.android.activity.chat;

import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import android.widget.Button;
import android.widget.ImageButton;
import com.zoyi.channel.plugin.android.ChannelStore;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity;
import com.zoyi.channel.plugin.android.activity.chat.listener.ChatActionCallback;
import com.zoyi.channel.plugin.android.activity.chat.listener.MessageClickListener;
import com.zoyi.channel.plugin.android.activity.chat.model.ActionMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.QuestionMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.type.ChatState;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.download.DownloadActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumActivity;
import com.zoyi.channel.plugin.android.activity.photo_picker.PhotoPickerActivity;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.MenuPosition;
import com.zoyi.channel.plugin.android.enumerate.MenuState;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatWrapper;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.util.lang.StringUtils;
import com.zoyi.channel.plugin.android.view.handler.BackgroundToucher;
import com.zoyi.channel.plugin.android.view.handler.InfiniteScrollListener;
import com.zoyi.channel.plugin.android.view.layout.WatchedEditText;

import java.util.*;

/**
 * Created by mika on 2016. 12. 7..
 */
public class ChatActivity extends BaseActivity implements ChatActionCallback, MessageClickListener {
  private String chatId;

  private RecyclerView recyclerView;
  private WatchedEditText editChat;
  private Button buttonSend;
  private ImageButton buttonAttach;
  private View toast, chatLayout;

  private ChatAdapter adapter;
  private ChatAction action;
  private LinearLayoutManager layoutManager;
  private ChatState chatState = ChatState.USER_CHAT_NOT_LOADED;
  private Queue<SendingMessageItem> sendQueue;
  private boolean sending = false;
  private UserChat userChat = null;
  private int chatCount = 0;

  @Override
  protected void onCreate(@Nullable Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    init(R.layout.ch_plugin_activity_chat);

    sendQueue = new LinkedList<>();

    chatLayout = findViewById(R.id.chat_layout);
    editChat = (WatchedEditText) findViewById(R.id.edit_chat);
    buttonSend = (Button) findViewById(R.id.button_send);
    buttonAttach = (ImageButton) findViewById(R.id.button_attach);
    toast = findViewById(R.id.layout_toast);
    findViewById(R.id.button_close_toast).setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        if (adapter != null) {
          adapter.setNewMessageDismiss();
          setAlertState(-1);
        }
      }
    });

    editChat.setHandledButton(
        buttonSend,
        ContextCompat.getColor(this, R.color.ch_cobalt),
        ContextCompat.getColor(this, R.color.ch_gray));

    buttonSend.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        sendTextMessage();
      }
    });
    buttonAttach.setOnClickListener(new View.OnClickListener() {
      @Override
      public void onClick(View v) {
        hideKeyboard(editChat);
        IntentUtils.setNextActivity(ChatActivity.this, PhotoPickerActivity.class)
            .startActivityForResult(Const.PHOTO_REQUEST_CODE);
      }
    });

    int requestCode = getRequestCode();

    getBigBar().withActivity(this)
        .addMenu(
            ActionType.BACK,
            MenuPosition.LEFT,
            Const.USER_CHATS_IS_EMPTY != requestCode ? MenuState.VISIBLE : MenuState.HIDDEN)
        .addMenu(
            ActionType.EXIT,
            MenuPosition.LEFT,
            Const.USER_CHATS_IS_EMPTY == requestCode ? MenuState.VISIBLE : MenuState.HIDDEN)
        .addText(MenuPosition.LEFT)
        .addMenu(ActionType.REFRESH, MenuPosition.RIGHT, MenuState.HIDDEN)
        .build();

    if (getIntent() != null) {
      chatId = getIntent().getStringExtra("chatId");
      chatCount = getIntent().getIntExtra("chatCount", 0);

      getBigBar().setText(chatCount);
    }

    TimeUtils.OFFSET = TimeZone.getDefault().getRawOffset();
    action = new ChatAction(this);

    setStyle();
  }

  @Override
  public void onBackPressed() {
    if (chatId == null) {
      setResult(Const.CLOSE_USER_CHATS);
    }
    finish();
  }

  private void setStyle() {
    Channel channel = ChannelStore.getChannel();
    if (channel == null) {
      finish();
    } else {
      chatLayout.setVisibility(View.VISIBLE);
      setAdapter(channel.getBackgroundColor(), channel.getTextColor());

      setStatusBarColor(channel.getBackgroundColor());
      getBigBar().setTheme(channel.getBackgroundColor(), channel.getTextColor());
      getBigBar().setTitle(channel.getName());

      fetchUserChat();
    }
  }

  private void setAdapter(int backColor, int textColor) {
    if (adapter == null) {
      adapter = new ChatAdapter(this, backColor, textColor);
      layoutManager = new LinearLayoutManager(this);
      layoutManager.setStackFromEnd(true);
      recyclerView = (RecyclerView) findViewById(R.id.recycler_chat);
      recyclerView.setLayoutManager(layoutManager);
      recyclerView.setAdapter(adapter);
      recyclerView.setItemAnimator(null);
      recyclerView.getRecycledViewPool().setMaxRecycledViews(MessageType.VERIFY.toInt(), 1);
      recyclerView.addOnScrollListener(new InfiniteScrollListener(layoutManager, InfiniteScrollListener.TOP) {
        @Override
        public void refresh() {
          adapter.loadMore(false);
        }

        @Override
        public void firstItem(int index) {
          setAlertState(index);
        }
      });

      BackgroundToucher.makeKeyboardEvent(this, recyclerView, editChat);
    }
  }

  private void fetchUserChat() {
    if (chatId != null) {
      startProcess(ChatState.USER_CHAT_LOADING);
      if (userChat == null) {
        showProgress(ResUtils.getString(this, "ch.loading_information"));
      }
      action.fetchUserChat(chatId);
    } else {
      // exception for hide refresh button
      startProcess(ChatState.USER_CHAT_NOT_LOADED);
      adapter.addWelcomeMessage();
    }
  }

  @Override
  public void userChatFetched(UserChatWrapper userChatWrapper) {
    hideProgress();

    if (ChatState.USER_CHAT_LOADING.equals(chatState)) {
      if (userChatWrapper == null) {
        stopProcess(ChatState.USER_CHAT_NOT_LOADED);
      } else {
        userChat = userChatWrapper.getUserChat();
        if (userChat != null) {
          chatId = userChat.getId();
          adapter.addMetaDatas(userChatWrapper.getManagers(), null);
          adapter.setSession(userChatWrapper.getSession());

          getBigBar().setMenuState(ActionType.EXIT, MenuState.HIDDEN);
          getBigBar().setMenuState(ActionType.BACK, MenuState.VISIBLE);

          if (SocketManager.isReady()) {
            startProcess(ChatState.CHAT_JOINING);
            SocketManager.joinChat(chatId);
          } else {
            startProcess(ChatState.WAITING_SOCKET);
            SocketManager.reconnect();
          }

          send(null);
        }
      }
    }
  }

  private void fetchMessages() {
    startProcess(ChatState.MESSAGES_LOADING);
    adapter.init();
  }

  void messagesFetchComplete(boolean successed, boolean init) {
    if (ChatState.MESSAGES_LOADING.equals(chatState)) {
      if (successed) {
        adapter.enableReceiveSocket();
        startProcess(ChatState.CHAT_READY);
        action.readAll(chatId);

        if (init) {
          scroll(true);
        }
      } else {
        adapter.disableReceiveSocket();
        stopProcess(ChatState.MESSAGES_NOT_LOADED);
      }
    }
  }

  private void refresh() {
    switch (chatState) {
      case USER_CHAT_NOT_LOADED:
        fetchUserChat();
        break;

      case WAITING_SOCKET:
        startProcess(ChatState.WAITING_SOCKET);
        SocketManager.reconnect();
        break;

      case NOT_JOINED:
        startProcess(ChatState.CHAT_JOINING);
        SocketManager.joinChat(chatId);
        break;

      case MESSAGES_NOT_LOADED:
        fetchMessages();
        break;
    }
  }

  public void scroll(boolean force) {
    if (layoutManager.findLastCompletelyVisibleItemPosition() == adapter.getItemCount() - 2 || force) {
      layoutManager.scrollToPosition(adapter.getItemCount() - 1);
    }
  }

  private void sendTextMessage() {
    String message = StringUtils.stripEnd(editChat.getString(), null);
    editChat.setText("");
    if ("".equals(message)) {
      return;
    }
    SendingMessageItem item = new SendingMessageItem(message, false);
    adapter.addSendingMessage(item);

    scroll(true);
    send(Collections.singletonList(item));
  }

  private void sendImageFiles(List<String> imageList) {
    List<SendingMessageItem> items = new ArrayList<>();
    if (imageList != null) {
      for (String p : imageList) {
        SendingMessageItem item = new SendingMessageItem(p, true);
        adapter.addSendingMessage(item);
        items.add(item);
      }
    }
    scroll(true);
    send(items);
  }

  private void send(List<SendingMessageItem> list) {
    if (list != null) {
      for (SendingMessageItem item : list) {
        sendQueue.offer(item);
      }
      L.i("Send: " + chatState + " " + list.size());
    }
    if (ChatState.USER_CHAT_NOT_LOADED.equals(chatState)) {
      if (chatId != null) {
        fetchUserChat();
      } else {
        startProcess(ChatState.USER_CHAT_LOADING);
        showProgress(ResUtils.getString(this, "ch.loading_information"));
        action.createUserChat();
      }
    } else if (!ChatState.USER_CHAT_LOADING.equals(chatState) && !sending && sendQueue.size() > 0) {
      sending = true;

      SendingMessageItem item = sendQueue.poll();
      if (item.isImage()) {
        action.uploadFile(chatId, item);
      } else {
        action.sendMessage(chatId, item);
      }
    }
  }

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    super.onActivityResult(requestCode, resultCode, data);
    switch (requestCode) {
      case Const.PHOTO_REQUEST_CODE:
        if (resultCode == Const.PHOTO_RESULT_CODE) {
          sendImageFiles(data.getStringArrayListExtra(Const.PHOTO_INTENT_KEY));
          break;
        }
    }
  }

  @Override
  public void receiveMessage(Message message) {
    if (message != null) {
      receiveData(message, true);
    }
    sending = false;
    send(null);
  }

  @Override
  public void receiveData(ChannelModel channelModel, boolean upsert) {
    if (chatId != null && channelModel != null) {
      switch (channelModel.getClass().getSimpleName()) {
        case Message.CLASSNAME:
          Message message = (Message) channelModel;
          boolean added = adapter.addMessage(message);

          if (added) {
            scroll(message.isUserMessage());
            action.readAll(chatId);
          }
          break;

        case File.CLASSNAME:
        case WebPage.CLASSNAME:
        case Bot.CLASSNAME:
        case Manager.CLASSNAME:
          adapter.addMetaData(channelModel);
          break;
      }
    }
  }

  @Override
  public void receiveCommand(Command command, Object object) {
    L.i("ReceiveCommand: " + command.name());
    switch (command) {
      case READY:
        if (ChatState.WAITING_SOCKET.equals(chatState)) {
          startProcess(ChatState.CHAT_JOINING);
          SocketManager.joinChat(chatId);
        } else {
          refresh();
        }
        break;

      case JOINED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && ChatState.CHAT_JOINING.equals(chatState)) {
          fetchMessages();
        }
        break;

      case LEAVED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && ChatState.CHAT_READY.equals(chatState)) {
          stopProcess(ChatState.NOT_JOINED);
          if (adapter != null) {
            adapter.disableReceiveSocket();
          }
        }
        break;

      case SOCKET_DISCONNECTED:
        switch (chatState) {
          case USER_CHAT_LOADING:
          case WAITING_SOCKET:
          case MESSAGES_NOT_LOADED:
          case MESSAGES_LOADING:
          case NOT_JOINED:
          case CHAT_JOINING:
          case CHAT_READY:
            stopProcess(ChatState.USER_CHAT_NOT_LOADED);
            if (adapter != null) {
              adapter.disableReceiveSocket();
            }
            break;
        }
        break;
    }
  }

  @Override
  public void receiveChatCounter(int count) {
    getBigBar().setText(count);
  }

  @Override
  public void optionClicked(ActionType actionType) {
    super.optionClicked(actionType);
    switch (actionType) {
      case REFRESH:
        refresh();
        break;
    }
  }

  void setAlertState(int firstIndex) {
    if (firstIndex == -1) {
      firstIndex = layoutManager.findFirstVisibleItemPosition();
    }
    toast.setVisibility(UIUtils.getVisible(adapter.setNewMessageAlertState(firstIndex), false));
  }

  private void startProcess(ChatState chatState) {
    this.chatState = chatState;
    getBigBar().setMenuState(ActionType.REFRESH, MenuState.HIDDEN);
  }

  private void stopProcess(ChatState chatState) {
    this.chatState = chatState;
    getBigBar().setMenuState(ActionType.REFRESH, MenuState.VISIBLE);
  }

  @Override
  protected void onDestroy() {
    if (ChatState.CHAT_READY.equals(chatState) && chatId != null) {
      SocketManager.leaveChat(chatId);
    }
    super.onDestroy();
  }

  public String getChatId() {
    return chatId;
  }

  @Override
  public void clicked(Message message) {
    if (message.getFile() != null) {
      if (message.getFile().getPreviewThumb() == null || !message.getFile().isImage()) {
        IntentUtils.setNextActivity(this, DownloadActivity.class)
            .putExtra("url", message.getFile().getUrl())
            .putExtra("filename", message.getFile().getFilename())
            .putExtra("image", message.getFile().isImage())
            .startActivity();
      } else {
        IntentUtils.setNextActivity(this, PhotoAlbumActivity.class)
            .putExtra("url", message.getFile().getUrl())
            .putExtra("filename", message.getFile().getFilename())
            .putExtra("thumbnailUrl", message.getFile().getPreviewThumb().getUrl())
            .putExtra("size", message.getFile().getSize())
            .startActivity();
      }
    } else if (message.getWebPage() != null) {
      urlClicked(message.getWebPage().getUrl());
    }
  }

  @Override
  public void actionClicked(ActionMessageItem item) {
    if (item != null && item.getActionType() != null) {
      switch (item.getActionType()) {
        case REQUEST_WORKING_TIME:
          item.setActionType(null);
          adapter.addActionMessage(item);

          QuestionMessageItem questionItem = new QuestionMessageItem(
              adapter.getMax() + 2,
              ResUtils.getString(this, "ch.out_of_work.user_answer"));
          adapter.addQuestionMessage(questionItem);

          scroll(true);

          String workingTime = ChannelStore.getChannel() != null
              ? ChannelStore.getChannel().getWorkingTime(this)
              : "?";
          addMessageDelay(new ActionMessageItem(null, null, workingTime));
          break;
      }
    }
  }

  @Override
  public void urlClicked(String url) {
    if (url != null) {
      IntentUtils.setUrl(this, url).startActivity();
    }
  }

  private void addMessageDelay(final ActionMessageItem item) {
    final Handler handler = new Handler();
    handler.postDelayed(new Runnable() {
      @Override
      public void run() {
        item.setTimestamp(adapter.getMax() + 3);
        adapter.addActionMessage(item);
        scroll(true);
      }
    }, 750);
  }
}