package com.zoyi.channel.plugin.android.activity.photo_picker;

import android.content.Context;
import android.database.Cursor;
import android.os.AsyncTask;
import android.provider.MediaStore;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.ViewGroup;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.wrapper.MarginBox;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Created by mika on 8/31/16.
 */
class PhotoPickerAdapter
    extends RecyclerView.Adapter<PhotoHolder>
    implements PhotoItemEventListener {
  private BaseActivity activity;
  private int layoutId;
  private boolean singleMode = false;
  private List<PhotoItem> photoItems;
  private List<PhotoItem> tempItems;
  private List<PhotoItem> selectedItems;
  private PhotoProcessListener listener;
  private int size = 1;
  private boolean loading = false;

  PhotoPickerAdapter(
      BaseActivity activity,
      int layoutId,
      PhotoProcessListener listener) {
    this.activity = activity;
    this.layoutId = layoutId;
    this.photoItems = new ArrayList<>();
    this.listener = listener;
    this.selectedItems = new ArrayList<>();
  }

  @Override
  public PhotoHolder onCreateViewHolder(ViewGroup parent, int viewType) {
    return new PhotoHolder(
        LayoutInflater.from(parent.getContext()).inflate(layoutId, parent, false),
        singleMode,
        activity.getBackgroundColor(),
        activity.getTextColor(),
        viewType,
        this);
  }

  @Override
  public void onBindViewHolder(PhotoHolder holder, int position) {
    MarginBox marginBox = listener.getMargin(position);
    holder.setValue(
        photoItems.get(position),
        getSelectedIndex(photoItems.get(position)) + 1,
        marginBox);
  }

  @Override
  public int getItemViewType(int position) {
    return size;
  }

  @Override
  public int getItemCount() {
    return photoItems.size();
  }

  private int getSelectedIndex(PhotoItem photoItem) {
    if (photoItem == null) {
      return -1;
    }
    for (int i = 0; i < selectedItems.size(); i++) {
      PhotoItem item = selectedItems.get(i);
      if (item.getId() == photoItem.getId()) {
        return i;
      }
    }
    return -1;
  }

  void setImageSize(int size) {
    this.size = size;
    notifyDataSetChanged();
  }

  public void clear() {
    photoItems.clear();
    notifyDataSetChanged();
  }

  public void addAll(List<PhotoItem> photoItems) {
    int startIndex = this.photoItems.size();
    this.photoItems.addAll(startIndex, photoItems);
    notifyItemRangeInserted(startIndex, photoItems.size());
  }

  ArrayList<String> getSelectedNames() {
    ArrayList<String> list = new ArrayList<>();
    for (PhotoItem item : selectedItems) {
      list.add(item.getPath());
    }
    return list;
  }

  @Override
  public void clicked(String path) {
    if (listener != null) {
      listener.onePhotoSelected(path);
    }
  }

  @Override
  public int requestCheck(int position, boolean state) {
    PhotoItem item = photoItems.get(position);
    int index = getSelectedIndex(item);
    if (state) {
      if (index < 0) {
        if (selectedItems.size() < Const.PHOTO_SELECT_LIMIT) {
          selectedItems.add(item);
          if (listener != null) {
            listener.photoSelected(selectedItems.size());
          }
          return selectedItems.size();
        } else {
//          UIUtils.showToast(App.getResString("photo.limit_alert"));
        }
      }
    } else {
      if (index >= 0) {
        selectedItems.remove(index);
        if (listener != null) {
          listener.photoSelected(selectedItems.size());
        }
        for (int i = index; i < selectedItems.size(); i++) {
          notifyItemChanged(selectedItems.get(i).getIndex());
        }
        return 0;
      }
    }
    return -1;
  }

  public void resetSelection() {
    selectedItems.clear();
  }

  void startFetchPhoto() {
    if (loading) {
      return;
    }
    new ImageTask().execute(activity);
  }

  private void fetchCallback(boolean flag) {
    if (flag) {
      Set<Long> set = new HashSet<>();
      for (PhotoItem item : tempItems) {
        set.add(item.getId());
      }

      for (int i = selectedItems.size() - 1; i >= 0; i--) {
        if (!set.contains(selectedItems.get(i).getId())) {
          selectedItems.remove(i);
        }
      }

      photoItems = tempItems;
      notifyDataSetChanged();

      if (listener != null) {
        listener.photoFetchSucceeded(photoItems.size());
        listener.photoSelected(selectedItems.size());
      }
    } else if (listener != null) {
      listener.photoFetchFailed();
    }
  }

  private class ImageTask extends AsyncTask<Context, Void, Boolean> {
    @Override
    protected Boolean doInBackground(Context... params) {
      if (params.length != 1) {
        return false;
      }

      final String[] projection = new String[] {
          MediaStore.Images.Media._ID,
          MediaStore.Images.Media.DISPLAY_NAME,
          MediaStore.Images.Media.DATA,
          MediaStore.Images.Media.BUCKET_DISPLAY_NAME
      };

      Context context = params[0];
      Cursor cursor = context.getContentResolver().query(
          MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
          projection,
          null,
          null,
          MediaStore.Images.Media.DATE_ADDED);

      if (cursor == null) {
        return false;
      }

      tempItems = new ArrayList<>();
      int index = 0;

      if (cursor.moveToLast()) {
        do {
          long id = cursor.getLong(cursor.getColumnIndex(projection[0]));
          String name = cursor.getString(cursor.getColumnIndex(projection[1]));
          String path = cursor.getString(cursor.getColumnIndex(projection[2]));

          File file = new File(path);
          if (file.exists()) {
            tempItems.add(new PhotoItem(id, name, path, index++));
          }
        } while (cursor.moveToPrevious());
      }
      cursor.close();
      return true;
    }

    @Override
    protected void onPostExecute(Boolean flag) {
      super.onPostExecute(flag);
      fetchCallback(flag != null ? flag : false);
      loading = false;
    }
  }
}
