package com.zoyi.channel.plugin.android.activity.chat;

import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.ViewGroup;
import com.zoyi.channel.plugin.android.ChannelPlugin;
import com.zoyi.channel.plugin.android.ChannelStore;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.activity.chat.listener.MessageCallback;
import com.zoyi.channel.plugin.android.activity.chat.listener.UserInfoListener;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.ActionType;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.chat.type.UserInfoType;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.*;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.MessagesWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.UserVeilWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.*;

/**
 * Created by mika on 2016. 12. 8..
 */
public class ChatAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder> implements UserInfoListener {
  private ChatActivity activity;
  private ChatDataDictionary dictionary;
  private Session session;
  private ChatAction action;

  private boolean init, loading, ready, saveMessage;
  private int fetchCounter = 0;
  private String next = Const.MESSAGE_ID_MAX;
  private Set<String> messageSet;

  private SortedList<MessageItem> items;
  private int userChatBackColor, userChatTextColor;

  private boolean isAutoMessageExists = false;
  private Long newMessageTimestamp = null;
  private boolean newMessageDismiss = true;
  private long minTimestamp = Long.MAX_VALUE, maxTimestamp = Long.MIN_VALUE;
  private String max = Const.MESSAGE_ID_MIN;
  private List<Message> tempQueue;

  private CountryDictionary countryDictionary;
  private UserInfoItem userInfoItem;
  private boolean receiveFirstMessage = false;

  public ChatAdapter(ChatActivity activity, int userChatBackColor, int userChatTextColor) {
    this.activity = activity;
    this.action = new ChatAction(null);
    this.userChatBackColor = userChatBackColor;
    this.userChatTextColor = userChatTextColor;
    tempQueue = new ArrayList<>();
    messageSet = new HashSet<>();

    dictionary = new ChatDataDictionary();
    items = new SortedList<>(MessageItem.class, new SortedListCallback(this, false));

    countryDictionary = new CountryDictionary(activity);
  }

  public void init() {
    if (!ready && activity.getChatId() != null) {
      fetchCounter++;
      this.saveMessage = true;
      tempQueue.clear();

      if (init && CompareUtils.compare(max, Const.MESSAGE_ID_MIN) > 0) {
        fetchContinuous(max);
      } else {
        action.fetchMessages(
            activity.getChatId(),
            Const.MESSAGE_ID_MAX,
            Const.DESC,
            fetchCounter,
            new MessageCallback() {
          @Override
          public void messagesFetched(int counter, MessagesWrapper wrapper) {
            if (counter == fetchCounter) {
              if (wrapper == null) {
                activity.messagesFetchComplete(false);
              } else {
                addMetaDatas(wrapper.getManagers(), wrapper.getBots());
                addMessages(wrapper.getNext(), wrapper.getMessages());
                addWorkingTimeMessage();
                activity.setAlertState(-1);
                activity.messagesFetchComplete(true);

                init = true;
              }
            }
          }
        });
      }
    }
  }

  private void fetchContinuous(String since) {
    if (activity.getChatId() != null) {
      action.fetchMessages(
          activity.getChatId(),
          since,
          Const.ASC,
          fetchCounter,
          new MessageCallback() {
            @Override
            public void messagesFetched(int counter, MessagesWrapper wrapper) {
              if (counter == fetchCounter) {
                if (wrapper == null) {
                  activity.messagesFetchComplete(false);
                } else {
                  addMetaDatas(wrapper.getManagers(), wrapper.getBots());
                  addMessages(wrapper.getNext(), wrapper.getMessages());

                  if (wrapper.getNext() == null) {
                    activity.setAlertState(-1);
                    activity.messagesFetchComplete(true);
                    init = true;
                  } else {
                    fetchContinuous(wrapper.getNext());
                  }
                }
              }
            }
          });
    }
  }

  public void enableReceiveSocket() {
    this.ready = true;
    L.i("tempQueue");
    addMessages(tempQueue);
    tempQueue.clear();
  }

  public void disableReceiveSocket() {
    this.ready = false;
    this.saveMessage = false;
    this.fetchCounter++;
    tempQueue.clear();
  }

  public void addMetaDatas(List<Manager> managers, List<Bot> bots) {
    dictionary.add(managers);
    dictionary.add(bots);
  }

  public void addMetaData(ChannelModel model) {
    dictionary.add(model);
  }

  public void loadMore(boolean force) {
    if (this.init && this.next != null && (!this.loading || force) && activity.getChatId() != null) {
      loading = true;
      action.fetchMessages(
          activity.getChatId(),
          next,
          Const.DESC,
          fetchCounter,
          new MessageCallback() {
            @Override
            public void messagesFetched(int counter, MessagesWrapper wrapper) {
              if (fetchCounter == counter) {
                if (wrapper == null) {
                  // show error
                } else {
                  addMetaDatas(wrapper.getManagers(), wrapper.getBots());
                  addMessages(wrapper.getNext(), wrapper.getMessages());
                  activity.setAlertState(-1);
                  loading = false;
                }
              }
            }
          });
    }
  }

  public boolean addMessage(Message message) {
    if (!receiveFirstMessage) {
      receiveFirstMessage = true;
      checkRequireUserInfo(message.getCreatedAt() + 1);
    }

    if (ready) {
      return addMessages(Collections.singletonList(message));
    } else if (saveMessage) {
      tempQueue.add(message);
    }

    return false;
  }

  public boolean addMessages(List<Message> messages) {
    if (activity.getChatId() == null) {
      return false;
    }

    List<MessageItem> items = new ArrayList<>();

    int addCount = 0;

    for (Message message : messages) {
      if (message != null && message.getLog() == null) {
        items.add(new ChatMessageItem(message));
        items.add(new DateItem(message.getCreatedAt()));

        if (!"Manager".equals(message.getPersonType()) && message.getRequestId() != null) {
          SendingMessageItem item = SendingMessageItem.create(
              message.getRequestId(),
              message.getChannelId(),
              message.getChatId());
          if (item != null) {
            this.items.remove(item);
          }
        }

        if (!messageSet.contains(message.getId())) {
          addCount++;
          messageSet.add(message.getId());
        }

        if (minTimestamp > message.getCreatedAt()) {
          minTimestamp = message.getCreatedAt();
        }
        if (maxTimestamp < message.getCreatedAt()) {
          maxTimestamp = message.getCreatedAt();
          max = message.getId();
        }
      }
    }

    this.items.addAll(items);
    if (!newMessageDismiss
        && newMessageTimestamp != null
        && minTimestamp < newMessageTimestamp
        && newMessageTimestamp < maxTimestamp) {
      this.items.add(new NewMessageItem(newMessageTimestamp));
    }
    if (newMessageTimestamp != null && newMessageTimestamp >= maxTimestamp) {
      newMessageDismiss = true;
    }

    return addCount > 0;
  }

  public void addMessages(String next, List<Message> messages) {
    this.next = next;
    addMessages(messages);
  }

  public void setNewMessageDismiss() {
    this.newMessageDismiss = true;
  }

  public void setSession(Session session) {
    this.session = session;
    if (this.newMessageTimestamp != null) {
      this.items.remove(new NewMessageItem(newMessageTimestamp));
    }
    this.newMessageTimestamp = session.getLastReadAt();
    this.newMessageDismiss = false;
  }

  public boolean setNewMessageAlertState(int firstIndex) {
    if (firstIndex < 0 || firstIndex >= items.size() || newMessageDismiss) {
      return false;
    }

    Long first = items.get(firstIndex).getFirst();
    if (first != null) {
      if (first <= newMessageTimestamp) {
        newMessageDismiss = true;
        return false;
      } else {
        return true;
      }
    }
    return false;
  }

  private MessageItem get(int position) {
    if (position < 0 || position >= items.size()) {
      return null;
    }
    return items.get(position);
  }

  @Override
  public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
    LayoutInflater i = LayoutInflater.from(parent.getContext());
    MessageType type = MessageType.fromId(viewType);

    switch (type) {
      case ACTION_MESSAGE:
        return new ActionMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_manager, parent, false),
            activity,
            dictionary);

      case QUESTION_MESSAGE:
        return new QuestionMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user, parent, false),
            activity,
            dictionary,
            userChatBackColor,
            userChatTextColor);

      case MANAGER_MESSSAGE:
        return new ManagerMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_manager, parent, false),
            activity,
            dictionary);

      case SENDING:
      case USER_MESSAGE:
        return new UserMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user, parent, false),
            activity,
            dictionary,
            userChatBackColor,
            userChatTextColor);

      case DATE:
        return new DateDividerHolder(i.inflate(R.layout.ch_plugin_item_message_date, parent, false));

      case NEW_MESSAGE_DIVIDER:
        return new NewMessageholder(i.inflate(R.layout.ch_plugin_item_message_unread_divider, parent, false));

      case USER_INFO:
        return new UserInfoMessageHolder(
            i.inflate(R.layout.ch_plugin_item_message_user_info, parent, false),
            countryDictionary,
            this);

      default:
        return null;
    }
  }

  @Override
  public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {
    MessageItem item = items.get(position);
    ChatMessageItem chatMessageItem;

    switch (item.getType()) {
      case ACTION_MESSAGE:
        ActionMessageItem actionMessageItem = (ActionMessageItem) item;

        ActionMessageHolder actionMessageHolder = (ActionMessageHolder) holder;
        actionMessageHolder.set(actionMessageItem);
        break;

      case QUESTION_MESSAGE:
        QuestionMessageItem questionMessageItem = (QuestionMessageItem) item;

        QuestionMessageHolder questionMessageHolder = (QuestionMessageHolder) holder;
        questionMessageHolder.set(questionMessageItem);
        break;

      case MANAGER_MESSSAGE:
        chatMessageItem = (ChatMessageItem) item;

        ManagerMessageHolder managerMessageHolder = (ManagerMessageHolder) holder;
        managerMessageHolder.set(chatMessageItem, chatMessageItem.isConnected(get(position - 1)));
        break;

      case USER_MESSAGE:
        chatMessageItem = (ChatMessageItem) item;

        UserMessageHolder userMessageHolder = (UserMessageHolder) holder;
        userMessageHolder.set(
            chatMessageItem.getMessage(),
            chatMessageItem.isConnected(get(position - 1)));
        break;

      case SENDING:
        SendingMessageItem sendingMessageItem = (SendingMessageItem) item;
        UserMessageHolder sendingHolder = (UserMessageHolder) holder;
        sendingHolder.set(sendingMessageItem);
        break;

      case DATE:
        DateDividerHolder dateDividerHolder = (DateDividerHolder) holder;
        dateDividerHolder.set((DateItem) item);
        break;

      case USER_INFO:
        UserInfoMessageHolder userInfoMessageHolder = (UserInfoMessageHolder) holder;
        userInfoMessageHolder.set(userInfoItem);
    }
  }

  @Override
  public int getItemViewType(int position) {
    return items.get(position).getType().toInt();
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  public Long getMax() {
    return maxTimestamp;
  }

  public void addSendingMessage(SendingMessageItem item) {
    items.add(item);
    maxTimestamp = Math.max(maxTimestamp, item.getCreatedAt());
  }

  public void addActionMessage(ActionMessageItem item) {
    items.add(item);
    maxTimestamp = Math.max(maxTimestamp, item.getTimestamp());
  }

  public void addQuestionMessage(QuestionMessageItem item) {
    items.add(item);
    maxTimestamp = Math.max(maxTimestamp, item.getTimestamp());
  }

  public void addWorkingTimeMessage() {
    if (isAutoMessageExists) {
      return;
    }
    if (!ChannelStore.isWorking()) {
      String message = ChannelStore.getScript("out_of_work", "ch.scripts.out_of_work.default");
      if (message != null) {
        isAutoMessageExists = true;
        items.add(new ActionMessageItem(maxTimestamp + 1, ActionType.REQUEST_WORKING_TIME, message));
      }
    }
  }

  public void addWelcomeMessage() {
    if (isAutoMessageExists) {
      return;
    }
    if (!ChannelStore.isWorking()) {
      addWorkingTimeMessage();
    } else {
      String message;
      String userName = ChannelStore.getUserName();

      if (userName != null) {
        message = ChannelStore.getScript("welcome_user", "ch.scripts.welcome_user.default");
        if (message != null) {
          message = message.replace("${user}", userName);
        }
      } else {
        message = ChannelStore.getScript("welcome_veil", "ch.scripts.welcome_veil.default");
      }

      if (message != null) {
        items.add(new ActionMessageItem(maxTimestamp + 1, null, message));
      }
    }
  }

  @Override
  public void userInfoChanged(Object object) {
    if (userInfoItem != null) {
      switch (userInfoItem.getUserInfoType()) {
        case NAME:
          if (object instanceof String) {
            userInfoItem.setObject(object);
          }
          break;

        case MOBILE_NUMBER:
          if (object instanceof MobileNumberItem) {
            userInfoItem.setObject(object);
          }
          break;
      }
    }
  }

  private void checkRequireUserInfo(long timestamp) {
    if (userInfoItem != null) {
      try {
        items.remove(userInfoItem);
        userInfoItem = null;
      } catch (Exception ex) {
      }
    }

    if (!ChannelStore.hasName()) {
      userInfoItem = new UserInfoItem(UserInfoType.NAME, timestamp, false, null);
    } else if (ChannelStore.getMobileNumber() == null) {
      userInfoItem = new UserInfoItem(
          UserInfoType.MOBILE_NUMBER,
          timestamp,
          false,
          new MobileNumberItem(countryDictionary.getDefaultCountryCode(), ""));
    }

    if (userInfoItem != null) {
      items.add(userInfoItem);
    }
  }

  @Override
  public void apply() {
    if (userInfoItem != null && userInfoItem.getObject() != null) {
      RequestUtils requestUtils = null;
      switch (userInfoItem.getUserInfoType()) {
        case NAME:
          if (userInfoItem.getObject() instanceof String) {
            requestUtils = RequestUtils.form().set("name", (String) userInfoItem.getObject());
          }
          break;

        case MOBILE_NUMBER:
          if (userInfoItem.getObject() instanceof MobileNumberItem) {
            MobileNumberItem item = (MobileNumberItem) userInfoItem.getObject();
            requestUtils = RequestUtils.form().set(
                "mobileNumber",
                String.format("+%s%s", item.getCountry(), item.getMobileNumber()));
          }
          break;

        default:
          return;
      }

      if (requestUtils != null) {
        ChannelPlugin.getService().updateGuest(requestUtils.create())
            .subscribeOn(Schedulers.newThread())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(new RestSubscriber<UserVeilWrapper>() {
              @Override
              public void onError(Throwable e) {
                L.e(e.getMessage());

                userInfoItem.setError(true);
                items.add(userInfoItem);
              }

              @Override
              public void onNext(UserVeilWrapper wrapper) {
                ChannelStore.setUserVeil(wrapper.getUser(), wrapper.getVeil());

                items.remove(userInfoItem);
                checkRequireUserInfo(userInfoItem.getTimestamp());
              }
            });
      }
    }
  }
}
