package com.zoyi.channel.plugin.android.socket;

import android.os.Handler;
import com.zoyi.com.google.gson.Gson;
import com.zoyi.channel.plugin.android.ChannelPlugin;
import com.zoyi.channel.plugin.android.ChannelStore;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.event.ChannelModelBus;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.PushBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.Utils;
import com.zoyi.channel.plugin.android.util.lang.StringUtils;
import com.zoyi.io.socket.client.IO;
import com.zoyi.io.socket.client.Socket;
import com.zoyi.io.socket.client.SocketIOException;
import com.zoyi.io.socket.emitter.Emitter;
import com.zoyi.io.socket.engineio.client.EngineIOException;
import com.zoyi.io.socket.engineio.client.transports.WebSocket;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.URISyntaxException;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Created by mika on 2016. 4. 8..
 */
public class SocketManager {
  private static SocketManager socketManager;

  private Socket socket;
  private Gson gson;
  private Timer heartbeatTimer;
  private Handler reconnectHandler;
  private AtomicBoolean forceDisconnect;
  private AtomicBoolean ready;
  private BlockingQueue<Integer> reconnectQueue;
  private int[] attemptDelay = {100, 3000, 4000, 5000, 7000, 8000, 10000};
  private AtomicInteger attemptCount = new AtomicInteger();

  private static final int HEARTBEAT_INTERVAL = 30000;
  private static final int RECONNECT_POP_INTERVAL = 500;
  private static final String CHAT_SERVER_URL_PRODUCTION = "https://ws.channel.io/app";

  private String channelId;

  public static void init() {
    socketManager = new SocketManager();
    socketManager.gson = new Gson();
    socketManager.forceDisconnect = new AtomicBoolean(false);
    socketManager.ready = new AtomicBoolean(false);
    socketManager.reconnectQueue = new ArrayBlockingQueue<>(1);
    socketManager.start();
  }

  private void start() {
    try {
      IO.Options options = new IO.Options();
      options.reconnection = false;
      options.transports = new String[] { WebSocket.NAME };
      socket = IO.socket(CHAT_SERVER_URL_PRODUCTION, options);
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }

    socket.on(Socket.EVENT_CONNECT, onConnect);                     // void
    socket.on(Socket.EVENT_CONNECT_ERROR, onConnectError);          // EngineIOException
    socket.on(Socket.EVENT_CONNECT_TIMEOUT, onConnectTimeout);
    socket.on(Socket.EVENT_CONNECTING, onConnecting);               // void
    socket.on(Socket.EVENT_DISCONNECT, onDisconnect);                // compare
    socket.on(Socket.EVENT_ERROR, onError);                         // EngineIOException
    //socket.on(Socket.EVENT_MESSAGE, onMessage);
    socket.on(Socket.EVENT_PING, onPing);                           // void
    socket.on(Socket.EVENT_PONG, onPong);                           // long
    socket.on(Socket.EVENT_RECONNECT, onReconnect);                 // int
    socket.on(Socket.EVENT_RECONNECT_ATTEMPT, onReconnectAttempt);  // int
    socket.on(Socket.EVENT_RECONNECT_ERROR, onReconnectError);      // SocketIOException
    socket.on(Socket.EVENT_RECONNECT_FAILED, onReconnectFailed);
    socket.on(Socket.EVENT_RECONNECTING, onReconnecting);           // int

    socket.on(SocketEvent.AUTHENTICATED, onAuthenticated);          // boolean
    socket.on(SocketEvent.READY, onReady);
    socket.on(SocketEvent.CREATE, onCreate);                        // JSONObject
    socket.on(SocketEvent.DELETE, onDelete);                        // JSONObject
    socket.on(SocketEvent.JOINED, onJoined);                        // JSONObject
    socket.on(SocketEvent.LEAVED, onLeaved);                        // JSONObject
    socket.on(SocketEvent.PUSH, onPush);                          // JSONObject
    //socket.on(SocketEvent.RECONNECT_ATTEMPT, onReconnectAttempt2);
    socket.on(SocketEvent.UNAUTHORIZED, onUnauthorized);            // JSONObject
    socket.on(SocketEvent.UPDATE, onUpdate);                        // JSONObject
  }

  public static void setChannelId(String channelId) {
    if (socketManager != null) {
      socketManager.channelId = channelId;
    }
  }

  // static functions

  public static boolean isReady() {
    if (socketManager != null) {
      return socketManager.ready.get();
    }
    return false;
  }

  public static void connect() {
    if (socketManager != null) {
      socketManager.connectSocket();
    }
  }

  public static void reconnect() {
    if (socketManager != null) {
      socketManager.enqueueReconnect();
    }
  }

  public static void joinChat(String chatId) {
    if (socketManager != null) {
      socketManager.chatAction(SocketEvent.ACTION_JOIN, chatId);
    }
  }

  public static void leaveChat(String chatId) {
    if (socketManager != null) {
      socketManager.chatAction(SocketEvent.ACTION_LEAVE, chatId);
    }
  }

  public static void disconnect() {
    if (socketManager != null) {
      socketManager.disconnect(true);
      socketManager.setHeartbeat(false);
    }
  }

  public static void destroy() {
    if (socketManager != null) {
      socketManager.setReconnectConsumer(false);
      socketManager.socket.off();
      socketManager.socket.disconnect();

      socketManager.channelId = null;
      socketManager.forceDisconnect = null;
      socketManager.ready = null;
      socketManager.reconnectQueue = null;
      socketManager.socket = null;
    }
  }

  // internal functions

  private void enqueueReconnect() {
    if (!ready.get()) {
      try {
        reconnectQueue.add(1);
      } catch (Exception ignored) {
      }
    }
  }

  private void clearReconnectQueue() {
    try {
      reconnectQueue.clear();
    } catch (Exception ignored) {
    }
  }

  private void connectSocket() {
    L.d("try connect");
    if (socket != null && !socket.connected() && channelId != null) {
      socket.connect();
      setReconnectConsumer(true);
    }
  }

  private void disconnect(boolean force) {
    forceDisconnect.set(force);

    try {
      clearReconnectQueue();
      socket.disconnect();
    } catch (Exception ex) {
    }

    if (!force) {
      enqueueReconnect();
    } else {
      setReconnectConsumer(false);
    }
  }

  private void authentication() {
    String personType = null;
    String personId = null;

    if (ChannelStore.getUserId() != null) {
      personType = "User";
      personId = ChannelStore.getUserId();
    } else if (ChannelStore.getVeilId() != null) {
      personType = "Veil";
      personId = ChannelStore.getVeilId();
    }

    if (personType != null && personId != null && channelId != null) {
      String info = String.format("{\n" +
          "  type: \"Plugin\",\n" +
          "  channelId: \"%s\",\n" +
          "  guestType: \"%s\",\n" +
          "  guestId: \"%s\"" +
          "}", channelId, personType, personId);
      try {
        JSONObject jsonObject = new JSONObject(info);
        emit(SocketEvent.ACTION_AUTHENTICATION, jsonObject);
      } catch (JSONException e) {
        e.printStackTrace();
        RxBus.post(new CommandBus(Command.UNAUTHORIZED));
      }
    } else {
      disconnect(true);
    }
  }

  private Runnable reconnectRunnable = new Runnable() {
    @Override
    public void run() {
      try {
        Integer data = reconnectQueue.peek();
        if (data != null) {
          L.i("Try reconnect");
          int index = Math.min(attemptCount.getAndIncrement(), attemptDelay.length - 1);
          Thread.sleep(attemptDelay[index]);
          reconnectQueue.remove();
          connectSocket();
        }
      } catch (Exception e) {
        L.e(e.getMessage());
      }
      if (reconnectHandler != null) {
        reconnectHandler.postDelayed(reconnectRunnable, RECONNECT_POP_INTERVAL);
      }
    }
  };

  private void setReconnectConsumer(boolean flag) {
    if (flag) {
      if (reconnectHandler == null) {
        reconnectHandler = new Handler();
        reconnectHandler.post(reconnectRunnable);
      }
    } else {
      if (reconnectHandler != null) {
        try {
          reconnectHandler.removeCallbacks(reconnectRunnable);
          reconnectHandler = null;
        } catch (Exception ex) {
        }
      }
    }
  }

  private void setHeartbeat(boolean flag) {
    if (heartbeatTimer != null) {
      try {
        heartbeatTimer.cancel();
      } catch (Exception ignored) {
      } finally {
        heartbeatTimer = null;
      }
    }

    if (flag) {
      heartbeatTimer = new Timer();
      heartbeatTimer.schedule(new TimerTask() {
        @Override
        public void run() {
          try {
            L.i("Heartbeat");
            socket.emit(SocketEvent.ACTION_HEARTBEAT, "");
          } catch (Exception ex) {
            L.e("socket error");
          }
        }
      }, HEARTBEAT_INTERVAL, HEARTBEAT_INTERVAL);
    }
  }

  private void emit(String event, Object object) {
    if (object == null || socket == null) {
      return;
    }
    socket.emit(event, object);
  }

  private void chatAction(String action, String chatId) {
    if (socket == null) {
      return;
    }
    if (chatId == null || !ready.get()) {
      return;
    }
    String message = String.format("/user_chats/%s", chatId);
    emit(action, message);
  }

  // Events

  private Emitter.Listener onConnect = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onConnect: ");
      try {
        forceDisconnect.set(false);
        attemptCount.set(0);
        authentication();
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
      RxBus.post(new CommandBus(Command.SOCKET_CONNECTED));
    }
  };

  private Emitter.Listener onConnectTimeout = new Emitter.Listener() {
    @Override
    public void call(Object... objects){
      L.d("onConnectTimeout");
    }
  };

  private Emitter.Listener onConnectError = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      ready.set(false);
      enqueueReconnect();
      RxBus.post(new CommandBus(Command.SOCKET_DISCONNECTED, forceDisconnect.get()));

      try {
        String message = "";
        if (objects[0] instanceof EngineIOException) {
          EngineIOException exception = (EngineIOException) objects[0];
          message = exception.getMessage();
        }
        if (objects[0] instanceof SocketIOException) {
          SocketIOException exception = (SocketIOException) objects[0];
          message = exception.getMessage();
        }
        L.e("onConnectError: " + message);
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
    }
  };

  private Emitter.Listener onConnecting = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onConnecting");
//      setSocketStatus(SocketStatus.CONNECTING);
    }
  };

  private Emitter.Listener onDisconnect = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onDisconnect: " + objects[0] + " " + forceDisconnect.get());
      ready.set(false);
      setHeartbeat(false);
      RxBus.post(new CommandBus(Command.SOCKET_DISCONNECTED, forceDisconnect.get()));

      if (!forceDisconnect.get()) {
        enqueueReconnect();
      }
    }
  };

  private Emitter.Listener onError = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      EngineIOException exception = (EngineIOException) objects[0];
      L.e("onError: " + objects.length + " " + exception.getMessage());
      ready.set(false);
      enqueueReconnect();
    }
  };

  private Emitter.Listener onPing = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      //L.d("onPing");
    }
  };

  private Emitter.Listener onPong = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      //L.d("onPong: " + (Long)objects[0]);
    }
  };

  private Emitter.Listener onReconnect = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onReconnect: " + (int) objects[0]);
    }
  };

  private Emitter.Listener onReconnectAttempt = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onReconnectAttempt: " + (int) objects[0]);
    }
  };

  private Emitter.Listener onReconnectError = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      SocketIOException exception = (SocketIOException) objects[0];
      L.e("onReconnectError: " + exception.getMessage());
      enqueueReconnect();
      ready.set(false);
      RxBus.post(new CommandBus(Command.SOCKET_DISCONNECTED, forceDisconnect.get()));
    }
  };

  private Emitter.Listener onReconnectFailed = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.e("onReconnectFailed");
      enqueueReconnect();
      ready.set(false);
      RxBus.post(new CommandBus(Command.SOCKET_DISCONNECTED, forceDisconnect.get()));
    }
  };

  private Emitter.Listener onReconnecting = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onReconnecting: " + (int) objects[0]);
      RxBus.post(new CommandBus(Command.SOCKET_RECONNECTING));
    }
  };

  private Emitter.Listener onAuthenticated = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onAuthenticated: " + objects[0]);
    }
  };

  private Emitter.Listener onReady = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("Ready");
      ready.set(true);
      setHeartbeat(true);
//      setHeartbeat(true);
      clearReconnectQueue();
      RxBus.post(new CommandBus(Command.READY));
    }
  };

  private Emitter.Listener onJoined = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onJoined: " + objects[0]);
      try {
        String[] split = StringUtils.split((String) objects[0], '/');
        RxBus.post(new CommandBus(Command.JOINED, split[1]));
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
    }
  };

  private Emitter.Listener onLeaved = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onLeaved: " + objects[0]);
      try {
        String[] split = StringUtils.split((String) objects[0], '/');
        RxBus.post(new CommandBus(Command.LEAVED, split[1]));
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
    }
  };

  private Emitter.Listener onUnauthorized = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.e("onUnauthorized: " + objects[0].toString());
      ready.set(false);
      forceDisconnect.set(true);
      socket.disconnect();
      RxBus.post(new CommandBus(Command.UNAUTHORIZED));
    }
  };

  private Emitter.Listener onPush = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onPush: " + objects[0].toString());

      try {
        JSONObject json = (JSONObject) objects[0];
        UserChat userChat = null;
        ProfileEntity person = null;

        String type = json.getString("type");

        if (json.has("refers")) {
          JSONObject refers = json.getJSONObject("refers");
          if (refers.has("bot")) {
            person = parseJson(refers.getString("bot"), Bot.class);
          }
          if (refers.has("manager")) {
            person = parseJson(refers.getString("manager"), Manager.class);
          }
          if (refers.has("userChat")) {
            userChat = parseJson(refers.getString("userChat"), UserChat.class);
          }
        }

        switch (type) {
          case Message.CLASSNAME:
            Message message = parseJson(json.getString("entity"), Message.class);
            if (message != null) {
              String string = null;
              if (message.getMessage() != null) {
                string = message.getMessage();
              } else if (ChannelPlugin.getApplication() != null && message.getFile() != null) {
                if (message.getFile().isImage()) {
                  string = Utils.getString(
                      ChannelPlugin.getApplication(),
                      "ch.notification.upload_image.description");
                } else {
                  string = Utils.getString(
                      ChannelPlugin.getApplication(),
                      "ch.notification.upload_file.description");
                }
              }

              if (string != null) {
                RxBus.post(new PushBus(string, person, userChat));
              }
            }
            break;
        }
      } catch (Exception e) {
        L.e(e.getMessage());
      }
    }
  };

  private String getTag(JSONObject json) {
    try {
      String type = json.getString("type");
      String id = json.getJSONObject("entity").getString("id");

      return String.format("%s (%s): ", type, id);
    } catch (Exception ex) {
      return "";
    }
  }

  private Emitter.Listener onCreate = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("+ onCreate) " + getTag((JSONObject) objects[0]) + objects[0].toString());
      onMessage((JSONObject) objects[0], true);
    }
  };

  private Emitter.Listener onUpdate = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("~ onUpdate) " + getTag((JSONObject) objects[0]) + objects[0].toString());
      onMessage((JSONObject) objects[0], true);
    }
  };

  private Emitter.Listener onDelete = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("- onDelete) " + getTag((JSONObject) objects[0]) + objects[0].toString());
      onMessage((JSONObject) objects[0], false);
    }
  };

  private void onMessage(JSONObject json, boolean upsert) {
    try {
      if (!json.has("type") || !json.has("entity")) {
        return;
      }

      String entity = json.getString("entity");
      String type = json.getString("type");

      switch (type) {
        case "Manager":
          Manager manager = parseJson(entity, Manager.class);
          if (manager == null) {
            return;
          }
          RxBus.post(new ChannelModelBus(manager, upsert));
          break;

        case "Message":
          Message message = parseJson(entity, Message.class);
          if (message == null) {
            return;
          }

          if (json.has("refers")) {
            JSONObject refers = json.getJSONObject("refers");
            if (refers.has("file")) {
              File fileRefers = parseJson(refers.getString("file"), File.class);
              RxBus.post(new ChannelModelBus(fileRefers, upsert));
            }
            if (refers.has("webPage")) {
              WebPage webPageRefers = parseJson(refers.getString("webPage"), WebPage.class);
              RxBus.post(new ChannelModelBus(webPageRefers, upsert));
            }
            if (refers.has("bot")) {
              Bot botRefers = parseJson(refers.getString("bot"), Bot.class);
              RxBus.post(new ChannelModelBus(botRefers, upsert));
            }
          }

          RxBus.post(new ChannelModelBus(message, upsert));
          break;

        case "UserChat":
          UserChat userChat = parseJson(entity, UserChat.class);
          if (userChat == null) {
            return;
          }

          if (json.has("refers")) {
            JSONObject refers = json.getJSONObject("refers");
            if (refers.has("message")) {
              ReferMessage referMessage = parseJson(refers.getString("message"), ReferMessage.class);
              RxBus.post(new ChannelModelBus(referMessage, upsert));
            }
            if (refers.has("manager")) {
              Manager referManager = parseJson(refers.getString("manager"), Manager.class);
              RxBus.post(new ChannelModelBus(referManager, upsert));
            }
            if (refers.has("bot")) {
              Bot referBot = parseJson(refers.getString("bot"), Bot.class);
              RxBus.post(new ChannelModelBus(referBot, upsert));
            }
          }
          RxBus.post(new ChannelModelBus(userChat, upsert));
          break;

        case "Session":
          Session session = parseJson(entity, Session.class);
          if (session == null) {
            return;
          }

          if (json.has("refers")) {
            JSONObject refers = json.getJSONObject("refers");
            if (refers.has("bot")) {
              Bot referBot = parseJson(refers.getString("bot"), Bot.class);
              RxBus.post(new ChannelModelBus(referBot, upsert));
            }
            if (refers.has("manager")) {
              Manager referManager = parseJson(refers.getString("manager"), Manager.class);
              RxBus.post(new ChannelModelBus(referManager, upsert));
            }
          }

          RxBus.post(new ChannelModelBus(session, upsert));

          break;

        case "File":
          File file = parseJson(entity, File.class);
          if (file != null) {
            RxBus.post(new ChannelModelBus(file, upsert));
          }
          break;

        case "Channel":
          Channel channel = parseJson(entity, Channel.class);
          if (channel != null) {
            RxBus.post(new ChannelModelBus(channel, upsert));
          }
          break;
      }
    } catch (Exception ex) {
      L.e(ex.getMessage());
    }
  }

  private <T> T parseJson(String entity, Class<T> target) {
    if (entity == null) {
      return null;
    }
    try {
      return gson.fromJson(entity, target);
    } catch (Exception ex) {
      ex.printStackTrace();
      return null;
    }
  }
}

