package com.zoyi.channel.plugin.android.view.video_player;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.util.AttributeSet;
import android.widget.FrameLayout;
import com.zoyi.channel.plugin.android.manager.ChatVideoManager;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

public abstract class AbsVideoPlayerView extends FrameLayout {

  public AbsVideoPlayerView(@NonNull Context context) {
    super(context);

    init(context);
  }

  public AbsVideoPlayerView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);

    init(context);
  }

  public AbsVideoPlayerView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);

    init(context);
  }

  @Initializer
  private void init(Context context) {
    this.context = context;
  }

  private Context context;

  @Nullable
  private Subscription playingStatusSubscription;

  @Nullable
  private String id;

  protected void setId(String id) {
    this.id = id;

    playingStatusSubscription = ChatVideoManager.get().attachStopSignal()
        .subscribeOn(Schedulers.io())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(it -> {
          if (id != null && id.equals(it)) {
            onVideoPaused();
          }
        });
  }

  public boolean getLatestVolumeState() {
    return Optional.ofNullable(id).map(it -> ChatVideoManager.get().getVideoVolumeState(it)).orElse(true);
  }

  public long getLatestSeconds() {
    return Optional.ofNullable(id).map(it -> ChatVideoManager.get().getVideoPosition(it)).orElse(0L);
  }

  protected void changeVideoState(boolean isPlaying) {
    if (isPlaying) {
      ChatVideoManager.get().onVideoStarted(context, id);
    } else {
      ChatVideoManager.get().onVideoStopped(context, id);
    }
  }

  protected void changeVideoPosition(@Nullable Long seconds) {
    if (seconds != null) {
      Optional.ofNullable(id).ifPresent(it -> ChatVideoManager.get().onVideoProgressChanged(it, seconds));
    }
  }

  protected void changeVideoVolumeState(boolean isVolumeOn) {
    if (id != null) {
      ChatVideoManager.get().onVideoVolumeStateChanged(id, isVolumeOn);
    }
  }

  abstract void onVideoPaused();

  public void clear() {
    if (this.playingStatusSubscription != null && !this.playingStatusSubscription.isUnsubscribed()) {
      this.playingStatusSubscription.unsubscribe();
    }
    this.playingStatusSubscription = null;

    onVideoPaused();

    changeVideoState(false);

    this.id = null;
  }
}
