package com.zoyi.channel.plugin.android.util.message;

import android.graphics.Typeface;
import androidx.annotation.Nullable;
import android.text.*;
import android.text.style.StyleSpan;

import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.antlr.*;
import com.zoyi.channel.plugin.android.enumerate.LinkType;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.etc.BlockParseResult;
import com.zoyi.channel.plugin.android.model.rest.Marketing;
import com.zoyi.channel.plugin.android.store.UserStore;
import com.zoyi.channel.plugin.android.util.AssetUtils;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.org.antlr.v4.runtime.*;
import com.zoyi.org.antlr.v4.runtime.tree.ParseTree;
import com.zoyi.org.antlr.v4.runtime.tree.ParseTreeWalker;

import java.util.*;

public class MessageParser extends TextBlockParserBaseListener {

  private String currentAttrName;
  private boolean stateLink;
  private boolean stateAttrValue;

  private StringBuilder linkTextBuilder;
  private StringBuilder attrValueBuilder;
  private StringBuilder variableFallbackTextBuilder;

  private StringBuilder contentBuilder;

  private SpannableStringBuilder spannableStringBuilder;
  private boolean hasOnlyEmoji;

  private ParseTreeWalker walker;

  private int boldCount;
  private int italicCount;

  private Stack<Tag> tags;

  private MessageParseOptions options;
  @Nullable
  private Marketing marketing;

  public class Tag {

    private String name;
    private Map<String, String> attributes;

    public Tag(String name) {
      this.name = name;
      this.attributes = new HashMap<>();
    }

    public String getName() {
      return name;
    }

    private Map<String, String> getAttributes() {
      return attributes;
    }

    private String get(String key) {
      return attributes.get(key);
    }
  }

  public MessageParser(MessageParseOptions options) {
    this(options, null);
  }

  public MessageParser(MessageParseOptions options, @Nullable Marketing marketing) {
    tags = new Stack<>();
    walker = new ParseTreeWalker();

    spannableStringBuilder = new SpannableStringBuilder();
    hasOnlyEmoji = true;

    linkTextBuilder = new StringBuilder();
    attrValueBuilder = new StringBuilder();
    variableFallbackTextBuilder = new StringBuilder();
    contentBuilder = new StringBuilder();

    this.options = options;
    this.marketing = marketing;
  }

  public BlockParseResult parse(String text) {
    tags.clear();

    boldCount = 0;
    italicCount = 0;

    stateLink = false;

    linkTextBuilder.setLength(0);
    attrValueBuilder.setLength(0);
    variableFallbackTextBuilder.setLength(0);

    TextBlockParser parser = Optional.of(text)
        .map(CharStreams::fromString)
        .map(TextBlockLexer::new)
        .map(CommonTokenStream::new)
        .map(TextBlockParser::new)
        .get();

    parser.setBuildParseTree(true);

    walker.walk(this, parser.block());

    return new BlockParseResult(spannableStringBuilder, hasOnlyEmoji);
  }

  @Override
  public void enterTag(TextBlockParser.TagContext ctx) {
    String tagName = Optional.ofNullable(ctx)
        .map(c -> c.TAG_NAME(0))
        .map(ParseTree::getText)
        .orElse(null);

    if (tagName != null) {
      tags.push(new Tag(tagName));

      switch (tagName) {
        case Const.TAG_TYPE_BOLD:
          boldCount++;
          break;
        case Const.TAG_TYPE_ITALIC:
          italicCount++;
          break;
        case Const.TAG_TYPE_LINK:
          stateLink = true;
          break;
      }
    } else {
      tags.push(null);
    }

    this.hasOnlyEmoji = false;
  }

  @Override
  public void exitTag(TextBlockParser.TagContext ctx) {
    Tag tag = tags.pop();

    if (tag != null && tag.getName() != null) {
      switch (tag.getName()) {
        case Const.TAG_TYPE_BOLD:
          boldCount--;
          break;
        case Const.TAG_TYPE_ITALIC:
          italicCount--;
          break;
        case Const.TAG_TYPE_LINK:
          parseLink(tag);
          break;
      }
    }

    this.hasOnlyEmoji = false;
  }

  @Override
  public void enterAttribute(TextBlockParser.AttributeContext ctx) {
    Optional.ofNullable(ctx)
        .map(TextBlockParser.AttributeContext::TAG_NAME)
        .map(ParseTree::getText)
        .ifPresent(tagName -> currentAttrName = tagName);
  }

  @Override
  public void exitAttribute(TextBlockParser.AttributeContext ctx) {
    Optional.ofNullable(tags.peek())
        .ifPresent(top -> top.getAttributes().put(currentAttrName, attrValueBuilder.toString()));

    this.hasOnlyEmoji = false;
  }

  @Override
  public void enterAttrValue(TextBlockParser.AttrValueContext ctx) {
    stateAttrValue = true;

    attrValueBuilder.setLength(0);
  }

  @Override
  public void exitAttrValue(TextBlockParser.AttrValueContext ctx) {
    stateAttrValue = false;
    attrValueBuilder.append(replaceAllEscape(ctx.getText()));
  }


  @Override
  public void exitContent(TextBlockParser.ContentContext ctx) {
    if (!stateLink) {
      spannableStringBuilder.append(getTextSpan(contentBuilder.toString()));

      contentBuilder.setLength(0);
    }
  }

  @Override
  public void exitEmoji(TextBlockParser.EmojiContext ctx) {
    String emojiText = AssetUtils.getEmoji(ChannelIO.getAppContext(), ctx.EMOJI().getText().replaceAll(":", ""));

    if (emojiText == null) {
      this.hasOnlyEmoji = false;
    }

    String text = Optional.ofNullable(emojiText).orElse(ctx.EMOJI().getText());

    if (stateLink) {
      linkTextBuilder.append(text);
    } else {
      contentBuilder.append(text);
    }
  }

  @Override
  public void exitVariable(TextBlockParser.VariableContext ctx) {
    if (this.options.isVariableEnabled()) {
      String key = Optional.ofNullable(ctx.VAR_NAME()).map(ParseTree::getText).orElse("");
      String fallback = Optional.ofNullable(ctx.variableFallback()).map(RuleContext::getText).orElse(String.format("${%s}", key));

      parseText(
          Optional.ofNullable(UserStore.get().user.get())
              .map(u -> u.getProfileValue(key))
              .map(Object::toString)
              .orElse(fallback)
      );
    } else {
      Optional.ofNullable(ctx.getText()).ifPresent(this::parseText);
    }

    this.hasOnlyEmoji = false;
  }

  @Override
  public void exitPlain(TextBlockParser.PlainContext ctx) {
    parseText(ctx.getText());

    this.hasOnlyEmoji = false;
  }

  @Override
  public void exitEscape(TextBlockParser.EscapeContext ctx) {
    if (!stateAttrValue) {
      Optional.ofNullable(ctx)
          .map(RuleContext::getText)
          .ifPresent(text -> parseText(replaceEscape(text)));
      this.hasOnlyEmoji = false;
    }
  }

  private void parseText(String text) {
    if (stateAttrValue) {
      attrValueBuilder.append(text);
      return;
    } else if (stateLink) {
      linkTextBuilder.append(text);
      return;
    }

    contentBuilder.append(text);
  }

  private void parseLink(Tag tag) {
    String inlineText = linkTextBuilder.toString();

    Optional.ofNullable(tag.get(Const.TAG_ATTR_KEY_TYPE))
        .ifPresent(type -> {
          switch (type) {
            case Const.LINK_TYPE_URL:
              if (tag.get(Const.TAG_ATTR_KEY_VALUE) != null) {
                spannableStringBuilder.append(getLinkSpannableString(inlineText, tag.get(Const.TAG_ATTR_KEY_VALUE)));
              } else {
                spannableStringBuilder.append(getURLSpannableString(inlineText));
              }
              break;
            case Const.LINK_TYPE_EMAIL:
              spannableStringBuilder.append(getEmailSpan(inlineText));
              break;
            case Const.LINK_TYPE_MENTION:
              spannableStringBuilder.append(inlineText);
              break;
          }
        });

    linkTextBuilder.setLength(0);

    stateLink = false;
  }

  private void applySpan(SpannableString spannableString, Object span) {
    spannableString.setSpan(span, 0, spannableString.length(), Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
  }

  private void applyStyleSpan(SpannableString spannableString) {
    if (boldCount > 0 && italicCount > 0) {
      applySpan(spannableString, new StyleSpan(Typeface.BOLD_ITALIC));
    } else if (boldCount > 0) {
      applySpan(spannableString, new StyleSpan(Typeface.BOLD));
    } else if (italicCount > 0) {
      applySpan(spannableString, new StyleSpan(Typeface.ITALIC));
    }
  }

  private SpannableString getTextSpan(String text) {
    SpannableString spannableString = new SpannableString(text);
    applyStyleSpan(spannableString);
    return spannableString;
  }

  private SpannableString getLinkSpannableString(String text, String targetUrl) {
    SpannableString spannableString = new SpannableString(text);
    LinkSpan span = new LinkSpan(targetUrl, LinkType.URL, marketing);
    applyStyleSpan(spannableString);
    applySpan(spannableString, span);
    return spannableString;
  }

  private SpannableString getURLSpannableString(String url) {
    return getLinkSpannableString(url, url);
  }

  private SpannableString getEmailSpan(String email) {
    SpannableString spannableString = new SpannableString(email);
    LinkSpan span = new LinkSpan(email, LinkType.EMAIL, marketing);
    applyStyleSpan(spannableString);
    applySpan(spannableString, span);
    return spannableString;
  }

  @Nullable
  private String replaceEscape(String text) {
    String resultText = text == null ? "" : text;

    switch (text) {
      case "&amp;":
        resultText = "&";
        break;
      case "&lt;":
        resultText = "<";
        break;
      case "&gt;":
        resultText = ">";
        break;
      case "&quot;":
        resultText = "\"";
        break;
      case "&dollar;":
        resultText = "$";
        break;
    }

    return resultText;
  }

  private String replaceAllEscape(String text) {
    return (text == null ? "" : text)
        .replaceAll("&amp;", "&")
        .replaceAll("&lt;", "<")
        .replaceAll("&gt;", ">")
        .replaceAll("&quot;", "\"")
        .replaceAll("&dollar;", "$");
  }
}