package com.zoyi.channel.plugin.android.activity.chat.viewholder;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.listener.viewholder.OnAttachmentUploadContentActionListener;
import com.zoyi.channel.plugin.android.activity.chat.model.SendFileItem;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManager;
import com.zoyi.channel.plugin.android.base.adapter.BaseViewHolder;
import com.zoyi.channel.plugin.android.enumerate.SendingState;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.com.bumptech.glide.Glide;
import com.zoyi.com.bumptech.glide.load.engine.DiskCacheStrategy;
import com.zoyi.rx.Subscription;

import java.util.Locale;

public class UploadStateHolder extends BaseViewHolder {

  private ImageView imageThumbnail;
  private TextView textFileName;
  private ImageView imageCancel;

  private ImageView imageRetry;

  private TextView textFileQueue;
  private ProgressBar progressBar;

  @Nullable
  private Subscription progressSubscription;

  @Nullable
  private Subscription waitingCountSubscription;

  @Nullable
  private SendFileItem fileItem;

  private UploadStateHolder(@NonNull View itemView, final @Nullable OnAttachmentUploadContentActionListener onAttachmentUploadContentActionListener) {
    super(itemView);
    imageThumbnail = itemView.findViewById(R.id.ch_imageFileUploadThumbnail);
    textFileName = itemView.findViewById(R.id.ch_textFileUploadFileName);
    imageCancel = itemView.findViewById(R.id.ch_buttonFileUploadCancel);

    imageRetry = itemView.findViewById(R.id.ch_buttonFileUploadRetry);

    textFileQueue = itemView.findViewById(R.id.ch_textFileUploadWaitingQueue);
    progressBar = itemView.findViewById(R.id.ch_progressFileUpload);

    imageCancel.setOnClickListener(view -> {
      if (onAttachmentUploadContentActionListener != null) {
        onAttachmentUploadContentActionListener.onCancelClick(fileItem);
      }
    });

    imageRetry.setOnClickListener(view -> {
      if (onAttachmentUploadContentActionListener != null && fileItem != null) {
        onAttachmentUploadContentActionListener.onResendButtonClick(fileItem);
      }
    });
  }

  public void bind(SendFileItem fileItem) {
    onRecycled();

    this.fileItem = fileItem;
    textFileName.setText(fileItem.getName());

    Glide.with(itemView.getContext())
        .load(fileItem.getUri())
        .diskCacheStrategy(DiskCacheStrategy.RESULT)
        .placeholder(R.drawable.ch_plugin_image_placeholder)
        .error(R.drawable.ch_plugin_image_placeholder)
        .centerCrop()
        .into(imageThumbnail);

    if (fileItem.getState() == SendingState.FAIL) {
      imageCancel.setVisibility(View.VISIBLE);
      imageRetry.setVisibility(View.VISIBLE);

      textFileQueue.setVisibility(View.GONE);
      progressBar.setVisibility(View.GONE);
    } else {
      if (fileItem.getChatId() == null) {
        imageCancel.setVisibility(View.GONE);
        imageRetry.setVisibility(View.GONE);

        // TODO : need to change waiting files count in local chat
        textFileQueue.setVisibility(View.GONE);
        progressBar.setVisibility(View.GONE);
      } else {
        imageCancel.setVisibility(View.VISIBLE);
        imageRetry.setVisibility(View.GONE);

        progressBar.setVisibility(View.VISIBLE);
        progressBar.setProgress(ChatManager.get(fileItem.getChatId()).getCurrentFileUploadProgress());

        progressSubscription = ChatManager.get(fileItem.getChatId())
            .getUploadProgressObservable()
            .subscribe(progress -> progressBar.setProgress(progress));

        setWaitingCount(ChatManager.get(fileItem.getChatId()).getWaitingFileCount());

        waitingCountSubscription = ChatManager.get(fileItem.getChatId())
            .getWaitingCountObservable()
            .subscribe(this::setWaitingCount);
      }
    }
  }

  private void setWaitingCount(int waitingCount) {
    if (waitingCount == 0) {
      textFileQueue.setVisibility(View.GONE);
    } else {
      textFileQueue.setVisibility(View.VISIBLE);
      textFileQueue.setText(
          String.format(
              Locale.getDefault(),
              ResUtils.getString(itemView.getContext(), "ch.file_upload.wait_count"),
              waitingCount
          )
      );
    }
  }

  @Override
  protected void onRecycled() {
    fileItem = null;

    progressBar.setProgress(0);

    if (progressSubscription != null && !progressSubscription.isUnsubscribed()) {
      progressSubscription.unsubscribe();
      progressSubscription = null;
    }

    if (waitingCountSubscription != null && !waitingCountSubscription.isUnsubscribed()) {
      waitingCountSubscription.unsubscribe();
      waitingCountSubscription = null;
    }

    if (imageThumbnail != null) {
      Glide.clear(imageThumbnail);
    }
  }

  public static UploadStateHolder newInstance(
      ViewGroup parent,
      @Nullable OnAttachmentUploadContentActionListener onAttachmentUploadContentActionListener
  ) {
    return new UploadStateHolder(
        LayoutInflater.from(parent.getContext()).inflate(R.layout.ch_plugin_item_upload_state, parent, false),
        onAttachmentUploadContentActionListener
    );
  }
}
