package com.zoyi.channel.plugin.android.activity.chat.view.chat;

import android.content.Context;
import androidx.annotation.Nullable;
import androidx.core.util.Pair;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.LinearLayout;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.listener.view.ChatInteractionActionListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.TypingListener;
import com.zoyi.channel.plugin.android.enumerate.ChatInteractionState;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.*;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.util.io.Keyboard;
import com.zoyi.channel.plugin.android.util.lang.StringUtils;
import com.zoyi.channel.plugin.android.view.button.ChatStartButton;
import com.zoyi.channel.plugin.android.view.layout.WatchedEditText;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.concurrent.TimeUnit;

public class ChatInteractionView extends LinearLayout {

  private Context context;

  private View viewChatBlocked;

  private ChatStartButton buttonNewChat;
  private LinearLayout layoutInteractionContent;
  private View layoutChatInput;
  private View buttonAttach;
  private WatchedEditText editChat;
  private View buttonSend;
  private View buttonMarketingSupportBot;

  @Nullable
  private String userChatId;

  @Nullable
  private ChatInteractionActionListener chatInteractionActionListener;

  @Nullable
  private TypingListener typingListener;

  @Nullable
  private Subscription typingSubscription;

  private ChatInteractionState chatInteractionState = ChatInteractionState.NONE;

  public ChatInteractionView(Context context) {
    super(context);
    init(context);
  }

  public ChatInteractionView(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context);
  }

  public ChatInteractionView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context);
  }

  @Initializer
  private void init(Context context) {
    this.context = context;

    View view = LayoutInflater.from(context).inflate(R.layout.ch_plugin_view_chat_interaction, this, true);

    buttonNewChat = view.findViewById(R.id.ch_buttonChatInteractionStartNewChat);
    buttonNewChat.setOnClickListener(v -> {
      if (chatInteractionActionListener != null) {
        chatInteractionActionListener.startNewChat(Const.RESULT_CHAT_OPEN_NEW_CHAT, Transition.NONE);
      }
    });

    layoutInteractionContent = view.findViewById(R.id.ch_layoutChatInteractionContent);

    buttonMarketingSupportBot = view.findViewById(R.id.ch_buttonChatInteractionMarketingSupportBot);
    buttonMarketingSupportBot.setOnClickListener(v -> {
      if (chatInteractionActionListener != null) {
        chatInteractionActionListener.startMarketingSupportBot();
      }
    });

    viewChatBlocked = view.findViewById(R.id.ch_textChatInteractionBlocked);

    layoutChatInput = view.findViewById(R.id.ch_layoutChatInteractionInput);

    buttonAttach = view.findViewById(R.id.ch_buttonChatInteractionAttach);
    buttonAttach.setOnClickListener(v -> {
      if (chatInteractionActionListener != null) {
        chatInteractionActionListener.onAttachmentButtonClick();
      }
    });

    editChat = view.findViewById(R.id.ch_editChatInteraction);

    buttonSend = view.findViewById(R.id.ch_buttonChatInteractionSend);
    buttonSend.setEnabled(true);
    buttonSend.setOnClickListener(v -> {
      String newString = StringUtils.stripEnd(editChat.getString(), null);

      if (chatInteractionActionListener != null && !TextUtils.isEmpty(newString)) {
        chatInteractionActionListener.onSendClick(newString);
      }
      editChat.setText("");
    });

    editChat.setWatchedTextChangedListener(s -> {
      String newString = StringUtils.stripEnd(s, null);
      boolean isEnabled = !TextUtils.isEmpty(newString);

      if (isEnabled != buttonSend.isEnabled()) {
        buttonSend.setEnabled(isEnabled);
      }

      if (this.userChatId != null) {
        if (this.typingListener != null) {
          this.typingListener.onTypingStateChange(!TextUtils.isEmpty(s));
        }

        removeTypingSubscription();

        if (!TextUtils.isEmpty(s)) {
          SocketManager.typing(Typing.create(Const.TYPING_START, this.userChatId));

          this.typingSubscription = Observable.timer(Typing.TTL, TimeUnit.SECONDS)
              .onBackpressureBuffer()
              .subscribeOn(Schedulers.io())
              .observeOn(AndroidSchedulers.mainThread())
              .subscribe(ignored -> SocketManager.typing(Typing.create(Const.TYPING_STOP, this.userChatId)));
        } else {
          SocketManager.typing(Typing.create(Const.TYPING_STOP, this.userChatId));
        }

        TypingTextSaveStore.get().savedTexts.upsert(new Pair<>(this.userChatId, s));
      }
    });

    editChat.setOnFocusChangeListener((v, hasFocus) -> {
      if (hasFocus) {
        setInputDim(false);
        ProfileBotStore.get().requestFocus.set(false);
      }
    });
  }

  public void initUserChat(String userChatId) {
    if (this.userChatId == null) {
      this.userChatId = userChatId;

      Pair<String, String> pair = TypingTextSaveStore.get().savedTexts.get(userChatId);

      if (pair != null && pair.second != null) {
        editChat.setText(pair.second);
      }
    }
  }

  @Override
  protected void onDetachedFromWindow() {
    super.onDetachedFromWindow();

    removeTypingSubscription();

    if (this.userChatId != null) {
      SocketManager.typing(Typing.create(Const.TYPING_STOP, this.userChatId));
    }
  }

  public void setChatInteractionActionListener(@Nullable ChatInteractionActionListener chatInteractionActionListener) {
    this.chatInteractionActionListener = chatInteractionActionListener;
  }

  public void setTypingListener(@Nullable TypingListener typingListener) {
    this.typingListener = typingListener;
  }

  public void setInputDim(boolean enabled) {
    layoutChatInput.setBackgroundColor(
        ResUtils.getColor(enabled ? R.color.ch_grey200 : R.color.ch_white)
    );
  }

  public void setState(ChatInteractionState chatInteractionState) {
    if (this.chatInteractionState != chatInteractionState) {
      this.chatInteractionState = chatInteractionState;

      Views.setVisibility(buttonNewChat, false);
      Views.setVisibility(layoutInteractionContent, false);
      Views.setVisibility(buttonMarketingSupportBot, false);
      Views.setVisibility(viewChatBlocked, false);
      Views.setVisibility(layoutChatInput, false);

      switch (chatInteractionState) {
        case NORMAL:
          Views.setVisibility(layoutInteractionContent, true);
          Views.setVisibility(layoutChatInput, true);
          break;

        case MARKETING_SUPPORT_BOT:
          Keyboard.close(context, editChat);

          Views.setVisibility(layoutInteractionContent, true);
          Views.setVisibility(buttonMarketingSupportBot, true);
          break;

        case DISABLE:
          Keyboard.close(context, editChat);

          Views.setVisibility(layoutInteractionContent, true);
          Views.setVisibility(viewChatBlocked, true);
          break;

        case NEW_CHAT_START:
          Keyboard.close(context, editChat);

          Views.setVisibility(buttonNewChat, true);
          break;

        case NONE:
          Keyboard.close(context, editChat);
          break;
      }
    }
  }

  public void setFocus() {
    editChat.requestFocus();
  }

  // private functions

  private void removeTypingSubscription() {
    if (this.typingSubscription != null && !this.typingSubscription.isUnsubscribed()) {
      this.typingSubscription.unsubscribe();
      this.typingSubscription = null;
    }
  }
}
