package com.zoyi.channel.plugin.android.activity.common.navigation;

import android.app.Activity;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity2;
import com.zoyi.channel.plugin.android.activity.common.badge.AlertBadge;
import com.zoyi.channel.plugin.android.selector.PluginSelector;
import com.zoyi.channel.plugin.android.selector.UserSelector;
import com.zoyi.channel.plugin.android.bind.Binder;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.util.RxUtils;
import com.zoyi.channel.plugin.android.util.Utils;
import com.zoyi.channel.plugin.android.view.button.MaskImageButton;
import com.zoyi.channel.plugin.android.view.button.TextButton;
import com.zoyi.channel.plugin.android.view.button.ThemeEffectiveView;
import com.zoyi.channel.plugin.android.view.textview.CHTextView;
import com.zoyi.rx.Subscription;

import java.util.ArrayList;
import java.util.List;

public class NavigationView extends LinearLayout {

  private Context context;

  private MaskImageButton buttonBack;
  private AlertBadge alertBadge;
  private ViewGroup contentContainer;
  private LinearLayout actionContainer;

  @Nullable
  private CHTextView textTitle;

  @Nullable
  private Binder dataBinder;
  @Nullable
  private Binder userBinder;

  private boolean hideAlertBadge = false;

  private boolean badgeNotInit = true;

  private int backColor = Color.BLACK;

  private int textColor = Color.WHITE;

  private List<ThemeEffectiveView> views;

  @Nullable
  private Subscription badgeSubscription;

  public NavigationView(Context context) {
    super(context);
    init(context, null);
  }

  public NavigationView(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context, attrs);
  }

  public NavigationView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context, attrs);
  }

  @Initializer
  private void init(Context context, @Nullable AttributeSet attrs) {
    this.context = context;

    View root = LayoutInflater.from(context).inflate(R.layout.ch_plugin_view_navigation, this, true);

    buttonBack = root.findViewById(R.id.ch_buttonNavigationBack);
    buttonBack.setOnClickListener(v -> onBackClick());

    alertBadge = root.findViewById(R.id.ch_alertNavigation);

    contentContainer = root.findViewById(R.id.ch_layoutNavigationContent);

    actionContainer = root.findViewById(R.id.ch_layoutNavigationActions);

    textTitle = root.findViewById(R.id.ch_textNavigationTitle);

    views = new ArrayList<>();

    String titleKey = null;
    String title = null;

    if (attrs != null) {
      TypedArray typedArray = context.getTheme().obtainStyledAttributes(
          attrs,
          R.styleable.NavigationView,
          0,
          0
      );

      try {
        titleKey = typedArray.getString(R.styleable.NavigationView_ch_nv_titleKey);
        title = typedArray.getString(R.styleable.NavigationView_ch_nv_title);
        hideAlertBadge = typedArray.getBoolean(R.styleable.NavigationView_ch_nv_hideAlertBadge, hideAlertBadge);
      } finally {
        typedArray.recycle();
      }
    }

    if (textTitle != null) {
      if (titleKey != null) {
        textTitle.setTextKey(titleKey);
      } else if (title != null) {
        textTitle.setText(title);
      }
    }
  }

  @Override
  protected void onAttachedToWindow() {
    super.onAttachedToWindow();

    dataBinder = PluginSelector.bindPlugin(plugin -> {
      backColor = plugin.getBackgroundColor();
      textColor = plugin.getTextColor();

      buttonBack.setColor(textColor);

      alertBadge.setTheme(backColor, textColor);

      if (textTitle != null) {
        textTitle.setTextColor(textColor);
      }

      if (views != null) {
        for (ThemeEffectiveView view : views) {
          if (view != null) {
            view.onThemeChange(backColor, textColor);
          }
        }
      }
    });

    userBinder = UserSelector.bindUserAlertCount(count -> {
      if (badgeNotInit) {
        badgeNotInit = false;
        setAlertBadge(count);
      } else {
        badgeSubscription = RxUtils.debounce(badgeSubscription, 600, () -> setAlertBadge(count));
      }
    });
  }

  private void setAlertBadge(int count) {
    if (!hideAlertBadge && count > 0) {
      alertBadge.setCount(count);
      alertBadge.setVisibility(View.VISIBLE);
    } else {
      alertBadge.setVisibility(View.INVISIBLE);
    }
  }

  @Override
  protected void onDetachedFromWindow() {
    super.onDetachedFromWindow();

    if (dataBinder != null) {
      dataBinder.unbind();
    }
    if (userBinder != null) {
      userBinder.unbind();
    }
    if (views != null) {
      views.clear();
    }
  }

  @Override
  public void addView(View child, int index, ViewGroup.LayoutParams params) {
    if (child == null) {
      return;
    }
    if (CompareUtils.exists(
        child.getId(),
        R.id.ch_rootNavigation,
        R.id.ch_buttonNavigationBack,
        R.id.ch_alertNavigation,
        R.id.ch_layoutNavigationContent,
        R.id.ch_layoutNavigationActions
    )) {
      super.addView(child, index, params);
    } else if (contentContainer != null) {
      if (textTitle != null) {
        contentContainer.removeAllViews();
        textTitle = null;
      }

      if (child instanceof ThemeEffectiveView) {
        ((ThemeEffectiveView) child).onThemeChange(backColor, textColor);
        addThemeEffectiveView((ThemeEffectiveView) child);
      }
      contentContainer.addView(child);
    }
  }

  private void onBackClick() {
    if (context instanceof BaseActivity2) {
      ((BaseActivity2) context).finish();
    } else if (context instanceof Activity) {
      ((Activity) context).finish();
    }
  }

  private void addThemeEffectiveView(ThemeEffectiveView themeEffectiveView) {
    if (views != null) {
      views.add(themeEffectiveView);
    }
  }

  public void setTitle(String title) {
    if (textTitle != null) {
      textTitle.setText(title);
    }
  }

  @Nullable
  public View addAction(String text, OnClickListener listener) {
    if (actionContainer != null) {
      TextButton button = new TextButton(context);
      button.setText(text);
      button.setTextColor(textColor);
      button.setOnClickListener(listener);

      addThemeEffectiveView(button);
      actionContainer.addView(button);

      return button;
    }
    return null;
  }

  @Nullable
  public View addAction(int resId, OnClickListener listener) {
    return addAction(resId, 30, listener);
  }

  @Nullable
  public View addAction(int resId, float iconDp, OnClickListener listener) {
    if (actionContainer != null) {
      MaskImageButton button = new MaskImageButton(context);
      LayoutParams layoutParams = new LayoutParams((int) Utils.dpToPx(30), (int) Utils.dpToPx(30));

      if (iconDp < 30f) {
        int padding = (int) Utils.dpToPx((30f - iconDp) / 2);
        button.setPadding(padding, padding, padding, padding);
      }
      if (actionContainer.getChildCount() > 0) {
        layoutParams.setMargins((int) Utils.dpToPx(18), 0, 0, 0);
      }

      button.setLayoutParams(layoutParams);

      button.setMask(resId);
      button.setColor(textColor);
      button.setOnClickListener(listener);

      addThemeEffectiveView(button);
      actionContainer.addView(button);

      return button;
    }
    return null;
  }
}
