package com.zoyi.channel.plugin.android.util;

import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.model.rest.TimeRange;
import com.zoyi.channel.plugin.android.open.option.Language;
import com.zoyi.channel.plugin.android.store.SettingsStore;
import com.zoyi.channel.plugin.android.util.lang.DateUtils;
import com.zoyi.rx.Single;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Action1;
import com.zoyi.rx.schedulers.Schedulers;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Created by mika on 2016. 1. 21..
 */
public class TimeUtils {

  public final static SimpleDateFormat FULL_YEAR = new SimpleDateFormat("yyyy", Locale.getDefault());
  public final static SimpleDateFormat FULL_DAY = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
  public final static SimpleDateFormat FULL_MINUTE = new SimpleDateFormat("yyyy-MM-dd HH:mm", Locale.getDefault());

  public final static SimpleDateFormat SIMPLE_DAY = new SimpleDateFormat("yyyy-M-d", Locale.getDefault());
  public final static SimpleDateFormat SIMPLE_DATE = new SimpleDateFormat("M/d", Locale.getDefault());
  public final static SimpleDateFormat SIMPLE_YEAR = new SimpleDateFormat("yy/M/d", Locale.getDefault());
  public final static SimpleDateFormat SECOND = new SimpleDateFormat("HH:mm:ss", Locale.getDefault());
  public final static SimpleDateFormat MINUTE_24 = new SimpleDateFormat("H:mm", Locale.getDefault());
  public final static SimpleDateFormat MINUTE_12 = new SimpleDateFormat("h:mm", Locale.getDefault());

  // new format

  public final static SimpleDateFormat KO_YEAR_DATE = new SimpleDateFormat("yyyy'%s' M'%s' dd'%s'", Locale.ENGLISH);
  public final static SimpleDateFormat JA_YEAR_DATE = new SimpleDateFormat("yyyy'%s'M'%s'dd'%s'", Locale.ENGLISH);
  public final static SimpleDateFormat DEFAULT_YEAR_DATE = new SimpleDateFormat("MMM dd, yyyy", Locale.ENGLISH);
  public final static SimpleDateFormat YEAR_DATE = new SimpleDateFormat("yyyy-MM-dd", Locale.US);
  public final static SimpleDateFormat TIME_12 = new SimpleDateFormat("hh:mm a", Locale.US);

  public final static Long MAX_MICROSECONDS = 10000000000000000L;
  public static long TIME_ZONE_OFFSET = 0L;
  public static long SERVER_TIME_OFFSET = 0L;

  private static final int ONE_MINUTE = 60;
  private static final int ONE_HOUR = ONE_MINUTE * 60;
  private static final int ONE_DAY = ONE_HOUR * 24;
  private static final int ONE_WEEK = ONE_DAY * 7;
  private static final int ONE_MONTH = ONE_DAY * 30;
  private static final int ONE_YEAR = ONE_DAY * 365;

  public static void refreshOffset() {
    TIME_ZONE_OFFSET = TimeZone.getDefault().getRawOffset();
  }

  public static void syncServerTime(long serverTime) {
    SERVER_TIME_OFFSET = serverTime - System.currentTimeMillis();
  }

  public static long getCurrentTime() {
    return System.currentTimeMillis() + SERVER_TIME_OFFSET;
  }

  public static String get(SimpleDateFormat format, Long ts) {
    return get(format, new Date(ts));
  }

  public static String get(SimpleDateFormat format, Date date) {
    return format.format(date);
  }

  public static String get(Long ts) {
    if (ts == null) {
      return "";
    }

    Date date = new Date(ts);
    Date now = new Date(getCurrentTime());

    if (DateUtils.isSameDay(now, date)) {
      return getTime(date);
    }

    Calendar cal1 = Calendar.getInstance();
    cal1.setTime(date);
    Calendar cal2 = Calendar.getInstance();
    cal2.setTime(now);

    int year1 = cal1.get(Calendar.YEAR) - 1900;
    int year2 = cal2.get(Calendar.YEAR) - 1900;

    if (year1 == year2) {
      return get(SIMPLE_DATE, date);
    }
    return get(FULL_DAY, date);
  }

  public static String getTime(Long ts) {
    if (ts == null || ts == 0L) {
      return "";
    }
    return getTime(new Date(ts));
  }

  public static String getTime(Date date) {
    /*
    if (DateFormat.is24HourFormat(ChannelIO.getContext())) {
      return getScript(MINUTE_24, date);
    }
    */

    Calendar cal = Calendar.getInstance();
    cal.setTime(date);

    String time = get(MINUTE_12, date);
    return String.format("%s %s", time, cal.get(Calendar.AM_PM) == Calendar.PM ? "PM" : "AM");
  }

  public static String getFullYear(Long ts) {
    String date = TimeUtils.get(TimeUtils.SIMPLE_DAY, ts);
    String time = TimeUtils.getTime(ts);

    return String.format("%s %s", date, time);
  }

  public static String getTimeRangeString(TimeRange timeRange) {
    return String.format("%s ~ %s", getTime(timeRange.getFrom()), getTime(timeRange.getTo()));
  }

  private static String getTime(int time) {
    time = Math.max(time, 0);
    time = Math.min(time, 1439);

    int hour = time / 60;
    int minute = time % 60;
    String add = hour < 12 ? "AM" : "PM";

    return String.format("%02d:%02d %s", hour > 12 ? hour - 12 : hour, minute, add);
  }

  // new functions

  public static String formatDatetime(@Nullable Long timestamp) {
    if (timestamp == null || timestamp == 0L) {
      return "";
    }
    return String.format(Locale.US, "%s %s", YEAR_DATE.format(new Date(timestamp)), TIME_12.format(new Date(timestamp)));
  }

  public static String formatTranslatedDate(@Nullable Long timestamp, @Nullable Language language) {
    if (timestamp == null || timestamp == 0L) {
      return "";
    }

    if (language != null) {
      switch (language) {
        case KOREAN:
          return String.format(
              KO_YEAR_DATE.format(new Date(timestamp)),
              ResUtils.getString("ch.year"),
              ResUtils.getString("ch.month"),
              ResUtils.getString("ch.day")
          );

        case JAPANESE:
          return String.format(
              JA_YEAR_DATE.format(new Date(timestamp)),
              ResUtils.getString("ch.year"),
              ResUtils.getString("ch.month"),
              ResUtils.getString("ch.day")
          );
      }
    }
    return DEFAULT_YEAR_DATE.format(new Date(timestamp));
  }

  public static Long calculateUntilTomorrow() {
    Calendar c = Calendar.getInstance();
    c.setTime(new Date(getCurrentTime()));
    c.add(Calendar.DAY_OF_MONTH, 1);
    c.set(Calendar.HOUR_OF_DAY, 0);
    c.set(Calendar.MINUTE, 0);
    c.set(Calendar.SECOND, 0);
    c.set(Calendar.MILLISECOND, 0);
    return c.getTimeInMillis() - getCurrentTime();
  }

  public static Boolean isSameDate(@Nullable Long timestamp) {
    return get(FULL_DAY, timestamp).equals(get(FULL_DAY, getCurrentTime()));
  }

  public static Subscription wait(@Nullable Long milliseconds, @Nullable OnWaitSuccessListener listener) {
    if (milliseconds != null && listener != null) {
      return Single.just(0)
          .delay(milliseconds, TimeUnit.MILLISECONDS)
          .subscribeOn(Schedulers.io())
          .observeOn(AndroidSchedulers.mainThread())
          .onErrorReturn(null)
          .subscribe(integer -> listener.onSuccess());
    } else {
      return null;
    }
  }

  public interface OnWaitSuccessListener {
    void onSuccess();
  }
}
