package com.zoyi.channel.plugin.android.action;

import android.app.Activity;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.global.Api;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.ActionButton;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.wrapper.*;
import com.zoyi.channel.plugin.android.network.*;
import com.zoyi.channel.plugin.android.store.*;
import com.zoyi.channel.plugin.android.util.ChannelUtils;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.functions.Action0;
import com.zoyi.rx.functions.Action1;

import java.util.HashMap;
import java.util.Map;

public class ChatAction {

  public static void createUserChat(@Nullable String pluginId, String url, RestSubscriber<UserChatWrapper> subscriber) {
    if (pluginId != null) {
      Api.createUserChat(pluginId, url)
          .runBy(ActionType.CREATE_USER_CHAT)
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void removeUserChat(@Nullable String chatId, Action1<UserChatWrapper> onSuccess, Action0 onFail) {
    if (chatId != null) {
      Api.removeUserChat(chatId)
          .cancelBy(ActionType.SHUTDOWN)
          .run(new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              onFail.call();
            }

            @Override
            public void onSuccess(@NonNull UserChatWrapper userChatWrapper) {
              onSuccess.call(userChatWrapper);
            }
          });
    } else {
      onFail.call();
    }
  }

  public static void fetchUserChat(@Nullable String chatId, RestSubscriber<UserChatWrapper> repo) {
    if (chatId != null) {
      Api.getUserChat(chatId)
          .runBy(ActionType.FETCH_USER_CHAT)
          .cancelBy(ActionType.SOCKET_DISCONNECTED, ActionType.CHAT_CLOSED, ActionType.SHUTDOWN)
          .run(repo);
    }
  }

  public static void fetchPopUpChat(@Nullable String chatId) {
    if (chatId != null) {
      Api.getUserChat(chatId)
          .runBy(ActionType.FETCH_LAST_CHAT)
          .cancelBy(ActionType.SHUTDOWN)
          .run(new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onSuccess(@NonNull UserChatWrapper repo) {
              Activity topActivity = GlobalStore.get().topActivity.get();
              if (repo.getMessage() != null && !repo.getMessage().isDeleted() && !ChannelUtils.isChannelPluginActivity(topActivity)) {
                InAppPushStore.get().inAppPushMessage.set(repo.getMessage());
              }
            }
          });
    }
  }

  public static void fetchMessages(
      @Nullable String chatId,
      RestSubscriber<MessagesWrapper> repo
  ) {
    if (chatId != null) {
      Api.getMessages(chatId, null, Const.MESSAGE_FETCH_LIMIT, Const.DESC)
          .runBy(ActionType.FETCH_MESSAGES)
          .cancelBy(ActionType.SOCKET_DISCONNECTED, ActionType.CHAT_CLOSED, ActionType.SHUTDOWN)
          .run(repo);
    }
  }

  public static void fetchBackwardMessages(
      @Nullable String chatId,
      String since,
      RestSubscriber<MessagesWrapper> repo
  ) {
    if (chatId != null) {
      Api.getMessages(chatId, since, Const.MESSAGE_FETCH_LIMIT, Const.DESC)
          .runBy(ActionType.FETCH_BACKWARD_MESSAGES)
          .cancelBy(ActionType.FETCH_MESSAGES, ActionType.CHAT_CLOSED, ActionType.SOCKET_DISCONNECTED, ActionType.SHUTDOWN)
          .run(repo);
    }
  }

  public static void sendTextMessage(@Nullable String chatId, SendTextItem item, RestSubscriber<MessageWrapper> subscriber) {
    if (chatId != null && item != null) {
      RequestBody body = RequestUtils.form()
          .set(Const.FIELD_PLAIN_TEXT, item.getText())
          .set(Const.FIELD_REQUEST_ID, item.getRequestId())
          .create();

      Api.sendMessage(chatId, body)
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  @Nullable
  public static Subscription sendFileMessage(
      @Nullable String chatId,
      SendFileItem item,
      Action1<Integer> progressAction,
      RestSubscriber<MessageWrapper> subscriber
  ) {
    String channelId = Optional.ofNullable(ChannelStore.get().channelState.get()).map(Channel::getId).orElse(null);

    if (channelId != null && chatId != null && item != null) {
      ProgressRequestBody progressRequestBody = new ProgressRequestBody(item.getUri(), item.getName(), item.getSize(), progressAction);
      return Api.uploadFile(channelId, chatId, item.getRequestId(), progressRequestBody)
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
    return null;
  }

  public static void closeChat(@Nullable String chatId, SendActionItem item, RestSubscriber<MessageWrapper> subscriber) {
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      Api.closeUserChat(chatId, actionButton.getMessage().getId(), item.getRequestId())
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void reviewChat(@Nullable String chatId, String review, SendActionItem item, RestSubscriber<MessageWrapper> subscriber) {
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      Api.reviewUserChat(chatId, actionButton.getMessage().getId(), item.getRequestId(), review)
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void selectForm(@Nullable String chatId, SendActionItem item, RestSubscriber<MessageWrapper> subscriber) {
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      Map<String, String> submit = new HashMap<>();
      submit.put(Const.FIELD_ID, actionButton.getMessage().getId());
      submit.put(Const.FIELD_KEY, actionButton.getKey());

      RequestBody body = RequestUtils.form()
          .set(Const.FIELD_PLAIN_TEXT, item.getText())
          .set(Const.FIELD_REQUEST_ID, item.getRequestId())
          .set(Const.FIELD_SUBMIT, submit)
          .create();

      Api.sendMessage(chatId, body)
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void selectSupportBotStep(@Nullable String chatId, SendActionItem item, RestSubscriber<MessageWrapper> subscriber) {
    ActionButton actionButton = item.getActionButton();

    if (chatId != null && actionButton != null) {
      Api.submitButtonAction(chatId, actionButton.getKey(), actionButton.getMessage().getId(), item.getRequestId())
          .cancelBy(ActionType.SHUTDOWN)
          .run(subscriber);
    }
  }

  public static void read(@Nullable String chatId) {
    if (chatId != null) {
      Api.read(chatId).cancelBy(ActionType.SHUTDOWN).run(new RestSubscriber<>());
    }
  }
}
