package com.zoyi.channel.plugin.android.activity.chat.viewholder;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;
import android.widget.LinearLayout;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.listener.*;
import com.zoyi.channel.plugin.android.activity.chat.view.attachment.AttachmentsLayout;
import com.zoyi.channel.plugin.android.activity.chat.view.viewholder.*;
import com.zoyi.channel.plugin.android.enumerate.TranslationState;
import com.zoyi.channel.plugin.android.model.etc.TranslationInfo;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.selector.TranslationSelector;
import com.zoyi.channel.plugin.android.store.binder.Binder;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.ListUtils;
import com.zoyi.com.annimon.stream.Stream;

import java.util.ArrayList;
import java.util.List;

public abstract class BaseMessageHolder
    extends AbsMessageHolder
    implements OnBaseMessageActionListener, OnAttachmentContentActionListener, OnLinkButtonActionListener {

  private AbsMessageTextView textMessage;

  protected Context context;

  private LinearLayout layoutContent;
  private AttachmentsLayout viewMessageContentAttachments;
  private WebPageContentView viewMessageContentWebPage;
  private TranslationContentView viewMessageContentTranslation;

  private List<MessageContentView> contents;

  @Nullable
  private OnBaseMessageActionListener listener;

  @Nullable
  private Binder translationBinder;

  BaseMessageHolder(View itemView, @Nullable OnBaseMessageActionListener listener) {
    super(itemView);

    this.context = itemView.getContext();
    this.listener = listener;

    textMessage = itemView.findViewById(R.id.ch_viewMessageHolderText);
    textMessage.setLinkButtonActionListener(this);

    // find content view
    layoutContent = itemView.findViewById(R.id.ch_rootLayoutMessageContent);

    viewMessageContentTranslation = itemView.findViewById(R.id.ch_viewMessageContentTranslation);

    viewMessageContentAttachments = itemView.findViewById(R.id.ch_viewMessageContentAttachments);

    viewMessageContentWebPage = itemView.findViewById(R.id.ch_viewMessageContentWebPage);
    viewMessageContentWebPage.setOnWebPageActionListener(this);

    // add all contents
    contents = new ArrayList<>();

    contents.add(viewMessageContentTranslation);
    contents.add(viewMessageContentAttachments);
    contents.add(viewMessageContentWebPage);
  }

  protected void setGravity(int gravity) {
    if (layoutContent != null) {
      layoutContent.setGravity(gravity);
    }
    if (viewMessageContentWebPage != null) {
      viewMessageContentWebPage.setGravity(gravity);
    }
    if (viewMessageContentAttachments != null) {
      viewMessageContentAttachments.setGravity(gravity);
    }
  }

  @Override
  public void bind(Long createdAt, boolean showDate, boolean isHead, boolean isTail) {
    super.bind(createdAt, showDate, isHead, isTail);

    Stream.ofNullable(contents).forEach(content -> {
      if (content != null) {
        content.clear();
        content.resetView();
        content.setVisibility(View.GONE);
      }
    });

    if (textMessage != null) {
      textMessage.setHead(isHead);
      textMessage.initViews();
    }
  }

  // content setter

  protected void setText(@Nullable CharSequence text) {
    if (textMessage != null) {
      if (text != null) {
        textMessage.setVisibility(View.VISIBLE);
        textMessage.setText(text);
      }
    }
  }

  protected void setBlocks(@Nullable List<Block> blocks) {
    if (textMessage != null) {
      if (blocks != null) {
        textMessage.setVisibility(View.VISIBLE);
        textMessage.setBlocks(blocks);
      }
    }
  }

  protected void setLinkButtons(@Nullable List<Button> buttons) {
    if (textMessage != null && buttons != null && buttons.size() > 0) {
      textMessage.setVisibility(View.VISIBLE);
      textMessage.setLinkButtons(buttons);
    }
  }

  protected void setDeleteMessage() {
    if (textMessage != null) {
      textMessage.setVisibility(View.VISIBLE);
      textMessage.setDelete();
    }
  }

  protected void setWebPage(@Nullable String messageId, WebPage webPage) {
    if (viewMessageContentWebPage != null) {
      viewMessageContentWebPage.setVisibility(View.VISIBLE);
      viewMessageContentWebPage.setWebPage(messageId, webPage);
    }
  }

  protected void setAttachments(
      @Nullable String messageId,
      @Nullable List<File> attachments,
      @Nullable OnAttachmentContentActionListener listener
  ) {
    if (viewMessageContentAttachments != null) {
      viewMessageContentAttachments.resetView();

      if (messageId != null && ListUtils.hasItems(attachments)) {
        viewMessageContentAttachments.setVisibility(View.VISIBLE);
        viewMessageContentAttachments.setAttachments(messageId, attachments, listener);
      }
    }
  }

  protected void setTranslation(
      @Nullable String chatId,
      @Nullable String messageId,
      String language,
      @Nullable List<Block> blocks
  ) {
    if (viewMessageContentTranslation != null) {
      viewMessageContentTranslation.setVisibility(View.VISIBLE);
      viewMessageContentTranslation.bind(chatId, messageId, language);

      // exception: attach translation binder
      translationBinder = TranslationSelector.bindTranslation(TranslationInfo.createKey(chatId, messageId, language), translationInfo -> {
        if (translationInfo != null && translationInfo.getState() == TranslationState.TRANSLATED && translationInfo.getBlocks() != null) {
          textMessage.setVisibility(View.VISIBLE);
          textMessage.setBlocks(translationInfo.getBlocks());
        } else if (blocks != null) {
          textMessage.setVisibility(View.VISIBLE);
          textMessage.setBlocks(blocks);
        }
      });
    }
  }

  @Override
  public void onActionClick(@NonNull String actionType, @NonNull ActionButton actionButton) {
    if (this.listener != null) {
      listener.onActionClick(actionType, actionButton);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), null);
      }
    }
  }

  @Override
  public void onUrlClick(String url) {
    if (this.listener != null) {
      this.listener.onUrlClick(url);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), url);
      }
    }
  }

  @Override
  public void onOpenVideoClick(File attachment, long startAt) {
    if (this.listener != null) {
      this.listener.onOpenVideoClick(attachment, startAt);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), null);
      }
    }
  }

  @Override
  public void onAttachmentClick(File attachment) {
    if (this.listener != null && getMarketing() != null) {
      this.listener.onMarketingAction(getMarketing(), null);
    }
  }

  @Override
  public void onAttachmentClick(File attachment, Message message) {
    if (this.listener != null) {
      this.listener.onAttachmentClick(attachment, message);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), null);
      }
    }
  }

  @Override
  public void onAttachmentActionClick() {
    if (this.listener != null && getMarketing() != null) {
      this.listener.onMarketingAction(getMarketing(), null);
    }
  }

  @Override
  public void onAttachmentLongClick(File attachment) {
    if (this.listener != null && getMarketing() != null) {
      this.listener.onMarketingAction(getMarketing(), null);
    }
  }

  @Override
  public void onLinkButtonClick(String url) {
    if (this.listener != null) {
      this.listener.onUrlClick(url);

      if (getMarketing() != null) {
        this.listener.onMarketingAction(getMarketing(), url);
      }
    }
  }

  @Override
  public void onMarketingAction(Marketing marketing, @Nullable String url) {
    if (this.listener != null) {
      this.listener.onMarketingAction(marketing, url);
    }
  }

  @Nullable
  protected abstract Marketing getMarketing();

  @Override
  protected void clear() {
    Stream.ofNullable(contents).forEach(content -> {
      if (content != null) {
        content.clear();
      }
    });

    // exception: release translation binder
    if (translationBinder != null) {
      translationBinder.unbind();
      translationBinder = null;
    }
  }
}
