package com.zoyi.channel.plugin.android.socket;

import android.app.Activity;
import android.app.Application;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.enumerate.SocketStatus;
import com.zoyi.channel.plugin.android.global.*;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.Manager;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.store.*;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.util.draw.Display;
import com.zoyi.channel.plugin.android.util.lang.StringUtils;
import com.zoyi.com.google.gson.Gson;
import com.zoyi.io.socket.client.*;
import com.zoyi.io.socket.emitter.Emitter;
import com.zoyi.io.socket.engineio.client.EngineIOException;
import com.zoyi.io.socket.engineio.client.transports.WebSocket;
import com.zoyi.rx.*;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.subjects.PublishSubject;

import org.json.JSONException;
import org.json.JSONObject;

import java.net.URISyntaxException;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Created by mika on 2016. 4. 8..
 */
public class SocketManager {

  private static SocketManager socketManager;

  private Application application;
  private Socket socket;
  private Timer reconnectConsumer;
  private AtomicBoolean forceDisconnect;
  private AtomicBoolean ready;
  private AtomicBoolean error;
  private Subscription heartbeat;
  private BlockingQueue<Integer> reconnectQueue;
  private int[] attemptDelay = { 5, 8, 13, 21, 34, 55, 89 };
  private AtomicInteger attemptCount = new AtomicInteger();

  private static final int HEARTBEAT_INTERVAL = 30000;
  private static final int RECONNECT_POP_INTERVAL = 500;

  private static PublishSubject<String> chatJoinedPublishSubject = PublishSubject.create();

  public static Observable<String> observable() {
    return chatJoinedPublishSubject.onBackpressureBuffer().observeOn(AndroidSchedulers.mainThread());
  }

  public static void post(String chatId) {
    if (chatJoinedPublishSubject != null) {
      chatJoinedPublishSubject.onNext(chatId);
    }
  }

  String getSocketEndPoint() {
    if (application != null) {
      return application.getString(R.string.socket_endpoint);
    }
    return "https://ws.channel.io/front";
  }

  private String channelId;

  public static void create(Application application) {
    if (socketManager == null) {
      socketManager = new SocketManager();
      socketManager.application = application;
      socketManager.init();
    }
  }

  private void init() {
    forceDisconnect = new AtomicBoolean(false);
    ready = new AtomicBoolean(false);
    error = new AtomicBoolean(false);
    reconnectQueue = new ArrayBlockingQueue<>(1);

    try {
      IO.Options options = new IO.Options();
      options.reconnection = false;
      options.transports = new String[] { WebSocket.NAME };
      socket = IO.socket(getSocketEndPoint(), options);
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }

    socket.on(Socket.EVENT_CONNECT, onConnect);                     // void
    socket.on(Socket.EVENT_CONNECT_ERROR, onConnectError);          // EngineIOException
    socket.on(Socket.EVENT_CONNECT_TIMEOUT, onConnectTimeout);
    socket.on(Socket.EVENT_CONNECTING, onConnecting);               // void
    socket.on(Socket.EVENT_DISCONNECT, onDisconnect);                // compare
    socket.on(Socket.EVENT_ERROR, onError);                         // EngineIOException
    //socket.on(Socket.EVENT_MESSAGE, onMessage);
    socket.on(Socket.EVENT_PING, onPing);                           // void
    socket.on(Socket.EVENT_PONG, onPong);                           // long
    socket.on(Socket.EVENT_RECONNECT, onReconnect);                 // int
    socket.on(Socket.EVENT_RECONNECT_ATTEMPT, onReconnectAttempt);  // int
    socket.on(Socket.EVENT_RECONNECT_ERROR, onReconnectError);      // SocketIOException
    socket.on(Socket.EVENT_RECONNECT_FAILED, onReconnectFailed);
    socket.on(Socket.EVENT_RECONNECTING, onReconnecting);           // int

    socket.on(SocketEvent.AUTHENTICATED, onAuthenticated);          // boolean
    socket.on(SocketEvent.READY, onReady);
    socket.on(SocketEvent.CREATE, onCreate);                        // JSONObject
    socket.on(SocketEvent.DELETE, onDelete);                        // JSONObject
    socket.on(SocketEvent.JOINED, onJoined);                        // JSONObject
    socket.on(SocketEvent.LEAVED, onLeaved);                        // JSONObject
    socket.on(SocketEvent.PUSH, onPush);                          // JSONObject
    //socket.on(SocketEvent.RECONNECT_ATTEMPT, onReconnectAttempt2);
    socket.on(SocketEvent.UNAUTHORIZED, onUnauthorized);            // JSONObject
    socket.on(SocketEvent.UPDATE, onUpdate);                        // JSONObject
    socket.on(SocketEvent.TYPING, onTyping);                        // JSONObject
  }

  public static void setChannelId(@Nullable String channelId) {
    if (socketManager != null) {
      socketManager.channelId = channelId;
    }
  }

  // static functions

  public static boolean isReady() {
    if (socketManager != null) {
      return socketManager.ready.get();
    }
    return false;
  }

  public static boolean isError() {
    if (socketManager != null) {
      return socketManager.error.get();
    }
    return false;
  }

  public static void connect() {
    if (socketManager != null) {
      socketManager.connectSocket(true);
    }
  }

  public static void reconnect() {
    if (socketManager != null && GlobalStore.get().messengerState.get()) {
      socketManager.enqueueReconnect();
    }
  }

  public static void joinChat(@Nullable String chatId) {
    if (socketManager != null) {
      socketManager.chatAction(SocketEvent.ACTION_JOIN, chatId);
    }
  }

  public static void leaveChat(@Nullable String chatId) {
    if (socketManager != null) {
      socketManager.chatAction(SocketEvent.ACTION_LEAVE, chatId);
    }
  }

  public static void typing(Typing typing) {
    try {
      if (socketManager != null) {
        socketManager.emit(SocketEvent.TYPING, new JSONObject(new Gson().toJson(typing)));
      }
    } catch (JSONException e) {
      e.printStackTrace();
    }
  }

  public static void disconnect() {
    if (socketManager != null) {
      socketManager.disconnect(true);
      socketManager.stopHeartbeat();
    }
  }

  public static void destroy() {
    if (socketManager != null) {
      socketManager.setReconnectConsumer(false);
      socketManager.socket.off();
      socketManager.socket.disconnect();

      socketManager.channelId = null;
      socketManager.forceDisconnect = null;
      socketManager.ready = null;
      socketManager.reconnectQueue = null;
      socketManager.socket = null;

      socketManager = null;
    }
  }

  // internal functions

  private void enqueueReconnect() {
    if (!ready.get()) {
      try {
        reconnectQueue.add(1);
      } catch (Exception ignored) {
      }
    }
  }

  private void clearReconnectQueue() {
    try {
      reconnectQueue.clear();
    } catch (Exception ignored) {
    }
  }

  private void connectSocket(boolean isManual) {
    if (socket != null && !socket.connected() && channelId != null) {
      L.d("try connect");
      socket.connect();
      setReconnectConsumer(true);

      if (isManual) {
        SocketStore.get().socketState.set(SocketStatus.CONNECTING);
      }
    }
  }

  private void disconnect(boolean force) {
    forceDisconnect.set(force);

    try {
      clearReconnectQueue();
      socket.disconnect();
    } catch (Exception ex) {
    }

    if (!force) {
      enqueueReconnect();
    } else {
      setReconnectConsumer(false);
    }
  }

  private void authentication() {
    String jwt = GlobalStore.get().jwt.get();

    if (jwt == null) {
      disconnect(true);
    } else {
      emit(SocketEvent.ACTION_AUTHENTICATION, jwt);
    }
  }

  private synchronized void setReconnectConsumer(boolean flag) {
    if (flag) {
      if (reconnectConsumer == null) {
        reconnectConsumer = new Timer();
        reconnectConsumer.schedule(new TimerTask() {
          @Override
          public void run() {
            try {
              Integer data = reconnectQueue.peek();
              if (data != null) {
                int index = Math.min(attemptCount.getAndIncrement(), attemptDelay.length - 1);
                Thread.sleep(attemptDelay[index] * 1000);
                reconnectQueue.remove();
                connectSocket(false);
              }
            } catch (Exception e) {
              L.e(e.getMessage());
            }
          }
        }, RECONNECT_POP_INTERVAL, RECONNECT_POP_INTERVAL);
      }
    } else {
      if (reconnectConsumer != null) {
        try {
          reconnectConsumer.cancel();
        } catch (Exception ignored) {
        } finally {
          reconnectConsumer = null;
        }
      }
    }
  }

  private void startHeartbeat() {
    stopHeartbeat();
    sendHeartbeat();

    heartbeat = Observable.interval(HEARTBEAT_INTERVAL, TimeUnit.MILLISECONDS)
        .subscribe(new Subscriber<Long>() {
          @Override
          public void onCompleted() {
          }

          @Override
          public void onError(Throwable e) {
          }

          @Override
          public void onNext(Long aLong) {
            if (Display.isLocked()) {
              L.d("Display locked. disconnect.");

              disconnect(true);
            } else {
              sendHeartbeat();
            }
          }
        });
  }

  private void sendHeartbeat() {
    try {
      L.i("Heartbeat");
      socket.emit(SocketEvent.ACTION_HEARTBEAT, "");
    } catch (Exception ex) {
      L.e("socket error");
    }
  }

  private synchronized void stopHeartbeat() {
    if (heartbeat != null && !heartbeat.isUnsubscribed()) {
      heartbeat.unsubscribe();
    }
  }

  private void emit(String event, Object object) {
    if (object == null || socket == null) {
      return;
    }
    socket.emit(event, object);
  }

  private void chatAction(String action, @Nullable String chatId) {
    if (socket == null) {
      return;
    }
    if (chatId == null || !ready.get()) {
      return;
    }
    String message = String.format("/user-chats/%s", chatId);
    emit(action, message);
  }

  // Events

  private Emitter.Listener onTyping = objects -> {
    L.d("onTyping: " + objects[0].toString());
    Typing typing = parseJson(objects[0].toString(), Typing.class);

    if (typing != null) {
      typing.setCreatedAt(TimeUtils.getCurrentTime());

      if (Const.TYPING_START.equals(typing.getAction())) {
        TypingStore.get().hostTypingState.upsert(typing);
      } else if (Const.TYPING_STOP.equals(typing.getAction())) {
        TypingStore.get().hostTypingState.remove(typing);
      }
    }
  };

  private Emitter.Listener onConnect = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onConnect: ");
      try {
        error.set(false);
        forceDisconnect.set(false);
        attemptCount.set(0);
        authentication();
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
    }
  };

  private Emitter.Listener onConnectTimeout = objects -> {
    L.d("onConnectTimeout");
    SocketStore.get().socketState.set(SocketStatus.ERROR);
  };

  private Emitter.Listener onConnectError = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      ready.set(false);
      enqueueReconnect();

      error.set(true);

      SocketStore.get().socketState.set(SocketStatus.ERROR);

      try {
        String message = "";
        if (objects[0] instanceof EngineIOException) {
          EngineIOException exception = (EngineIOException) objects[0];
          message = exception.getMessage();
        }
        if (objects[0] instanceof SocketIOException) {
          SocketIOException exception = (SocketIOException) objects[0];
          message = exception.getMessage();
        }
        L.e("onConnectError: " + message);
      } catch (Exception ex) {
        L.e(ex.getMessage());
      }
    }
  };

  private Emitter.Listener onConnecting = objects -> {
    L.d("onConnecting");
//      setSocketStatus(SocketStatus.CONNECTING);
  };

  private Emitter.Listener onDisconnect = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.d("onDisconnect: " + objects[0] + " " + forceDisconnect.get());
      ready.set(false);
      stopHeartbeat();

      if (!forceDisconnect.get()) {
        enqueueReconnect();

        SocketStore.get().socketState.set(SocketStatus.ERROR);
      } else {
        SocketStore.get().socketState.set(SocketStatus.DISCONNECTED);
      }

      Action.invoke(ActionType.SOCKET_DISCONNECTED);
    }
  };

  private Emitter.Listener onError = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      try {
        EngineIOException exception = (EngineIOException) objects[0];
        L.e("onError: " + objects.length + " " + exception.getMessage());
      } catch (Exception e) {
        L.e("onError: " + objects.length + " " + objects[0].toString());
      }

      error.set(true);

      SocketStore.get().socketState.set(SocketStatus.ERROR);

      ready.set(false);
      enqueueReconnect();
    }
  };

  private Emitter.Listener onPing = objects -> {
    //L.d("onPing");
  };

  private Emitter.Listener onPong = objects -> {
    //L.d("onPong: " + (Long)objects[0]);
  };

  private Emitter.Listener onReconnect = objects -> L.d("onReconnect: " + (int) objects[0]);

  private Emitter.Listener onReconnectAttempt = objects -> L.d("onReconnectAttempt: " + (int) objects[0]);

  private Emitter.Listener onReconnectError = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      SocketIOException exception = (SocketIOException) objects[0];
      L.e("onReconnectError: " + exception.getMessage());
      enqueueReconnect();
      ready.set(false);

      SocketStore.get().socketState.set(SocketStatus.ERROR);
    }
  };

  private Emitter.Listener onReconnectFailed = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.e("onReconnectFailed");
      enqueueReconnect();
      ready.set(false);

      SocketStore.get().socketState.set(SocketStatus.ERROR);
    }
  };

  private Emitter.Listener onReconnecting = objects -> L.d("onReconnecting: " + (int) objects[0]);

  private Emitter.Listener onAuthenticated = objects -> L.d("onAuthenticated: " + objects[0]);

  private Emitter.Listener onReady = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      if (Display.isLocked()) {
        L.d("Locked. disconnect.");
        disconnect();
      } else {
        L.d("Ready");
        ready.set(true);
        startHeartbeat();
        clearReconnectQueue();

        SocketStore.get().socketState.set(SocketStatus.READY);
      }
    }
  };

  private Emitter.Listener onJoined = objects -> {
    L.d("onJoined: " + objects[0]);
    try {
      String[] split = StringUtils.split((String) objects[0], '/');

      if (split[1] != null) {
        post(split[1]);
      }
    } catch (Exception ex) {
      L.e(ex.getMessage());
    }
  };

  private Emitter.Listener onLeaved = objects -> {
    L.d("onLeaved: " + objects[0]);
    try {
      String[] split = StringUtils.split((String) objects[0], '/');
    } catch (Exception ex) {
      L.e(ex.getMessage());
    }
  };

  private Emitter.Listener onUnauthorized = new Emitter.Listener() {
    @Override
    public void call(Object... objects) {
      L.e("onUnauthorized: " + objects[0].toString());
      ready.set(false);
      forceDisconnect.set(true);
      socket.disconnect();
    }
  };

  private Emitter.Listener onPush = objects -> {
    L.d("onPush: " + objects[0].toString());

    try {
      JSONObject json = (JSONObject) objects[0];
      String type = json.getString(Const.FIELD_SOCKET_TYPE);

      if (json.has(Const.FIELD_SOCKET_REFERS)) {
        JSONObject refers = json.getJSONObject(Const.FIELD_SOCKET_REFERS);
        if (refers.has(Const.FIELD_SOCKET_BOT)) {
          BotStore.get().bots.upsert(parseJson(refers.getString(Const.FIELD_SOCKET_BOT), Bot.class));
        }
        if (refers.has(Const.FIELD_SOCKET_MANAGER)) {
          ManagerStore.get().managers.upsert(parseJson(refers.getString(Const.FIELD_SOCKET_MANAGER), Manager.class));
        }
        if (refers.has(Const.FIELD_SOCKET_USER_CHAT)) {
          UserChatStore.get().userChats.upsert(parseJson(refers.getString(Const.FIELD_SOCKET_USER_CHAT), UserChat.class));
        }
        if (refers.has(Const.FIELD_SOCKET_USER)) {
          UserStore.get().user.set(parseJson(refers.getString(Const.FIELD_SOCKET_USER), User.class), false);
        }
      }

      if (Message.CLASSNAME.equals(type)) {
        Message message = parseJson(json.getString(Const.FIELD_SOCKET_ENTITY), Message.class);
        if (message != null) {
          Activity topActivity = GlobalStore.get().topActivity.get();
          if (!ChannelUtils.isChannelPluginActivity(topActivity)) {
            InAppPushStore.get().inAppPushMessage.set(message);
          }
          UserChatStore.get().messages.upsert(message);
        }
      }
    } catch (Exception e) {
      L.e(e.getMessage());
    }
  };

  private String getTag(JSONObject json) {
    try {
      String type = json.getString(Const.FIELD_SOCKET_TYPE);
      String id = json.getJSONObject(Const.FIELD_SOCKET_ENTITY).getString(Const.FIELD_SOCKET_ID);

      return String.format("%s (%s): ", type, id);
    } catch (Exception ex) {
      return "";
    }
  }

  private Emitter.Listener onCreate = objects -> {
    L.d("+ onCreate) " + getTag((JSONObject) objects[0]) + objects[0].toString());
    onMessage((JSONObject) objects[0], true);
  };

  private Emitter.Listener onUpdate = objects -> {
    L.d("~ onUpdate) " + getTag((JSONObject) objects[0]) + objects[0].toString());
    onMessage((JSONObject) objects[0], true);
  };

  private Emitter.Listener onDelete = objects -> {
    L.d("- onDelete) " + getTag((JSONObject) objects[0]) + objects[0].toString());
    onMessage((JSONObject) objects[0], false);
  };

  private void onMessage(JSONObject json, boolean upsert) {
    try {
      if (!json.has(Const.FIELD_SOCKET_TYPE) || !json.has(Const.FIELD_SOCKET_ENTITY)) {
        return;
      }

      String entity = json.getString(Const.FIELD_SOCKET_ENTITY);
      String type = json.getString(Const.FIELD_SOCKET_TYPE);

      switch (type) {
        case Bot.CLASSNAME:
          Bot bot = parseJson(entity, Bot.class);
          if (bot != null) {
            BotStore.get().bots.upsert(bot);
          }
          break;

        case Manager.CLASSNAME:
          Manager manager = parseJson(entity, Manager.class);
          if (manager == null) {
            return;
          }
          ManagerStore.get().managers.upsert(manager);
          break;

        case Message.CLASSNAME:
          Message message = parseJson(entity, Message.class);
          if (message == null) {
            return;
          }

          if (json.has(Const.FIELD_SOCKET_REFERS)) {
            JSONObject refers = json.getJSONObject(Const.FIELD_SOCKET_REFERS);

            if (refers.has(Const.FIELD_SOCKET_BOT)) {
              Bot botRefers = parseJson(refers.getString(Const.FIELD_SOCKET_BOT), Bot.class);
              BotStore.get().bots.upsert(botRefers);
            }
          }

          UserChatStore.get().messages.upsert(message);

          RxBus.post(message);

          if (message.getProfileBot() != null) {
            ProfileBotStore.get().requestFocus.set(true);
          }

          if (message.isDeleted()) {
            Message inAppPushMessage = InAppPushStore.get().inAppPushMessage.get();

            if (inAppPushMessage != null && inAppPushMessage.getId() != null && inAppPushMessage.getId().equals(message.getId())) {
              InAppPushStore.get().inAppPushMessage.reset();
            }
          }
          break;

        case UserChat.CLASSNAME:
          UserChat userChat = parseJson(entity, UserChat.class);

          if (userChat == null) {
            return;
          }

          if (json.has(Const.FIELD_SOCKET_REFERS)) {
            JSONObject refers = json.getJSONObject(Const.FIELD_SOCKET_REFERS);
            if (refers.has(Const.FIELD_SOCKET_MESSAGE)) {
              Message referMessage = parseJson(refers.getString(Const.FIELD_SOCKET_MESSAGE), Message.class);
              UserChatStore.get().messages.upsert(referMessage);
            }
            if (refers.has(Const.FIELD_SOCKET_MANAGER)) {
              Manager referManager = parseJson(refers.getString(Const.FIELD_SOCKET_MANAGER), Manager.class);
              ManagerStore.get().managers.upsert(referManager);
            }
            if (refers.has(Const.FIELD_SOCKET_BOT)) {
              Bot referBot = parseJson(refers.getString(Const.FIELD_SOCKET_BOT), Bot.class);
              BotStore.get().bots.upsert(referBot);
            }
          }
          UserChatStore.get().userChats.upsert(userChat);
          break;

        case Session.CLASSNAME:
          Session session = parseJson(entity, Session.class);
          if (session == null) {
            return;
          }

          if (json.has(Const.FIELD_SOCKET_REFERS)) {
            JSONObject refers = json.getJSONObject(Const.FIELD_SOCKET_REFERS);
            if (refers.has(Const.FIELD_SOCKET_BOT)) {
              Bot referBot = parseJson(refers.getString(Const.FIELD_SOCKET_BOT), Bot.class);
              BotStore.get().bots.upsert(referBot);
            }
            if (refers.has(Const.FIELD_SOCKET_MANAGER)) {
              Manager referManager = parseJson(refers.getString(Const.FIELD_SOCKET_MANAGER), Manager.class);
              ManagerStore.get().managers.upsert(referManager);
            }
          }

          if (upsert) {
            UserChatStore.get().sessions.upsert(session);
          } else {
            UserChatStore.get().sessions.remove(session);
          }
          break;

        case Channel.CLASSNAME:
          Channel channel = parseJson(entity, Channel.class);

          if (channel != null) {
            ChannelStore.get().channelState.set(channel);
          }
          break;

        case User.CLASSNAME:
          User user = parseJson(entity, User.class);
          if (user != null) {
            UserStore.get().user.set(user);
          }
          break;
      }
    } catch (Exception ex) {
      L.e(ex.getMessage());
    }
  }

  private <T> T parseJson(String entity, Class<T> target) {
    if (entity == null) {
      return null;
    }
    try {
      return ParseUtils.getCustomGson().fromJson(entity, target);
    } catch (Exception ex) {
      ex.printStackTrace();
      return null;
    }
  }
}

