package com.zoyi.channel.plugin.android.network;

import android.content.ContentResolver;
import android.content.Context;
import android.net.Uri;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.webkit.MimeTypeMap;

import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.util.io.FilenameUtils;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.okhttp3.MediaType;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.okio.BufferedSink;
import com.zoyi.rx.functions.Action1;

import java.io.IOException;
import java.io.InputStream;

public class ProgressRequestBody extends RequestBody {

  private static final int DEFAULT_BUFFER_SIZE = 2048;

  private Uri uri;

  @Nullable
  private String name;

  @Nullable
  private MediaType mediaType;

  @Nullable
  private ContentResolver contentResolver;

  private long size;

  @NonNull
  private Action1<Integer> progressAction;

  public ProgressRequestBody(Uri uri, @Nullable String name, long size, @NonNull Action1<Integer> progressAction) {
    this.uri = uri;
    this.name = name;
    this.size = size;
    this.mediaType = Optional.ofNullable(name)
        .map(FilenameUtils::getExtension)
        .map(extension -> MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension))
        .map(MediaType::parse)
        .orElse(MediaType.parse("application/octet-stream"));
    this.contentResolver = Optional.ofNullable(ChannelIO.getAppContext()).map(Context::getContentResolver).orElse(null);
    this.progressAction = progressAction;
  }

  @NonNull
  public String getName() {
    return Optional.ofNullable(this.name).orElse(Const.DEFAULT_FILE_NAME);
  }

  @Nullable
  @Override
  public MediaType contentType() {
    return mediaType;
  }

  @Override
  public long contentLength() {
    return size;
  }

  @Override
  public void writeTo(BufferedSink sink) throws IOException {
    long fileLength = contentLength();
    byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];

    if (contentResolver != null) {
      try (InputStream in = contentResolver.openInputStream(uri)) {
        long uploaded = 0L;
        int read;
        int lastProgressPercentUpdate = -1;
        read = in.read(buffer);
        while (read != -1) {
          uploaded += read;
          sink.write(buffer, 0, read);
          read = in.read(buffer);

          int progress = Math.min((int) (uploaded * 100 / Math.max(1, fileLength)), 100);
          //prevent publishing too many updates, which slows upload, by checking if the upload has progressed by at least 1 percent
          if (progress != lastProgressPercentUpdate || progress == 100) {
            // publish progress
            progressAction.call(progress);
            lastProgressPercentUpdate = progress;
          }
        }
      }
    }
  }
}
