package com.zoyi.channel.plugin.android.activity.chat.viewholder;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageContentClickListener;
import com.zoyi.channel.plugin.android.activity.chat.model.SendFileItem;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManager;
import com.zoyi.channel.plugin.android.base.adapter.BaseViewHolder;
import com.zoyi.channel.plugin.android.enumerate.SendingState;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.com.bumptech.glide.Glide;
import com.zoyi.com.bumptech.glide.load.engine.DiskCacheStrategy;
import com.zoyi.rx.Subscription;

import java.util.Locale;

public class UploadStateHolder extends BaseViewHolder {

  private ImageView imageThumbnail;
  private TextView textFileName;
  private ImageView imageCancel;

  private ImageView imageRetry;

  private TextView textFileQueue;
  private ProgressBar progressBar;

  @Nullable
  private Subscription progressSubscription;

  @Nullable
  private Subscription waitingCountSubscription;

  @Nullable
  private SendFileItem fileItem;

  private UploadStateHolder(@NonNull View itemView, final @Nullable OnMessageContentClickListener onMessageContentClickListener) {
    super(itemView);
    imageThumbnail = itemView.findViewById(R.id.ch_image_thumbnail);
    textFileName = itemView.findViewById(R.id.ch_text_file_name);
    imageCancel = itemView.findViewById(R.id.ch_image_cancel);

    imageRetry = itemView.findViewById(R.id.ch_image_retry);

    textFileQueue = itemView.findViewById(R.id.ch_text_file_queue);
    progressBar = itemView.findViewById(R.id.ch_progress);

    imageCancel.setOnClickListener(view -> {
      if (onMessageContentClickListener != null) {
        onMessageContentClickListener.onCancelClick(fileItem);
      }
    });

    imageRetry.setOnClickListener(view -> {
      if (onMessageContentClickListener != null && fileItem != null) {
        onMessageContentClickListener.onResendButtonClick(fileItem);
      }
    });
  }

  public void bind(SendFileItem fileItem) {
    clear();

    this.fileItem = fileItem;
    textFileName.setText(fileItem.getName());

    Glide.with(itemView.getContext())
        .load(fileItem.getUri())
        .diskCacheStrategy(DiskCacheStrategy.RESULT)
        .placeholder(R.drawable.ch_plugin_image_placeholder)
        .error(R.drawable.ch_plugin_image_placeholder)
        .centerCrop()
        .into(imageThumbnail);

    if (fileItem.getState() == SendingState.FAIL) {
      imageRetry.setVisibility(View.VISIBLE);

      textFileQueue.setVisibility(View.GONE);
      progressBar.setVisibility(View.GONE);
    } else {
      imageRetry.setVisibility(View.GONE);

      progressBar.setVisibility(View.VISIBLE);
      progressBar.setProgress(ChatManager.get(fileItem.getChatId()).getCurrentFileUploadProgress());

      progressSubscription = ChatManager.get(fileItem.getChatId())
          .getUploadProgressObservable()
          .subscribe(progress -> progressBar.setProgress(progress));

      setWaitingCount(ChatManager.get(fileItem.getChatId()).getWaitingFileCount());

      waitingCountSubscription = ChatManager.get(fileItem.getChatId())
          .getWaitingCountObservable()
          .subscribe(this::setWaitingCount);
    }
  }

  private void setWaitingCount(int waitingCount) {
    if (waitingCount == 0) {
      textFileQueue.setVisibility(View.GONE);
    } else {
      textFileQueue.setVisibility(View.VISIBLE);
      textFileQueue.setText(
          String.format(
              Locale.getDefault(),
              ResUtils.getString(itemView.getContext(), "ch.file_upload.wait_count"),
              waitingCount
          )
      );
    }
  }

  @Override
  protected void clear() {
    fileItem = null;

    progressBar.setProgress(0);

    if (progressSubscription != null && !progressSubscription.isUnsubscribed()) {
      progressSubscription.unsubscribe();
      progressSubscription = null;
    }

    if (waitingCountSubscription != null && !waitingCountSubscription.isUnsubscribed()) {
      waitingCountSubscription.unsubscribe();
      waitingCountSubscription = null;
    }

    if (imageThumbnail != null) {
      Glide.clear(imageThumbnail);
    }
  }

  public static UploadStateHolder newInstance(
      ViewGroup parent,
      @Nullable OnMessageContentClickListener onMessageContentClickListener
  ) {
    return new UploadStateHolder(
        LayoutInflater.from(parent.getContext()).inflate(R.layout.ch_plugin_item_upload_state, parent, false),
        onMessageContentClickListener
    );
  }
}
