package com.zoyi.channel.plugin.android.activity.chat3;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import com.zoyi.channel.plugin.android.action.*;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat3.contract.ChatAdapterContract;
import com.zoyi.channel.plugin.android.activity.chat3.contract.ChatContract;
import com.zoyi.channel.plugin.android.activity.chat3.listener.MessageSendListener;
import com.zoyi.channel.plugin.android.activity.chat3.manager.chat.ChatManager;
import com.zoyi.channel.plugin.android.activity.chat3.manager.chat.ChatManagerImpl;
import com.zoyi.channel.plugin.android.activity.chat3.model.ProfileBotMessageItem;
import com.zoyi.channel.plugin.android.activity.common.chat.ChatContentType;
import com.zoyi.channel.plugin.android.base.AbstractAdapterPresenter;
import com.zoyi.channel.plugin.android.enumerate.*;
import com.zoyi.channel.plugin.android.global.Api;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.ActionButton;
import com.zoyi.channel.plugin.android.model.PushBotItem;
import com.zoyi.channel.plugin.android.model.TranslationInfo;
import com.zoyi.channel.plugin.android.model.entity.ProfileEntity;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.MessagesWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.selector2.*;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store2.*;
import com.zoyi.channel.plugin.android.store2.binder.Binder;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.rx.Subscription;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import static com.zoyi.channel.plugin.android.global.Const.ACTION_TYPE_CLOSE;
import static com.zoyi.channel.plugin.android.global.Const.ACTION_TYPE_SOLVE;

public class ChatPresenter
    extends AbstractAdapterPresenter<ChatContract.View, ChatAdapterContract.View, ChatAdapterContract.Model>
    implements ChatContract.Presenter, MessageSendListener {

  private ChatContentType contentType;
  @Nullable
  private String contentId;

  @Nullable
  private Binder welcomeBinder;

  @Nullable
  private Binder welcomeAcceptInputBinder;

  @Nullable
  private Subscription joinSubscription;

  @Nullable
  private UserChat userChat;

  @Nullable
  private Session lastReadSession;

  @Nullable
  private List<SendingMessageItem> tempItems;

  public ChatPresenter(
      ChatContract.View view,
      ChatAdapterContract.View adapterView,
      ChatAdapterContract.Model adapterModel,
      ChatContentType contentType,
      @Nullable String contentId
  ) {
    super(view, adapterView, adapterModel);

    this.contentType = contentType;
    this.contentId = contentId;
  }

  private ChatManagerImpl getChatManager() {
    return ChatManager.get(contentType == ChatContentType.USER_CHAT ? contentId : null);
  }

  @Override
  public void init() {
    switch (contentType) {
      case USER_CHAT:
        view.onFetchStateChange(FetchState.LOADING);

        if (contentId != null) {
          bindUserChatEventReceiver();
        }
        break;

      case PUSH_BOT_CHAT:
        this.tempItems = new ArrayList<>();

        PushBotItem pushBotItem = PushBotStore.get().pushBots.get(contentId);

        if (pushBotItem != null) {
          pushBotItem.read();
          PushBotStore.get().pushBots.upsert(pushBotItem);

          adapterModel.setInitMessage(new PushBotMessageItem(pushBotItem));

          view.onWelcomeStateChange(ChatContentType.PUSH_BOT_CHAT, false);
          view.onFetchStateChange(FetchState.COMPLETE);
        } else {
          view.onFetchStateChange(FetchState.FAILED);
        }
        break;

      case NONE:
        this.tempItems = new ArrayList<>();

        view.onFetchStateChange(FetchState.LOADING);

        Plugin plugin = PluginStore.get().pluginState.get();
        if (plugin != null && plugin.getId() != null) {
          bindWelcomeMessage();

          PluginAction.fetchPlugin(plugin.getId(), false);
          SupportBotAction.fetchSupportBot(plugin.getId(), false);
        }
        break;
    }
  }

  @Override
  protected void handleAction(ActionType actionType) {
    switch (actionType) {
      case SOCKET_DISCONNECTED:
      case SHUTDOWN:
        ChatStore.get().messages.clear();
        ChatStore.get().backwardId.set(null);
        break;
    }
  }

  // welcome message, push bot logic

  private void bindWelcomeMessage() {
    welcomeBinder = ChatSelector.bindWelcomeMessage((fetchState, supportBotEntry, welcomeMessage) -> {
      switch (fetchState) {
        case FAILED:
          view.onFetchStateChange(FetchState.FAILED);
          break;

        case COMPLETE:
          if (supportBotEntry != null) {
            this.contentType = ChatContentType.SUPPORT_BOT_CHAT;
            this.contentId = supportBotEntry.getId();

            adapterModel.setInitMessage(new SupportBotMessageItem(supportBotEntry));

            view.onWelcomeStateChange(ChatContentType.SUPPORT_BOT_CHAT, false);
            view.onFetchStateChange(FetchState.COMPLETE);

          } else if (welcomeMessage != null) {
            adapterModel.setInitMessage(new WelcomeMessageItem(welcomeMessage));

            bindWelcomeAcceptInput();
            view.onFetchStateChange(FetchState.COMPLETE);
          } else {
            view.onFetchStateChange(FetchState.FAILED);
          }

          if(welcomeBinder != null) {
            welcomeBinder.unbind();
          }
          break;
      }
    });
  }

  private void bindWelcomeAcceptInput() {
    if (welcomeAcceptInputBinder == null) {
      welcomeAcceptInputBinder = ChannelSelector.bindAcceptAction(acceptInput ->
          view.onWelcomeStateChange(ChatContentType.NONE, acceptInput)
      );
    }
  }

  // chat init logic for normal chat

  private void createUserChat() {
    if (contentType == ChatContentType.NONE && !Api.isRunning(ActionType.CREATE_USER_CHAT)) {
      Plugin plugin = PluginStore.get().pluginState.get();
      String pluginId = plugin != null ? plugin.getId() : null;

      if (pluginId != null) {
        ChatAction.createUserChat(pluginId, new RestSubscriber<UserChatWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            setTempSendingItemsFail();
          }

          @Override
          public void onSuccess(@NonNull UserChatWrapper repo) {
            repo.update();

            handleUserChatInit(repo.getUserChat(), repo.getMessage());
          }
        });
      }
    }
  }

  // chat init logic for push bot

  @Override
  public void createPushBotChat() {
    if (contentType == ChatContentType.PUSH_BOT_CHAT && !Api.isRunning(ActionType.CREATE_USER_CHAT)) {
      String pushBotId = contentId;

      view.setPushBotSaveButtonVisibility(false);

      SendingMessageItem pushBotMessageItem = new SendingMessageItem(null);
      pushBotMessageItem.setActionBundle(Const.ACTION_TYPE_KEEP_PUSH_BOT, null);
      pushBotMessageItem.setText(":+1:");

      addTempSendingItem(pushBotMessageItem);

      PushBotAction.createPushBotUserChat(pushBotId, new RestSubscriber<UserChatWrapper>() {
        @Override
        public void onError(RetrofitException error) {
          view.setPushBotSaveButtonVisibility(true);

          removeTempSendingItem(pushBotMessageItem);
        }

        @Override
        public void onSuccess(@NonNull UserChatWrapper repo) {
          repo.update();

          PushBotItem item = PushBotStore.get().pushBots.get(pushBotId);

          if (item != null) {
            item.remove();
            PushBotStore.get().pushBots.upsert(item);
          }

          handleUserChatInit(repo.getUserChat(), repo.getMessage());
        }
      });
    }
  }

  // chat init logic for support bot

  private void createSupportBotUserChat(SendingMessageItem item) {
    if (contentType == ChatContentType.SUPPORT_BOT_CHAT && !Api.isRunning(ActionType.CREATE_USER_CHAT)) {
      String supportBotId = contentId;

      addTempSendingItem(item);

      SupportBotAction.createSupportBotUserChat(supportBotId, new RestSubscriber<UserChatWrapper>() {
        @Override
        public void onError(RetrofitException error) {
          removeTempSendingItem(item);
        }

        @Override
        public void onSuccess(@NonNull UserChatWrapper repo) {
          repo.update();

          item.updateMessageOnActionInput(repo.getMessage());

          handleUserChatInit(repo.getUserChat(), repo.getMessage());
        }
      });
    }
  }

  // Add temporary sending items before chat created

  private void addTempSendingItem(SendingMessageItem item) {
    if (item != null) {
      addTempSendingItems(Collections.singleton(item));
    }
  }

  private void addTempSendingItems(Collection<SendingMessageItem> items) {
    if (tempItems != null && items != null && items.size() > 0) {
      adapterModel.addMessageItems(items);
      tempItems.addAll(items);
    }
  }

  private boolean removeTempSendingItem(SendingMessageItem item) {
    if (tempItems != null && item != null) {
      adapterModel.removeMessageItem(item);
      return tempItems.remove(item);
    }
    return false;
  }

  private void setTempSendingItemsChatId(String chatId) {
    if (tempItems != null) {
      for (SendingMessageItem item : tempItems) {
        item.setChatId(chatId);
      }
      adapterModel.addMessageItems(tempItems);
    }
  }

  private void setTempSendingItemsFail() {
    if (tempItems != null) {
      for (SendingMessageItem item : tempItems) {
        item.setSendingState(SendingState.FAIL);
      }
      adapterModel.addMessageItems(tempItems);
    }
  }

  // Call when user chat create by user manually

  private void handleUserChatInit(UserChat userChat, Message message) {
    if (welcomeBinder != null) {
      welcomeBinder.unbind();
      welcomeBinder = null;
    }
    if (welcomeAcceptInputBinder != null) {
      welcomeAcceptInputBinder.unbind();
      welcomeAcceptInputBinder = null;
    }
    if (userChat != null && userChat.getId() != null) {
      this.userChat = userChat;

      this.contentType = ChatContentType.USER_CHAT;
      this.contentId = userChat.getId();

      adapterModel.replaceInitMessageItem(message);

      // set chat id to pre-sending items
      setTempSendingItemsChatId(userChat.getId());

      // attach chat id for init actions and messages
      ChatStore.get().messages.attachChatId(contentId);

      bindUserChatEventReceiver();
    }
  }

  // user chat logic

  private void bindUserChatEventReceiver() {
    // subscribe only once

    if (joinSubscription == null && contentType == ChatContentType.USER_CHAT && contentId != null) {
      getChatManager().attachListener(this);

      bind(SocketSelector.bindSocket(socketStatus -> {
        if (socketStatus == SocketStatus.READY) {
          fetchUserChat();
        }
      }));

      bind(ChatSelector.bindChat(contentId, (userChat, acceptInput, temporaryInputOpened) -> {
        // when after select 'keep going chatting' and chat state changed, we recognize who send message after that situation.

        if (temporaryInputOpened && !Const.USER_CHAT_STATE_SOLVED.equals(userChat.getState())) {
          ChatStore.get().temporaryInputOpenedState.set(false);
        }

        ChatAction.read(contentId);

        this.userChat = userChat;

        view.onChatStateChange(userChat, acceptInput, temporaryInputOpened);
      }));

      bind(ProfileBotSelector.bindProfileBotActivation(isActivated -> {
        if (isActivated) {
          view.setInputDim(true);
        }
      }));

      bind(ChatSelector.bindTyping(isTyping -> SocketManager.typing(
          Typing.create(
              isTyping ? Const.TYPING_START : Const.TYPING_STOP,
              contentId
          )
      )));

      bind(ChatSelector.bindHostTyping(contentId, typings -> adapterModel.setTypings(typings)));

      bind(MessageSelector.bindMessages((initMessages, upsertMessages) -> {
        if (initMessages != null) {
          // set messages by initialize

          if (tempItems != null && tempItems.size() > 0) {
            getChatManager().send(tempItems);

            tempItems.clear();
            tempItems = null;
          }

          adapterModel.setMessages(initMessages, getChatManager().getUnsentItems(), lastReadSession);

          view.scrollToBottom();
          view.onFetchStateChange(FetchState.COMPLETE);

          ChatAction.read(contentId);
        } else if (upsertMessages != null) {
          // add messages

          boolean isScrollOnBottom = view.isScrollOnBottom();
          Long lastKey = adapterModel.getLastMessageItemKey();

          adapterModel.addMessages(upsertMessages, lastReadSession);

          Long newLastKey = adapterModel.getLastMessageItemKey();

          if (isScrollOnBottom) {
            view.scrollToBottom();
          } else if (lastKey != null && newLastKey != null && lastKey < newLastKey) {
            // Show new message alert in bottom

            Message message = adapterModel.getLastMessage();

            if (message != null) {
              ProfileEntity profileEntity = ProfileSelector.getProfile(message.getPersonType(), message.getPersonId());

              if (profileEntity != null) {
                view.showNewMessageAlert(profileEntity);
              }
            }
          }
        }
      }));

      joinSubscription = SocketManager.observable().subscribe(joinedChatId -> {
        if (contentId != null && contentId.equals(joinedChatId)) {
          ChatStore.get().messages.attachChatId(contentId);
          fetchInitMessages();
        }
      });
    }
  }

  // data fetch

  private void fetchUserChat() {
    ChatAction.fetchUserChat(contentId, new RestSubscriber<UserChatWrapper>() {
      @Override
      public void onError(RetrofitException error) {
        view.onFetchStateChange(FetchState.FAILED);
      }

      @Override
      public void onSuccess(@NonNull UserChatWrapper repo) {
        repo.update();

        ChatStore.get().messages.clear();
        ChatStore.get().backwardId.set(null);

        lastReadSession = repo.getSession();

        SocketManager.joinChat(repo.getUserChat().getId());
      }
    });
  }

  private void fetchInitMessages() {
    ChatAction.fetchMessages(contentId, new RestSubscriber<MessagesWrapper>() {
      @Override
      public void onError(RetrofitException error) {
        SocketManager.leaveChat(contentId);
        view.onFetchStateChange(FetchState.FAILED);
      }

      @Override
      public void onSuccess(@NonNull MessagesWrapper repo) {
        repo.update();
        ChatStore.get().messages.set(repo.getMessages());
        ChatStore.get().backwardId.set(repo.getNext());
      }
    });
  }

  @Override
  public void fetchBackwardMessages() {
    String backwardId = ChatStore.get().backwardId.get();

    if (!Api.isRunning(ActionType.FETCH_BACKWARD_MESSAGES) && backwardId != null) {
      ChatAction.fetchBackwardMessages(contentId, backwardId, new RestSubscriber<MessagesWrapper>() {
        @Override
        public void onSuccess(@NonNull MessagesWrapper repo) {
          repo.update();
          ChatStore.get().messages.add(repo.getMessages());
          ChatStore.get().backwardId.set(repo.getNext());
        }
      });
    }
  }

  // handle action

  @Override
  public void sendText(String message) {
    if (tempItems != null) {
      addTempSendingItem(new SendingMessageItem(null).setText(message));
      createUserChat();
    } else if (contentType == ChatContentType.USER_CHAT && contentId != null) {
      SendingMessageItem item = new SendingMessageItem(contentId).setText(message);
      adapterModel.addMessageItem(item);
      getChatManager().send(item);
    }

    view.scrollToBottom();
  }

  @Override
  public void uploadFiles(ArrayList<String> paths) {
    String chatId = contentType == ChatContentType.USER_CHAT ? contentId : null;

    if (paths != null && paths.size() > 0) {
      List<SendingMessageItem> items = new ArrayList<>();

      for (String filePath : paths) {
        if (filePath != null) {
          items.add(new SendingMessageItem(chatId).setFilePath(filePath));
        }
      }

      if (tempItems != null) {
        addTempSendingItems(items);
        createUserChat();
      } else if (contentType == ChatContentType.USER_CHAT && contentId != null) {
        adapterModel.addMessageItems(items);
        getChatManager().send(items);
      }

      view.scrollToBottom();
    }
  }

  @Override
  public void translate(Message message) {
    TranslationInfo translation = TranslationStore.get().translation.get(message.getId());

    if (translation == null) {
      TranslationStore.get().translation.upsert(TranslationInfo.createProgressStateInfo(message.getId()));
      adapterModel.addMessage(message);

      TranslateAction.translate(message, () -> adapterModel.addMessage(message));
    } else if (translation.getState() == TranslationState.TRANSLATED) {
      TranslationStore.get().translation.upsert(translation.setState(TranslationState.ORIGIN));
      adapterModel.addMessage(message);
    } else if (translation.getState() == TranslationState.ORIGIN && translation.getTranslatedMessage() != null) {
      TranslationStore.get().translation.upsert(translation.setState(TranslationState.TRANSLATED));
      adapterModel.addMessage(message);
    }
  }

  @Override
  public void onActionClick(@NonNull String actionType, @NonNull ActionButton actionButton) {
    if (contentType == ChatContentType.SUPPORT_BOT_CHAT && Const.ACTION_TYPE_SUPPORT_BOT.equals(actionType)) {
      createSupportBotUserChat(new SendingMessageItem(null).setActionBundle(actionType, actionButton));
      return;
    }

    if (CompareUtils.exists(actionType, ACTION_TYPE_SOLVE, ACTION_TYPE_CLOSE) &&
        CompareUtils.isSame(actionButton.getKey(), Const.ACTION_KEY_REOPEN)
    ) {
      reopenChat(actionButton);
      return;
    }

    if (contentType == ChatContentType.USER_CHAT && contentId != null) {
      SendingMessageItem messageItem = new SendingMessageItem(contentId).setActionBundle(actionType, actionButton);
      adapterModel.addMessageItem(messageItem);

      getChatManager().send(messageItem);
    }

    view.scrollToBottom();
  }

  @Override
  public void resend(SendingMessageItem item) {
    if (tempItems != null && removeTempSendingItem(item)) {
      tempItems.add(SendingMessageItem.copyContentFrom(item));

      createUserChat();
    } else if (contentType == ChatContentType.USER_CHAT) {
      getChatManager().resend(item);
    }

    view.scrollToBottom();
  }

  @Override
  public void removeFailedItem(SendingMessageItem item) {
    if (tempItems == null || !removeTempSendingItem(item)) {
      adapterModel.removeMessageItem(item);
      getChatManager().removeFailedItem(item);
    }
  }

  // profile bot action

  @Override
  public void requestCountries(ProfileBotMessageItem item) {
    if (ProfileBotStore.get().requestState.get() != FetchState.LOADING) {
      List<Country> countries = CountryStore.get().countries.get();

      if (countries != null) {
        view.showCountryDialog(countries, item);
      } else if (!Api.isRunning(ActionType.FETCH_COUNTRIES)) {
        CountryAction.fetchCountries(new RestSubscriber<List<Country>>() {
          @Override
          public void onSuccess(@NonNull List<Country> countries) {
            CountryStore.get().countries.set(countries);

            if (countries.size() > 0) {
              view.showCountryDialog(countries, item);
            }
          }
        });
      }
    }
  }

  @Override
  public void selectCountry(String countryCode, ProfileBotMessageItem item) {
    if (ProfileBotStore.get().requestState.get() != FetchState.LOADING &&
        Const.PROFILE_MOBILE_NUMBER_KEY.equals(ProfileBotStore.get().inputKey.get())
    ) {
      Object inputValue = ProfileBotStore.get().inputValue.get();

      if (inputValue instanceof MobileNumber) {
        ((MobileNumber) inputValue).setCountryCode(Integer.valueOf(countryCode));

        ProfileBotStore.get().inputValue.set(inputValue);
        ProfileBotStore.get().requestState.set(FetchState.COMPLETE);

        adapterModel.addMessageItem(item);
      }
    }
  }

  @Override
  public void updateProfile(ProfileBotMessageItem item) {
    ProfileBotStore.get().requestState.set(FetchState.LOADING);

    adapterModel.addMessageItem(item);

    GuestAction.updateProfileBot(
        item.getMessageId(),
        repo -> {
          ProfileBotStore.get().requestFocus.set(true);
          ProfileBotStore.get().requestState.set(FetchState.COMPLETE);
          ProfileBotStore.get().inputKey.set(null);
          ProfileBotStore.get().inputValue.set(null);

          ChatStore.get().messages.add(repo.getMessage());
        }, () -> {
          ProfileBotStore.get().requestFocus.set(true);
          ProfileBotStore.get().requestState.set(FetchState.FAILED);

          adapterModel.addMessageItem(item);
        }
    );
  }

  // private functions

  private void reopenChat(@NonNull ActionButton actionButton) {
    Message message = actionButton.getMessage();
    message.clearAction();
    adapterModel.addMessage(message);

    ChatStore.get().temporaryInputOpenedState.set(true);

    view.scrollToBottom();
  }

  // release

  @Override
  public void release() {
    super.release();

    getChatManager().clearListener();

    if (welcomeBinder != null) {
      welcomeBinder.unbind();
      welcomeBinder = null;
    }
    if (welcomeAcceptInputBinder != null) {
      welcomeAcceptInputBinder.unbind();
      welcomeAcceptInputBinder = null;
    }
    if (joinSubscription != null && !joinSubscription.isUnsubscribed()) {
      joinSubscription.unsubscribe();
      joinSubscription = null;
    }

    if (contentType == ChatContentType.USER_CHAT && contentId != null) {
      SocketManager.leaveChat(contentId);
    }

    TypingStore.get().myTypingState.upsert(Typing.dummy());

    if (contentType == ChatContentType.USER_CHAT && contentId != null) {
      SocketManager.typing(Typing.create(Const.TYPING_STOP, contentId));
    }
  }

  @Override
  public void onFail(SendingMessageItem item) {
    adapterModel.addMessageItem(item);
  }

  @Override
  public void onResend(SendingMessageItem newItem, SendingMessageItem oldItem) {
    adapterModel.replaceMessageItem(newItem, oldItem);
  }
}
