package com.zoyi.channel.plugin.android;

import android.content.Context;
import android.graphics.Color;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;

import com.zoyi.channel.plugin.android.selector.ChannelIOSelector;
import com.zoyi.channel.plugin.android.util.Utils;
import com.zoyi.channel.plugin.android.view.external.fab.FloatingActionButton;
import com.zoyi.channel.plugin.android.view.layout.CHTextView;

import static android.widget.RelativeLayout.ALIGN_PARENT_BOTTOM;
import static android.widget.RelativeLayout.ALIGN_PARENT_END;
import static android.widget.RelativeLayout.ALIGN_PARENT_LEFT;
import static android.widget.RelativeLayout.ALIGN_PARENT_RIGHT;
import static android.widget.RelativeLayout.ALIGN_PARENT_START;
import static android.widget.RelativeLayout.ALIGN_PARENT_TOP;

/**
 * Created by jerry on 2018. 11. 16..
 */

public class ChannelLauncherView extends FrameLayout {

  public static final int BOTTOM_LEFT = 0;
  public static final int BOTTOM_RIGHT = 1;

  private Context context;

  private ViewGroup root;
  private FloatingActionButton buttonFloatingAction;
  private CHTextView textBadgeCount;

  private int defaultMargin;
  private boolean visible = false;

  public ChannelLauncherView(@NonNull Context context) {
    super(context);
    initView(context);
  }

  public ChannelLauncherView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    initView(context);
  }

  public ChannelLauncherView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    initView(context);
  }

  private void initView(final Context context) {
    this.context = context;

    View view = LayoutInflater.from(context).inflate(R.layout.ch_plugin_view_launcher, this, true);

    root = view.findViewById(R.id.ch_root_launcher_view);
    textBadgeCount = view.findViewById(R.id.text_badge);
    buttonFloatingAction = view.findViewById(R.id.fab);
    buttonFloatingAction.setColorRipple(Color.WHITE);
    buttonFloatingAction.setOnClickListener(new OnClickListener() {
      @Override
      public void onClick(View v) {
        ChannelIO.open(context);
      }
    });

    defaultMargin = context.getResources().getDimensionPixelSize(R.dimen.ch_default_channel_button_margin);
  }

  public void show() {
    visible = true;
    root.setVisibility(VISIBLE);
  }

  public void hide() {
    visible = false;
    root.setVisibility(GONE);
  }

  public void setLauncherLayout(@Nullable LauncherConfig launcherConfig) {
    RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);

    params.addRule(ALIGN_PARENT_BOTTOM);
    params.addRule(ALIGN_PARENT_TOP, 0);

    switch (getGravity(launcherConfig)) {
      case BOTTOM_LEFT:
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
          params.addRule(ALIGN_PARENT_START);
          params.addRule(ALIGN_PARENT_END, 0);
        }
        params.addRule(ALIGN_PARENT_LEFT);
        params.addRule(ALIGN_PARENT_RIGHT, 0);
        params.setMargins(getXMargin(), 0, 0, getYMargin());
        break;

      case BOTTOM_RIGHT:
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
          params.addRule(RelativeLayout.ALIGN_PARENT_START, 0);
          params.addRule(RelativeLayout.ALIGN_PARENT_END);
        }
        params.addRule(ALIGN_PARENT_LEFT, 0);
        params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
        params.setMargins(0, 0, getXMargin(), getYMargin());
        break;
    }

    setLayoutParams(params);
    requestLayout();
    textBadgeCount.bringToFront();
  }

  public void setLauncherBackgroundColor(int backgroundColor, int gradientColor) {
    buttonFloatingAction.setColorNormalWithGradient(backgroundColor, gradientColor);
    buttonFloatingAction.setColorPressed(Utils.addBlackMask(backgroundColor));
  }

  public void setBadgeCount(int chatCount) {
    if (chatCount <= 0) {
      textBadgeCount.setVisibility(INVISIBLE);
    } else {
      textBadgeCount.setVisibility(VISIBLE);
      textBadgeCount.setText(Utils.getCount(chatCount, true));
    }
  }

  public void setLauncherIcon(int launcherIcon) {
    buttonFloatingAction.setImageResource(launcherIcon);
  }

  public boolean isVisible() {
    return visible;
  }

  private int getXMargin() {
    LauncherConfig config = ChannelIOSelector.getLauncherConfig();

    if (config != null) {
      return (int) Utils.dpToPx(context, config.getXMargin());
    } else {
      return defaultMargin;
    }
  }

  private int getYMargin() {
    LauncherConfig config = ChannelIOSelector.getLauncherConfig();

    if (config != null) {
      return (int) Utils.dpToPx(context, config.getYMargin());
    } else {
      return defaultMargin;
    }
  }

  private int getGravity(@Nullable LauncherConfig launcherConfig) {
    if (launcherConfig != null) {
      return launcherConfig.getLauncherGravity();
    } else {
      return BOTTOM_RIGHT;
    }
  }
}
