package com.zoyi.channel.plugin.android.activity.userchat_list;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.view.ViewGroup;

import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.activity.userchat_list.contract.UserChatAdapterContract;
import com.zoyi.channel.plugin.android.activity.userchat_list.holder.UserChatHolder;
import com.zoyi.channel.plugin.android.activity.userchat_list.listener.OnUserChatClickListener;
import com.zoyi.channel.plugin.android.activity.userchat_list.model.ChatListItem;
import com.zoyi.channel.plugin.android.activity.userchat_list.model.ChatContentType;
import com.zoyi.channel.plugin.android.activity.userchat_list.model.PushBotChatItem;
import com.zoyi.channel.plugin.android.activity.userchat_list.model.UserChatItem;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.PushBotItem;
import com.zoyi.channel.plugin.android.model.rest.Session;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.selector.PushBotSelector;
import com.zoyi.channel.plugin.android.selector.UserChatSelector;
import com.zoyi.channel.plugin.android.util.CompareUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by mika on 2016. 12. 6..
 */
class UserChatListAdapter
    extends RecyclerView.Adapter<UserChatHolder>
    implements UserChatAdapterContract.View, UserChatAdapterContract.Model {

  @Nullable
  private OnUserChatClickListener listener;

  @NonNull
  private Map<String, ChatListItem> userChatItems;
  @NonNull
  private SortedList<ChatListItem> items;

  UserChatListAdapter() {
    userChatItems = new HashMap<>();
    items = new SortedList<>(ChatListItem.class, new SortedListCallback<ChatListItem>(this, true));
  }

  @Override
  public void setOnUserChatClickListener(OnUserChatClickListener clickListener) {
    this.listener = clickListener;
  }

  @Override
  public void setUserChats(List<UserChat> userChats) {
    clear();

    items.beginBatchedUpdates();

    for (UserChat userChat : userChats) {
      if (canUpsertUserChat(userChat)) {
        addNewChatListItem(new UserChatItem(userChat));
      }
    }

    addLocalChats();

    items.endBatchedUpdates();
  }

  public void clear() {
    userChatItems.clear();
    items.clear();
  }

  @Override
  public UserChatHolder onCreateViewHolder(ViewGroup parent, int viewType) {
    return UserChatHolder.newInstance(parent, listener);
  }

  @Override
  public void onBindViewHolder(UserChatHolder holder, int position) {
    ChatListItem chatListItem = items.get(position);
    if (chatListItem != null) {
      holder.setChatListItem(chatListItem);
    }
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  @Override
  public void updateManager(String managerId) {
    for (int i = 0; i < items.size(); i++) {
      if (items.get(i) instanceof UserChatItem) {
        UserChatItem item = (UserChatItem) items.get(i);

        if (CompareUtils.isSame(item.getHostType(), Const.MANAGER) && CompareUtils.isSame(item.getHostId(), managerId)) {
          items.updateItemAt(i, item);
        }
      }
    }
  }

  @Override
  public void updateSession(@NonNull Session session) {
    upsertUserChat(UserChatSelector.get(session.getChatId()));
  }

  @Override
  public void upsertPushBotItem(@NonNull PushBotItem pushBotItem) {
    PushBotChatItem pushBotChatItem = new PushBotChatItem(pushBotItem);
    int position = items.indexOf(pushBotChatItem);

    if (position >= 0) {
      items.updateItemAt(position, pushBotChatItem);
    } else {
      items.add(pushBotChatItem);
    }
  }

  @Override
  public void removePushBotItem(@NonNull PushBotItem pushBotItem) {
    items.remove(new PushBotChatItem(pushBotItem));
  }

  @Override
  public void upsertUserChat(@Nullable UserChat userChat) {
    if (canUpsertUserChat(userChat) && userChat != null) {
      ChatListItem oldUserChatItem = userChatItems.get(userChat.getId());
      ChatListItem newUserChatItem = new UserChatItem(userChat);

      if (oldUserChatItem != null) {
        updateUserChatItem(oldUserChatItem, newUserChatItem);
      } else {
        addNewChatListItem(newUserChatItem);
      }
    }
  }

  @Override
  public void removeUserChat(@NonNull String chatId) {
    if (userChatItems.containsKey(chatId)) {
      items.remove(userChatItems.get(chatId));
      userChatItems.remove(chatId);
    }
  }

  private boolean canUpsertUserChat(@Nullable UserChat userChat) {
    return userChat != null && userChat.getId() != null && !(userChat.isStateReady() && userChat.getLastMessageId() == null);
  }

  private void updateUserChatItem(@NonNull ChatListItem oldChatListItem, @NonNull ChatListItem newChatListItem) {
    int position = items.indexOf(oldChatListItem);

    if (position >= 0) {
      if (newChatListItem.getItemType() == ChatContentType.USER_CHAT) {
        userChatItems.put(newChatListItem.getId(), newChatListItem);
      }
      items.updateItemAt(position, newChatListItem);
    }
  }

  private void addNewChatListItem(@NonNull ChatListItem chatListItem) {
    if (chatListItem.getItemType() == ChatContentType.USER_CHAT) {
      userChatItems.put(chatListItem.getId(), chatListItem);
    }
    items.add(chatListItem);
  }

  private void addLocalChats() {
    List<PushBotItem> pushBotItems = PushBotSelector.getVisiblePushBotItems();

    for (PushBotItem pushBotItem : pushBotItems) {
      addNewChatListItem(new PushBotChatItem(pushBotItem));
    }
  }

  @Nullable
  @Override
  public ChatListItem getFirstChatListItem() {
    if (items.size() >= 1) {
      return items.get(0);
    }

    return null;
  }

  @Nullable
  @Override
  public ChatListItem getUserChatItem(@NonNull String id) {
    return userChatItems.get(id);
  }
}
