package com.zoyi.channel.plugin.android.view.layout;

import android.annotation.TargetApi;
import android.content.Context;
import android.graphics.Color;
import android.graphics.Typeface;
import android.os.Build;
import android.support.annotation.ColorInt;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity;
import com.zoyi.channel.plugin.android.util.Utils;
import com.zoyi.channel.plugin.android.view.CHHeaderView;
import com.zoyi.channel.plugin.android.view.handler.ButtonTouchHandler;
import com.zoyi.channel.plugin.android.view.layout.MenuButton.ActionType;
import com.zoyi.channel.plugin.android.view.layout.MenuButton.MenuState;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by mika on 9/28/16.
 */
public class BigBar extends LinearLayout {

  private static final int MENU_BUTTON_WIDTH = 40;
  private static final int MENU_TEXT_WIDTH = 32;
  private static final int TITLE_LAYOUT_MARGIN = 16;

  private RelativeLayout layout;
  private LinearLayout leftLayout;
  private LinearLayout rightLayout;
  private CHTextView textTitle;
  private FrameLayout titleLayout;
  private FrameLayout frameCustomHeader;

  private Context context;
  @Nullable
  private BaseActivity activity;
  private Animation pushAnimation;
  private Map<Integer, CHTextView> textMap;
  private List<MenuButton> buttons;
  private boolean init = false;

  @ColorInt private int backgroundColor = Color.BLACK;
  @ColorInt private int textColor = Color.WHITE;

  @Nullable
  private CHHeaderView headerView;

  public BigBar(Context context) {
    super(context);
    init(context);
  }

  public BigBar(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context);
  }

  public BigBar(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context);
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP)
  public BigBar(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
    super(context, attrs, defStyleAttr, defStyleRes);
    init(context);
  }

  private void init(Context context) {
    this.context = context;

    LayoutInflater.from(context).inflate(R.layout.ch_plugin_layout_bigbar, this, true);

    layout = findViewById(R.id.bar_layout);
    leftLayout = findViewById(R.id.layout_left);
    rightLayout = findViewById(R.id.layout_right);
    textTitle =  findViewById(R.id.bar_title_main);
    titleLayout = findViewById(R.id.layout_title);
    frameCustomHeader = findViewById(R.id.frameBigbarHeader);

    textMap = new HashMap<>();
    buttons = new ArrayList<>();
    pushAnimation = AnimationUtils.loadAnimation(getContext(), R.anim.ch_plugin_push);
    pushAnimation.setAnimationListener(new Animation.AnimationListener() {
      @Override
      public void onAnimationStart(android.view.animation.Animation animation) {
//        layoutPush.setVisibility(VISIBLE);
      }

      @Override
      public void onAnimationEnd(android.view.animation.Animation animation) {
//        hidePush();
      }

      @Override
      public void onAnimationRepeat(android.view.animation.Animation animation) {
      }
    });
  }

  public BigBar withActivity(@Nullable BaseActivity activity) {
    this.activity = activity;
    return this;
  }

  public BigBar addMenu(ActionType actionType, MenuPosition menuPosition) {
    return addMenu(actionType, menuPosition, MenuState.VISIBLE);
  }

  public BigBar addMenu(final ActionType actionType, MenuPosition menuPosition, MenuState menuState) {
    MenuButton menuButton = new MenuButton(context);
    menuButton.setAction(actionType);
    menuButton.setColor(textColor);
    menuButton.setMenuPosition(menuPosition);
    menuButton.setOnTouchListener(new ButtonTouchHandler(menuButton));
    menuButton.setOnClickListener(new OnClickListener() {
      @Override
      public void onClick(View v) {
        eventRaised(actionType);
      }
    });
    setViewState(menuButton, menuState);
    addView(menuButton, menuPosition);
    buttons.add(menuButton);

    return this;
  }

  public BigBar addText(MenuPosition menuPosition) {
    return addText(0, menuPosition, "");
  }

  public BigBar addText(MenuPosition menuPosition, String string) {
    return addText(0, menuPosition, string);
  }

  private BigBar addText(int id, MenuPosition menuPosition, String string) {
    if (!textMap.containsKey(id)) {
      LinearLayout.LayoutParams params = new LinearLayout.LayoutParams(
          (int) Utils.dpToPx(context, 32),
          LayoutParams.WRAP_CONTENT);
      params.weight = 1.0f;
      params.leftMargin = MenuPosition.LEFT.equals(menuPosition) ? (int) Utils.dpToPx(context, 8) * -1 : 0;
      params.rightMargin = MenuPosition.RIGHT.equals(menuPosition) ? (int) Utils.dpToPx(context, 8) * -1 : 0;
      params.gravity = Gravity.CENTER_VERTICAL
          | (MenuPosition.LEFT.equals(menuPosition) ? Gravity.START : Gravity.END);

      CHTextView textView = new CHTextView(context);
      textView.setTypeface(textView.getTypeface(), Typeface.BOLD);
      textView.setTextSize(TypedValue.COMPLEX_UNIT_SP, 14);
      textView.setTextColor(textColor);
      textView.setLayoutParams(params);
      textView.setText(string);

      textMap.put(id, textView);
      addView(textView, menuPosition);
    }
    return this;
  }

  public void setText(int number) {
    if (textMap.size() == 1) {
      setText(textMap.entrySet().iterator().next().getKey(), number);
    }
  }

  public void setText(int id, int number) {
    CHTextView chTextView = textMap.get(id);
    if (textMap.containsKey(id) && chTextView != null) {
      chTextView.setText(Utils.getCount(number, true));
    }
  }

  public void setMenuState(ActionType actionType, MenuState buttonState) {
    if (actionType == null) {
      return;
    }
    for (MenuButton menuButton : buttons) {
      if (actionType.equals(menuButton.getAction())) {
        setViewState(menuButton, buttonState);
      }
    }
  }

  public BigBar setHeaderView(@Nullable CHHeaderView headerView) {
    if (headerView != null) {
      textTitle.setVisibility(GONE);

      this.headerView = headerView;
      headerView.onTextColorChange(textColor);
      frameCustomHeader.addView(headerView);
      frameCustomHeader.setVisibility(VISIBLE);
    }
    return this;
  }

  public BigBar setTitle(String text) {
    textTitle.setText(text);
    return this;
  }

  //Don't write hard coding. Use string resource key
  public BigBar setTitleByKey(String textKey) {
    textTitle.setTextByKey(textKey);
    return this;
  }

  public BigBar setTheme(@ColorInt int backgroundColor, @ColorInt int textColor) {
    this.backgroundColor = backgroundColor;
    this.textColor = textColor;

    layout.setBackgroundColor(backgroundColor);
    textTitle.setTextColor(textColor);

    if (headerView != null) {
      headerView.onTextColorChange(textColor);
    }
    for (CHTextView textView : textMap.values()) {
      textView.setTextColor(textColor);
    }
    for (MenuButton menuButton : buttons) {
      menuButton.setColor(textColor);
    }
    return this;
  }

  public void hide() {
    layout.setVisibility(GONE);
  }

  public void build() {
    if (init) { return; }
    init = true;

    setTitleLayoutMargin();
  }

  private void setViewState(View view, MenuState menuState) {
    if (menuState == null) {
      return;
    }

    switch (menuState) {
      case VISIBLE:
        view.setVisibility(VISIBLE);
        view.setAlpha(1.0f);
        view.setEnabled(true);
        break;

      case DISABLED:
        view.setVisibility(VISIBLE);
        view.setAlpha(0.3f);
        view.setEnabled(false);
        break;

      case HIDDEN:
        view.setVisibility(GONE);
        break;
    }
    setTitleLayoutMargin();
  }

  private int getMarginForTitle() {
    int left = TITLE_LAYOUT_MARGIN, right = TITLE_LAYOUT_MARGIN;

    if (textMap.size() > 0) {
      left += MENU_TEXT_WIDTH;
    }

    for (MenuButton button : buttons) {
      switch (button.getMenuPosition()) {
        case LEFT:
          if (button.isVisible()) {
            left += MENU_BUTTON_WIDTH;
          }
          break;

        case RIGHT:
          if (button.isVisible()) {
            right += MENU_BUTTON_WIDTH;
          }
          break;
      }
    }

    return (int) Math.max(Utils.dpToPx(context, left), Utils.dpToPx(context, right));
  }

  private void setTitleLayoutMargin() {
    int margin = getMarginForTitle();
    RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) titleLayout.getLayoutParams();

    layoutParams.leftMargin = margin;
    layoutParams.rightMargin = margin;

    titleLayout.setLayoutParams(layoutParams);
  }

  private void addView(View view, MenuPosition menuPosition) {
    if (MenuPosition.LEFT.equals(menuPosition)) {
      leftLayout.addView(view);
    } else {
      rightLayout.addView(view);
    }
  }

  private void eventRaised(ActionType actionType) {
    if (activity != null) {
      activity.optionClicked(actionType);
    }
  }

  // internal class

  public enum  MenuPosition {
    LEFT, RIGHT
  }
}
