package com.zoyi.channel.plugin.android.activity.userchat_list;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.ChannelApiManager;
import com.zoyi.channel.plugin.android.activity.userchat_list.contract.UserChatAdapterContract;
import com.zoyi.channel.plugin.android.activity.userchat_list.contract.UserChatListContract;
import com.zoyi.channel.plugin.android.activity.userchat_list.model.ChatListItem;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.StoreType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.enumerate.UpdateType;
import com.zoyi.channel.plugin.android.enumerate.UserChatListState;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.PushBotItem;
import com.zoyi.channel.plugin.android.model.entity.Entity;
import com.zoyi.channel.plugin.android.model.rest.Manager;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.rest.Session;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.model.wrapper.*;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.selector.PluginSelector;
import com.zoyi.channel.plugin.android.selector.PushBotSelector;
import com.zoyi.channel.plugin.android.selector.UiStateSelector;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.PushBotItemStore;
import com.zoyi.channel.plugin.android.store.Store;
import com.zoyi.channel.plugin.android.store.SupportBotStore;
import com.zoyi.channel.plugin.android.store.UserChatStore;
import com.zoyi.channel.plugin.android.util.ConditionUtils;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.rx.Observable;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Func1;
import com.zoyi.rx.functions.Func2;
import com.zoyi.rx.functions.Func3;
import com.zoyi.rx.schedulers.Schedulers;

public class UserChatListPresenter implements UserChatListContract.Presenter {

  private UserChatListContract.View view;
  private UserChatAdapterContract.View adapterView;
  private UserChatAdapterContract.Model adapterModel;

  private Context context;

  private UserChatListState state = UserChatListState.WAITING_SOCKET;
  private boolean init = true;
  private boolean chatLoading = false;

  public UserChatListPresenter(Context context) {
    this.context = context;
  }

  @Initializer
  @Override
  public void setView(UserChatListContract.View view) {
    this.view = view;
  }

  @Initializer
  @Override
  public void setAdapterView(UserChatAdapterContract.View adapterView) {
    this.adapterView = adapterView;
  }

  @Initializer
  @Override
  public void setAdapterModel(UserChatAdapterContract.Model adapterModel) {
    this.adapterModel = adapterModel;
  }

  @Override
  public void refresh() {
    switch (state) {
      case WAITING_SOCKET:
        if (SocketManager.isReady()) {
          getUserChats();
        } else {
          SocketManager.connect();
        }
        break;

      case READY:
      case USER_CHAT_LIST_NOT_LOADED:
        getUserChats();
        break;
    }
  }

  @Override
  public void redrawList() {
    adapterView.notifyDataSetChanged();
  }

  private void setState(UserChatListState state) {
    this.state = state;
  }

  private boolean canUpdate() {
    return UserChatListState.READY.equals(state);
  }

  private void getUserChats() {
    view.setErrorView(false);

    if (SocketManager.isReady()) {
      view.showProgress(ResUtils.getString(context, "ch.loading_information"));

      setState(UserChatListState.USER_CHAT_LIST_LOADING);
      fetchUserChats();
    } else {
      setState(UserChatListState.WAITING_SOCKET);
      SocketManager.reconnect();
    }
  }

  @Override
  public void receiveCommand(Command command, @Nullable Object object) {
    switch (command) {
      case READY:
        view.setErrorView(false);

        if (UserChatListState.WAITING_SOCKET.equals(state)) {
          getUserChats();
        } else {
          refresh();
        }
        break;

      case SOCKET_DISCONNECTED:
        setState(UserChatListState.WAITING_SOCKET);
        Store.getInstance(UserChatStore.class).setCanUpdate(false);
        if (object != null && object instanceof Boolean) {
          boolean force = (boolean) object;
          if (!force) {
            view.setErrorView(true);
          }
        }
        break;

      case SOCKET_ERROR:
        setState(UserChatListState.WAITING_SOCKET);
        Store.getInstance(UserChatStore.class).setCanUpdate(false);
        view.setErrorView(true);
        break;
    }
  }

  @Override
  public void receiveData(Entity entity, boolean upsert) {
    if (entity != null && canUpdate()) {
      if (entity instanceof Session) {
        adapterModel.updateSession((Session) entity);
      } else if (entity instanceof Manager) {
        adapterModel.updateManager(entity.getId());
      }
    }
  }

  @Override
  public void receiveStoreEvent(StoreType storeType, UpdateType updateType, @Nullable Entity entity) {
    if (entity != null && canUpdate()) {
      switch (storeType) {
        case USER_CHAT:
          UserChat userChat = (UserChat) entity;

          switch (updateType) {
            case DELETE:
              adapterModel.removeUserChat(userChat.getId());
              break;

            case UPDATE:
              adapterModel.upsertUserChat(userChat);
              break;
          }
          break;

        case PUSH_BOT_ITEM:
          PushBotItem pushBotItem = (PushBotItem) entity;

          if (updateType == UpdateType.UPDATE) {
            if (pushBotItem.shouldShowToList()) {
              adapterModel.upsertPushBotItem(pushBotItem);
            } else {
              adapterModel.removePushBotItem(pushBotItem);
            }
          }
          break;
      }
    }
  }

  private void fetchUserChats() {
    ChannelApiManager.get()
        .getUserChats(Const.UPDATED_AT, Const.DESC, 50, UiStateSelector.shouldShowClosedChat())
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<UserChatsWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            init = false;
            view.hideProgress();
            handleError();
          }

          @Override
          public void onNext(UserChatsWrapper repo) {
            view.hideProgress();

            if (repo != null) {
              repo.set();

              setState(UserChatListState.READY);

              adapterModel.setUserChats(repo.getUserChats());

              view.userChatsFetched(init);
              init = false;

            } else {
              handleError();
            }
          }
        });
  }

  public void handleError() {
    if (UserChatListState.USER_CHAT_LIST_LOADING.equals(state)) {
      setState(UserChatListState.USER_CHAT_LIST_NOT_LOADED);

      view.setErrorView(true);
    }
  }

  @Override
  public void removeUserChat(@NonNull final String chatId) {
    view.showProgress(ResUtils.getString(context, "ch.chat.delete_progress"));

    ChannelApiManager.get()
        .removeUserChat(chatId)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<Void>() {
          @Override
          public void onError(RetrofitException error) {
          }

          @Override
          public void onNext(Void aVoid) {
            UserChat userChat = Store.getInstance(UserChatStore.class).get(chatId);
            if (userChat != null) {
              userChat.setState(Const.USER_CHAT_STATE_REMOVED);
              Store.getInstance(UserChatStore.class).add(userChat);
            }

            adapterModel.removeUserChat(chatId);
            view.completeRemoveChat();
          }
        });
  }

  @Override
  public void removePushBotChat(@NonNull String pushBotId) {
    PushBotItem pushBotItem = PushBotSelector.get(pushBotId);

    if (pushBotItem != null) {
      pushBotItem.remove();
      Store.getInstance(PushBotItemStore.class).add(pushBotItem);
      adapterModel.removePushBotItem(pushBotItem);

      view.completeRemoveChat();
    }
  }

  @Override
  public void fetchPreloadData(@Nullable final ChatListItem chatListItem, final Transition transition) {
    Plugin plugin = PluginSelector.getPlugin();
    if (plugin == null) {
      return;
    }

    if (chatLoading) {
      return;
    }

    chatLoading = true;

    Observable.zip(
        ChannelApiManager.get().getFollowingManagers(),
        ChannelApiManager.get().getPlugin(plugin.getId()),
        ChannelApiManager.get().getSupportBots(plugin.getId()),
        new Func3<ManagersWrapper, PluginWrapper, SupportBotsWrapper, ChatPreloadWrapper>() {
          @Override
          public ChatPreloadWrapper call(
              ManagersWrapper managersWrapper,
              PluginWrapper pluginWrapper,
              SupportBotsWrapper supportBotsWrapper
          ) {
            return new ChatPreloadWrapper(managersWrapper, pluginWrapper, supportBotsWrapper);
          }
        })
        .flatMap(new Func1<ChatPreloadWrapper, Observable<ChatPreloadWrapper>>() {
          @Override
          public Observable<ChatPreloadWrapper> call(ChatPreloadWrapper chatPreloadWrapper) {
            String validSupportBotId = ConditionUtils.getFirstValidSupportBotId(chatPreloadWrapper.getSupportBotsWrapper());
            if (validSupportBotId != null) {
              return Observable.zip(
                  Observable.just(chatPreloadWrapper),
                  ChannelApiManager.get().getSupportBotEntry(validSupportBotId),
                  new Func2<ChatPreloadWrapper, SupportBotEntry, ChatPreloadWrapper>() {
                    @Override
                    public ChatPreloadWrapper call(ChatPreloadWrapper chatPreloadWrapper, SupportBotEntry supportBotEntry) {
                      return chatPreloadWrapper.updateSupportBotEntry(supportBotEntry);
                    }
                  });
            }
            return Observable.just(chatPreloadWrapper);
          }
        })
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<ChatPreloadWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            L.e(error.getMessage());
            chatLoading = false;
          }

          @Override
          public void onNext(ChatPreloadWrapper repo) {
            if (repo != null) {
              repo.update();
              Store.getInstance(SupportBotStore.class).setSupportBotEntry(repo.getSupportBotEntry());
              view.successFetchPreloadData(chatListItem, transition);
            }
            chatLoading = false;
          }
        });
  }

  @Nullable
  @Override
  public ChatListItem getUserChatItem(@NonNull String id) {
    return adapterModel.getUserChatItem(id);
  }
}
