package com.zoyi.channel.plugin.android.activity.chat;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;

import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.event.TypingBus;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.TimeUtils;
import com.zoyi.channel.plugin.android.view.layout.WatchedEditText;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscriber;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;
import com.zoyi.rx.subscriptions.CompositeSubscription;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class TypingManager implements TextWatcher {

  private static final int TYPING_THROTTLE = 1000;

  @Nullable
  private String chatId;

  private long typingStartTime = 0;
  private Map<String, Subscription> typingSubscriptionItems;
  private CompositeSubscription ttlSubscriptions;

  TypingManager(@Nullable String chatId, WatchedEditText editText) {
    this(editText);
    this.chatId = chatId;
  }

  TypingManager(WatchedEditText editText) {
    editText.addTextChangedListener(this);

    typingSubscriptionItems = new HashMap<>();
    ttlSubscriptions = new CompositeSubscription();
  }

  public void release() {
    if (ttlSubscriptions != null && !ttlSubscriptions.isUnsubscribed()) {
      ttlSubscriptions.unsubscribe();
    }
    typingSubscriptionItems.clear();
    stopTyping();
  }

  public void setChatId(@Nullable String chatId) {
    this.chatId = chatId;
  }

  public boolean isReady() {
    return chatId != null;
  }

  @Override
  public void beforeTextChanged(CharSequence s, int start, int count, int after) {}

  @Override
  public void onTextChanged(CharSequence s, int start, int before, int count) {
    if (isReady()) {
      String msg = s.toString();
      if (TextUtils.isEmpty(msg)) {
        stopTyping();
      } else if (typingStartTime == 0 || TimeUtils.getCurrentTime() - typingStartTime > TYPING_THROTTLE) {
        startTyping();
      }
    }
  }

  @Override
  public void afterTextChanged(Editable s) {}

  private void startTyping() {
    if (isReady()) {
      typingStartTime = TimeUtils.getCurrentTime();

      if (chatId != null) {
        SocketManager.typing(new Typing("start", chatId, UserChat.CLASSNAME));
      }
    }
  }

  private void stopTyping() {
    if (isReady()) {
      typingStartTime = 0;

      if (chatId != null) {
        SocketManager.typing(new Typing("stop", chatId, UserChat.CLASSNAME));
      }
    }
  }

  void refreshTypingExpiresAt(Typing typing) {
    if (isReady()) {
      Subscription subscription = typingSubscriptionItems.get(typing.getKey());
      if (subscription != null) {
        removeTTLSubscription(typing);
      }

      if ("start".equals(typing.getAction())) {
        addTTLSubscription(typing);
      }
    }
  }

  private void removeTTLSubscription(Typing typing) {
    Subscription subscription = typingSubscriptionItems.remove(typing.getKey());
    if (subscription != null) {
      ttlSubscriptions.remove(subscription);
    }
  }

  private void addTTLSubscription(final Typing typing) {
    Subscription subscription = Observable.timer(
        Typing.TTL,
        TimeUnit.SECONDS,
        Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new Subscriber<Long>() {
          @Override
          public void onCompleted() {}

          @Override
          public void onError(Throwable e) {}

          @Override
          public void onNext(Long aLong) {
            if (typing != null && typing.isExpired()) {
              typing.setAction("stop");
              RxBus.post(new TypingBus(typing));
            }
          }
        });

    typingSubscriptionItems.put(typing.getKey(), subscription);
    ttlSubscriptions.add(subscription);
  }
}
