package com.zoyi.channel.plugin.android.activity.chat;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v7.util.SortedList;
import android.support.v7.widget.RecyclerView;
import android.view.ViewGroup;
import com.zoyi.channel.plugin.android.activity.base.SortedListCallback;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageContentClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnProfileBotUpdateListener;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.chat.viewholder.*;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.store.ImageFileStore;
import com.zoyi.channel.plugin.android.store.Store;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.L;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Created by mika on 2018. 11. 6..
 */

public class ChatAdapter2 extends RecyclerView.Adapter<RecyclerView.ViewHolder>
    implements ChatAdapterContract2.Model, ChatAdapterContract2.View {

  private SortedList<MessageItem> items;

  @Nullable
  private OnProfileBotUpdateListener onProfileBotUpdateListener;
  @Nullable
  private OnMessageContentClickListener onMessageContentClickListener;

  // Specific message item
  @NonNull
  private TypingItem typingItem;

  @Nullable
  private InitMessageItem initMessageItem;
  @Nullable
  private DateItem initMessageDateItem;

  @Nullable
  private NewMessageItem newMessageItem;

  // chat edit text focus flag
  private boolean editTextFocused = false;

  ChatAdapter2() {
    items = new SortedList<>(MessageItem.class, new SortedListCallback<MessageItem>(this, false));

    // Typing layout must always exists
    typingItem = new TypingItem();
    items.add(typingItem);
  }

  public void setOnProfileBotUpdateListener(@Nullable OnProfileBotUpdateListener onProfileBotUpdateListener) {
    this.onProfileBotUpdateListener = onProfileBotUpdateListener;
  }

  public void setOnMessageContentClickListener(@Nullable OnMessageContentClickListener onMessageContentClickListener) {
    this.onMessageContentClickListener = onMessageContentClickListener;
  }

  @Override
  public void setMessages(List<Message> messages) {
    resetSpecialMessageItems();
    items.replaceAll(getItemsFromMessages(messages));
    items.add(typingItem);
  }

  @Override
  public int addMessage(Message message) {
    return addMessages(Collections.singletonList(message));
  }

  @Override
  public int addMessages(List<Message> messages) {
    int prevSize = items.size();
    List<MessageItem> newItems = getItemsFromMessages(messages);

    items.beginBatchedUpdates();

    for (MessageItem item : newItems) {
      items.add(item);
    }

    for (Message message : messages) {
      if (message == null) {
        continue;
      }
      if (CompareUtils.exists(message.getPersonType(), User.CLASSNAME, Veil.CLASSNAME)) {
        SendingMessageItem item = SendingMessageItem.createDummy(message.getRequestId());
        if (item != null) {
          items.remove(item);
        }
      }
    }

    items.endBatchedUpdates();

    return items.size() - prevSize;
  }

  @NonNull
  private List<MessageItem> getItemsFromMessages(List<Message> messages) {
    List<MessageItem> newItems = new ArrayList<>();

    for (Message message : messages) {
      if (message == null) {
        continue;
      }

      if (message.getLog() != null) {
        newItems.add(new LogMessageItem(message));
        continue;
      }

      if (message.getFile() != null && message.getFile().isImageFile()) {
        Store.getInstance(ImageFileStore.class).add(message.getFile());
      }

      newItems.add(new ChatMessageItem(message));
      newItems.add(new DateItem(message.getCreatedAt()));

      if (message.getProfileBot() != null) {
        newItems.add(new ProfileBotMessageItem(message));
      }
    }

    return newItems;
  }

  @Override
  public void addMessageItem(MessageItem item) {
    this.items.add(item);
  }

  @Override
  public void addMessageItems(List<? extends MessageItem> items) {
    this.items.addAll((Collection<MessageItem>) items);
  }

  @Override
  public void removeMessageItem(MessageItem item) {
    if (item != null) {
      items.remove(item);
    }
  }

  // Specific message item getter/setter

  @Override
  public void upsertTyping(Typing typing) {
    typingItem.addOrUpdateTyping(typing);
    addMessageItem(typingItem);
  }

  @Override
  public void removeTyping(Typing typing) {
    typingItem.removeTyping(typing);
    addMessageItem(typingItem);
  }

  @Override
  public void upsertInitMessage(InitMessageItem item) {
    items.beginBatchedUpdates();

    if (initMessageItem != null) {
      items.remove(initMessageItem);
      initMessageItem = null;
    }
    if (initMessageDateItem != null ) {
      items.remove(initMessageDateItem);
      initMessageDateItem = null;
    }

    if (item != null) {
      initMessageItem = item;
      items.add(item);

      initMessageDateItem = new DateItem(initMessageItem.getCreatedAt());
      items.add(initMessageDateItem);
    }

    items.endBatchedUpdates();
  }

  @Override
  public void replaceInitMessage(Message message) {
    items.beginBatchedUpdates();

    if (initMessageItem != null) {
      items.remove(initMessageItem);
      initMessageItem = null;
    }
    if (initMessageDateItem != null ) {
      items.remove(initMessageDateItem);
      initMessageDateItem = null;
    }

    for (MessageItem item : getItemsFromMessages(Collections.singletonList(message))) {
      items.add(item);
    }

    items.endBatchedUpdates();
  }

  @Override
  public void upsertNewMessageItem(long createdAt) {
    if (newMessageItem != null) {
      items.remove(newMessageItem);
    }
    newMessageItem = new NewMessageItem(createdAt);
    items.addAll(newMessageItem);
  }

  // RecyclerView adapter functions

  @Nullable
  @Override
  public RecyclerView.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
    MessageType type = MessageType.fromId(viewType);

    switch (type) {
      case NEW_MESSAGE_DIVIDER:
        return NewMessageHolder.newInstance(parent);

      case TYPING:
        return TypingHolder.newInstance(parent);

      case DATE:
        return DateDividerHolder.newInstance(parent);

      case LOG:
        return LogMessageHolder.newInstance(parent);

      case SENDING:
        return SendingMessageHolder.newInstance(parent, type, onMessageContentClickListener);

      case HOST:
      case GUEST:
        return ChatMessageHolder.newInstance(parent, type, onMessageContentClickListener);

      case PROFILE_BOT:
        return ProfileBotMessageHolder.newInstance(parent, onProfileBotUpdateListener);

      case WELCOME:
        return WelcomeMessageHolder.newInstance(parent);

      case SUPPORT_BOT:
        return SupportBotMessageHolder.newInstance(parent, onMessageContentClickListener);

      case PUSH_BOT:
        return PushBotMessageHolder.newInstance(parent, onMessageContentClickListener);

      default:
        return null;
    }
  }

  @Override
  public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {
    MessageItem item = items.get(position);
    ChatMessageItem chatMessageItem;

    switch (item.getType()) {
      case TYPING:
        TypingHolder typingHolder = (TypingHolder) holder;
        typingHolder.bind((TypingItem) item);
        break;

      case DATE:
        DateDividerHolder dateDividerHolder = (DateDividerHolder) holder;
        dateDividerHolder.bind((DateItem) item);
        break;

      case LOG:
        LogMessageHolder logMessageHolder = (LogMessageHolder) holder;
        logMessageHolder.bind((LogMessageItem) item);
        break;

      case SENDING:
        SendingMessageItem sendingMessageItem = (SendingMessageItem) item;
        SendingMessageHolder sendingHolder = (SendingMessageHolder) holder;
        sendingHolder.bind((SendingMessageItem) item, sendingMessageItem.isConnected(getItem(position - 1)));
        break;

      case HOST:
      case GUEST:
        chatMessageItem = (ChatMessageItem) item;
        ChatMessageHolder chatMessageHolder = (ChatMessageHolder) holder;
        chatMessageHolder.bind(
            chatMessageItem,
            chatMessageItem.isConnected(getItem(position - 1)),
            isLastPosition(position)
        );
        break;

      case PROFILE_BOT:
        ProfileBotMessageItem profileItem = (ProfileBotMessageItem) item;
        ProfileBotMessageHolder profileBotMessageHolder = (ProfileBotMessageHolder) holder;
        profileBotMessageHolder.bind(profileItem, isEditTextFocused());
        break;

      case WELCOME:
        WelcomeMessageItem welcomeMessageItem = (WelcomeMessageItem) item;
        WelcomeMessageHolder welcomeMessageHolder = (WelcomeMessageHolder) holder;
        welcomeMessageHolder.bind(welcomeMessageItem);
        break;

      case SUPPORT_BOT:
        SupportBotMessageItem supportBotMessageItem = (SupportBotMessageItem) item;
        SupportBotMessageHolder supportBotMessageHolder = (SupportBotMessageHolder) holder;
        supportBotMessageHolder.bind(supportBotMessageItem, isLastPosition(position));
        break;

      case PUSH_BOT:
        PushBotMessageItem pushBotMessageItem = (PushBotMessageItem) item;
        PushBotMessageHolder pushBotMessageHolder = (PushBotMessageHolder) holder;
        pushBotMessageHolder.bind(pushBotMessageItem);
        break;
    }
  }

  @Override
  public int getItemCount() {
    return items.size();
  }

  @Override
  public int getItemViewType(int position) {
    return items.get(position).getType().toInt();
  }

  // etc functions

  private void resetSpecialMessageItems() {
    initMessageItem = null;
    initMessageDateItem = null;
    newMessageItem = null;
  }

  @Nullable
  private MessageItem getItem(int position) {
    if (position >= 0 && position < items.size()) {
      return items.get(position);
    }
    return null;
  }

  private boolean isLastPosition(int position) {
    return position == getItemCount() - 2;
  }

  private boolean isEditTextFocused() {
    return editTextFocused;
  }

  void setEditTextFocused(boolean editTextFocused) {
    this.editTextFocused = editTextFocused;
  }
}
