package com.zoyi.channel.plugin.android.activity.chat.view.overlayheader;

import android.content.Context;
import android.support.annotation.Nullable;
import android.support.v7.widget.RecyclerView;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.view.handler.OnSizeChangeListener;

/**
 * Created by mika on 2018. 10. 24..
 */

public class OverlayHeaderRecyclerView extends FrameLayout implements View.OnTouchListener {

  private SizeObservableRecyclerView recyclerView;
  private SizeObservableFrameLayout bannerFrame;
  private SizeObservableFrameLayout rootFrame;

  @Nullable
  private OverlayHeaderCallback callback;
  @Nullable
  private OnHeaderStateChangeListener listener;

  private OverlayHeaderState state = OverlayHeaderState.HIDDEN;
  
  private int bannerHeight = 0;
  private float lastTouchY = 0;

  /**
   * Banner와 List가 붙어있을 때 true
   * true 일 경우에는 List toggle 시에 List가 따라다니게 됨
   */
  private boolean isBannerAttachedToList = false;

  public OverlayHeaderRecyclerView(Context context) {
    super(context);
    init(context);
  }

  public OverlayHeaderRecyclerView(Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context);
  }

  public OverlayHeaderRecyclerView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context);
  }

  private void init(Context context) {
    View view = LayoutInflater.from(context).inflate(R.layout.view_overlay_header_recycler_view, this, true);

    bannerFrame = (SizeObservableFrameLayout) view.findViewById(R.id.frameOverlayHeaderRecyclerViewHeader);
    bannerFrame.setOnSizeChangeListener(new OnSizeChangeListener() {
      @Override
      public void onSizeChange(int width, int height, int oldWidth, int oldHeight) {
        bannerHeight = height;
        applyTranslationY(height, oldHeight, false);
      }
    });

    recyclerView = (SizeObservableRecyclerView) view.findViewById(R.id.recyclerOverlayHeaderRecyclerView);
    recyclerView.setOnTouchListener(this);
    recyclerView.setOnSizeChangeListener(new OnSizeChangeListener() {
      @Override
      public void onSizeChange(int width, final int height, int oldWidth, final int oldHeight) {
        resolveRecyclerHeight(rootFrame.getLastHeight(), rootFrame.getLastHeight(), oldHeight, height);
      }
    });

    rootFrame = (SizeObservableFrameLayout) view.findViewById(R.id.frameOverlayHeaderRecyclerViewRoot);
    rootFrame.setOnTouchListener(this);
    rootFrame.setOnSizeChangeListener(new OnSizeChangeListener() {
      @Override
      public void onSizeChange(final int width, final int height, int oldWidth, final int oldHeight) {
        resolveRecyclerHeight(oldHeight, height, recyclerView.getCalculatedHeight(), recyclerView.getCalculatedHeight());
      }
    });
  }

  public void addBannerView(View view) {
    bannerFrame.addView(view);
  }

  public void setCallback(OverlayHeaderCallback callback) {
    this.callback = callback;
  }

  public void setListener(OnHeaderStateChangeListener listener) {
    this.listener = listener;
  }

  public boolean isHeaderShown() {
    return state == OverlayHeaderState.VISIBLE;
  }

  public RecyclerView getRecyclerView() {
    return recyclerView;
  }

  /**
   * Navigation click 시에만 호출되는 함수
   * Stable한 상태가 아닐 경우 (VISIBLE, HIDDEN 이외) 무시
   */
  public void toggleBanner() {
    switch (state) {
      case HIDDEN:
        state = OverlayHeaderState.VISIBLE;
        applyTranslationY(bannerHeight, bannerHeight, true);
        break;

      case VISIBLE:
        state = OverlayHeaderState.HIDDEN;
        applyTranslationY(bannerHeight, bannerHeight, true);
    }
  }

  /**
   * 헤더를 강제적으로 보여줘야 할 경우
   * Stable하게 Hidden일 경우에만 Visible로 state 변경, 이외에는 Translation만 맞춤
   *
   * @param animate 애니메이션 설정
   */
  public void showBanner(boolean animate) {
    if (state == OverlayHeaderState.HIDDEN) {
      state = OverlayHeaderState.VISIBLE;
    }
    setTranslateY(0, bannerHeight, animate);
  }

  /**
   * 헤더를 강제적으로 숨겨야 할 경우
   * Stable하게 Visible일 경우에만 Hidden으로 state 변경, 이외에는 Translation만 맞춤
   *
   * @param animate 애니메이션 설정
   */
  public void hideBanner(boolean animate) {
    if (state == OverlayHeaderState.VISIBLE) {
      state = OverlayHeaderState.HIDDEN;
    }
    setTranslateY(-bannerHeight, 0, animate);
  }

  /**
   * List 혹은 Root 사이즈가 변경 감지되었을 때 뷰 리사이징 등을 처리하는 함수
   *
   * @param oldFrameHeight 프레임의 이전 높이
   * @param newFrameHeight 프레임의 현재 높이
   * @param oldRecyclerViewHeight RecyclerView의 이전 높이
   * @param newRecyclerViewHeight RecyclerView의 현재 높이
   */
  private void resolveRecyclerHeight(int oldFrameHeight, int newFrameHeight, int oldRecyclerViewHeight, int newRecyclerViewHeight) {
    LayoutParams params = (LayoutParams) recyclerView.getLayoutParams();

    // 새로운 높이는 프레임과 RecyclerView 높이의 최소
    int newRecyclerHeight = Math.min(newFrameHeight, newRecyclerViewHeight);
    int oldRecyclerHeight = params.height;

    // RecyclerView가 프레임에 꽉 차지 않은 경우는 wrap content, 아니면 루트 크기에 맞춥니다.
    params.height = newRecyclerHeight < newFrameHeight ? ViewGroup.LayoutParams.WRAP_CONTENT : newFrameHeight;
    recyclerView.setLayoutParams(params);

    float recyclerViewTranslationY = recyclerView.getTranslationY();

    if (recyclerViewTranslationY > 0) {
      // recyclerView 자체에 translation 걸려있는 상태에서 사이즈가 커져 컨텐츠가 overflow 되었을 경우
      // overflow 된 크기만큼 translation을 올려줘야 함
      if (oldRecyclerViewHeight < newRecyclerHeight
          && oldFrameHeight == newFrameHeight
          && recyclerViewTranslationY + oldRecyclerViewHeight <= newFrameHeight
          && recyclerViewTranslationY + newRecyclerHeight > newFrameHeight) {

        float delta = recyclerViewTranslationY + newRecyclerHeight - newFrameHeight;
        isBannerAttachedToList = false;
        setTranslateY(bannerFrame.getTranslationY(), recyclerViewTranslationY - delta);
      }
    }
    // 키보드가 켜지는 등 화면 가용 크기가 줄어들 경우에 헤더를 숨김
    if (oldFrameHeight > newFrameHeight) {
      hideBanner(true);
    }
    // 화면이 줄어들 때 RecyclerView의 컨텐츠 사이즈가 프레임을 처음 벗어났을 때, 최하단으로 강제 스크롤
    if (oldFrameHeight > newFrameHeight && oldRecyclerHeight == ViewGroup.LayoutParams.WRAP_CONTENT && params.height > 0) {
      recyclerView.getLayoutManager().scrollToPosition(recyclerView.getAdapter().getItemCount() - 1);
    }
  }

  /**
   * Banner의 높이 변경이나, 스크롤 종료 시에 List와 Banner의 TranslationY을 정하는 함수
   *
   * @param newHeight 현재 Banner 높이를 가지는 값
   * @param oldHeight 현재 Banner의 높이가 변경되기 이전의 값, 만약 Banner 높이 변경 이외의 이유로 불렸을 경우에는 <code>newHeight</code> 와 동일함
   * @param animateOnHide Banner를 숨겨야 할 경우에 애니메이션을 줄 지 정해주는 플래그
   * @see OverlayHeaderRecyclerView#toggleBanner()
   */
  private void applyTranslationY(int newHeight, int oldHeight, boolean animateOnHide) {
    switch (state) {
      case HIDDEN:
        setTranslateY(-newHeight, 0, animateOnHide);
        break;

      case VISIBLE:
        applyBannerAttachmentToList(oldHeight);
        setTranslateY(0, isBannerAttachedToList ? newHeight : recyclerView.getTranslationY(), true);
        break;

      default:
        applyBannerAttachmentToList(oldHeight);
        float bannerTranslateY = Math.max(-newHeight, bannerFrame.getTranslationY());
        setTranslateY(
            bannerTranslateY,
            isBannerAttachedToList ? bannerTranslateY + newHeight : Math.min(recyclerView.getTranslationY(), newHeight)
        );
        break;
    }
  }

  /**
   * 상단으로 스크롤이 가능하고 Infinite scroll 의 경우에 상단 컨텐츠가 있는지 체크하는 함수
   *
   * @return List 의 contents 를 최상단까지 스크롤한 상태에서, List 를 상단으로 스크롤했을 때 더이상 불러올 contents 가 없을 경우 true
   */
  private boolean isScrollOnTop() {
    return !recyclerView.canScrollVertically(-1) && (callback == null || !callback.canScrollTop());
  }

  /**
   * @return List 의 contents 를 최하단까지 스크롤한 상태일 경우 true
   */
  private boolean isScrollOnBottom() { return !recyclerView.canScrollVertically(1); }

  /**
   * Banner가 List와 attach 되어있는지 체크하는 함수
   * (기존에 attach 되어있는지 || List에 Banner가 새로 달라붙었는지) && 스크롤이 최상단인 경우
   *
   * @param targetBannerHeight 계산 기준이 되는 banner height, 변경된 경우에 불리는 경우에는 이전 높이
   */
  private void applyBannerAttachmentToList(int targetBannerHeight) {
    boolean isListAttachToBanner = bannerFrame.getTranslationY() + targetBannerHeight == recyclerView.getTranslationY();
    isBannerAttachedToList = (isBannerAttachedToList || isListAttachToBanner) && isScrollOnTop();
  }

  @Override
  public boolean onInterceptTouchEvent(MotionEvent ev) {
    switch (ev.getActionMasked()) {
      case MotionEvent.ACTION_DOWN:
        state = OverlayHeaderState.TOUCH_DOWN;
        lastTouchY = ev.getRawY();
        break;

      case MotionEvent.ACTION_UP:
        if (handleTouchUp()) {
          return true;
        }
        break;
    }
    return super.onInterceptTouchEvent(ev);
  }

  @Override
  public boolean onTouch(View v, MotionEvent event) {
    switch (event.getActionMasked()) {
      case MotionEvent.ACTION_DOWN:
        return true;

      case MotionEvent.ACTION_MOVE:
        if (state == OverlayHeaderState.TOUCH_DOWN) {
          state = OverlayHeaderState.SCROLLING;
        }

        if (state == OverlayHeaderState.SCROLLING) {
          // 이전 터치 위치 기준으로 움직인 Y delta 구한 후에 터치 위치 갱신
          // delta > 0: 위로 스크롤
          // delta < 0: 아래로 스크롤
          float delta = event.getRawY() - lastTouchY;
          lastTouchY = event.getRawY();

          // 현재 헤더와 리스트 Translation
          float bannerTranslation = bannerFrame.getTranslationY();
          float listTranslation = recyclerView.getTranslationY();

          // Infinite scroll 포함 스크롤이 최상단인 경우
          if (isScrollOnTop()) {
            // 상단 스크롤
            if (delta >= 0) {
              applyBannerAttachmentToList(bannerHeight);
              setTranslateY(Math.max(bannerTranslation, listTranslation + delta - bannerHeight), listTranslation + delta);
              return true;
            }

            // 하단 스크롤 중에, 리스트에 Translation 걸려있을 경우
            if (delta < 0 && listTranslation > 0) {
              applyBannerAttachmentToList(bannerHeight);

              if (isBannerAttachedToList) {
                setTranslateY(listTranslation + delta - bannerHeight, listTranslation + delta);
              } else {
                setTranslateY(bannerTranslation, listTranslation + delta);
              }

              return true;
            }
          }
        }

        // 위 케이스에서 핸들링 되지 않았을 경우, Banner와 List는 attach 되지 않았다고 결정
        isBannerAttachedToList = false;
        break;

      case MotionEvent.ACTION_UP:
      case MotionEvent.ACTION_CANCEL:
        if (handleTouchUp()) {
          return true;
        }
        break;
    }
    return super.onTouchEvent(event);
  }

  /**
   * Touch up 이벤트가 발생했을 때 Banner 의 translationY 와 height / 2 를 기준으로 banner 의 위치를 재조정하는 함수
   *
   * @return 스크롤 핸들링이 되었으면 true
   */
  private boolean handleTouchUp() {
    if (state == OverlayHeaderState.SCROLLING || state == OverlayHeaderState.TOUCH_DOWN) {
      boolean handleScroll = state == OverlayHeaderState.SCROLLING;

      float bannerTranslationY = bannerFrame.getTranslationY();
      if (bannerTranslationY > -bannerHeight / 2f) {
        state = OverlayHeaderState.VISIBLE;
      } else {
        state = OverlayHeaderState.HIDDEN;
      }
      applyTranslationY(bannerHeight, bannerHeight, true);

      // 가속도 무시
      if (handleScroll && recyclerView.computeVerticalScrollOffset() == 0) {
        return true;
      }
    }
    return false;
  }

  private void setTranslateY(float bannerTranslateY, float listTranslationY) {
    setTranslateY(bannerTranslateY, listTranslationY, false);
  }

  private void setTranslateY(float bannerTranslateY, float listTranslationY, boolean animate) {
    bannerTranslateY = Math.min(0, bannerTranslateY);
    bannerTranslateY = Math.max(-bannerHeight, bannerTranslateY);

    listTranslationY = Math.max(0, listTranslationY);
    listTranslationY = Math.min(bannerHeight, listTranslationY);

    if (listener != null) {
      if (bannerTranslateY == 0) {
        listener.onBannerShow();
      } else {
        listener.onBannerHide();
      }
    }

    if (animate) {
      bannerFrame.animate().translationY(bannerTranslateY);
      recyclerView.animate().translationY(listTranslationY);
    } else {
      bannerFrame.animate().cancel();
      recyclerView.animate().cancel();
      bannerFrame.setTranslationY(bannerTranslateY);
      recyclerView.setTranslationY(listTranslationY);
    }
  }
}
